//                                               -*- C++ -*-
/**
 *  @file  CompositeRandomVector.hxx
 *  @brief An implementation class for composite random vectors
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-04-11 12:32:27 +0200 (Mon, 11 Apr 2011) $
 *  Id:      $Id: CompositeRandomVector.hxx 1866 2011-04-11 10:32:27Z schueller $
 */
#ifndef OPENTURNS_COMPOSITERANDOMVECTOR_HXX
#define OPENTURNS_COMPOSITERANDOMVECTOR_HXX

#include "RandomVectorImplementation.hxx"
#include "RandomVector.hxx"
#include "Pointer.hxx"
#include "NumericalMathFunction.hxx"
#include "CovarianceMatrix.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Model {



      /**
       * @class CompositeRandomVector
       *
       * An implementation class for composite random vectors
       */
      class CompositeRandomVector
        : public RandomVectorImplementation
      {
        CLASSNAME;

      public:

        /** Some typedefs to ease reading */
        typedef RandomVectorImplementation::Antecedent               Antecedent;
        typedef RandomVectorImplementation::NumericalPoint           NumericalPoint;
        typedef RandomVectorImplementation::NumericalSample          NumericalSample;
        typedef RandomVectorImplementation::CovarianceMatrix         CovarianceMatrix;
        typedef RandomVectorImplementation::NumericalMathFunction    NumericalMathFunction;
        typedef RandomVectorImplementation::StorageManager           StorageManager;
        typedef RandomVectorImplementation::InvalidArgumentException InvalidArgumentException;

        /** Default constructor */
        CompositeRandomVector(const String & name = OT::DefaultName);

        /** Standard constructor */
        CompositeRandomVector(const NumericalMathFunction & function,
                              const Antecedent & p_antecedent,
                              const String & name = OT::DefaultName)
          /* throw (InvalidArgumentException) */;

        /** Standard constructor */
        CompositeRandomVector(const NumericalMathFunction & function,
                              const RandomVector & antecedent,
                              const String & name = OT::DefaultName)
          /* throw (InvalidArgumentException) */;


        /** Virtual constructor */
        CompositeRandomVector * clone() const;

        /** String converter */
        String __repr__() const;



        /** Is the underlying random vector composite ? */
        Bool isComposite() const;


        /* Here is the interface that all derived class must implement */

        /** Dimension accessor */
        UnsignedLong getDimension() const;

        /** Realization accessor */
        NumericalPoint getRealization() const;

        /** Numerical sample accessor */
        NumericalSample getNumericalSample(const UnsignedLong size) const;

        /** Mean accessor */
        NumericalPoint getMean() const;

        /** Covariance accessor */
        CovarianceMatrix getCovariance() const;

        /** Get the random vector corresponding to the i-th marginal component */
        CompositeRandomVector * getMarginal(const UnsignedLong i) const /* throw(InvalidArgumentException) */;

        /** Get the marginal random vector corresponding to indices components */
        CompositeRandomVector * getMarginal(const Indices & indices) const /* throw(InvalidArgumentException) */;

        /** Antecedent accessor */
        Antecedent getAntecedent() const;

        /** Function accessor */
        NumericalMathFunction getFunction() const;

        /** Distribution accessor */
        Distribution getDistribution() const;

        /** Method save() stores the object through the StorageManager */
        void save(StorageManager::Advocate & adv) const;

        /** Method load() reloads the object from the StorageManager */
        void load(StorageManager::Advocate & adv);

      protected:
        /** The function the vector is the output */
        NumericalMathFunction function_;

        /** The antecedent of the vector through the numerical math function */
        Antecedent p_antecedent_;

      private:

      }; /* class CompositeRandomVector */


    } /* namespace Model */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_COMPOSITERANDOMVECTOR_HXX */
