//                                               -*- C++ -*-
/**
 *  @file  Multinomial.cxx
 *  @brief The Multinomial distribution
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-07-28 18:31:43 +0200 (Thu, 28 Jul 2011) $
 *  Id:      $Id: Multinomial.cxx 2046 2011-07-28 16:31:43Z schueller $
 */
#include <cmath>
#include "Collection.hxx"
#include "Multinomial.hxx"
#include "Binomial.hxx"
#include "Poisson.hxx"
#include "TruncatedDistribution.hxx"
#include "RandomGenerator.hxx"
#include "SpecFunc.hxx"
#include "DistFunc.hxx"
#include "Exception.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Base::Common::NotYetImplementedException NotYetImplementedException;
      typedef Base::Func::SpecFunc                     SpecFunc;
      typedef Base::Stat::RandomGenerator              RandomGenerator;
      typedef Base::Type::Collection<UnsignedLong>     UnsignedLongCollection;

      CLASSNAMEINIT(Multinomial);

      static Base::Common::Factory<Multinomial> RegisteredFactory("Multinomial");

      /* Default constructor */
      Multinomial::Multinomial()
        /* throw (InvalidArgumentException) */
        : DiscreteDistribution("Multinomial"),
          n_(1),
          p_(0),
          sumP_(0.0)
      {
        setP( NumericalPoint(1, 0.5) );
      }

      /* Parameters constructor */
      Multinomial::Multinomial(const UnsignedLong n,
                               const NumericalPoint & p)
        /* throw (InvalidArgumentException) */
        : DiscreteDistribution("Multinomial"),
          n_(n),
          p_(0),
          sumP_(0.0)
      {
        setP( p );
      }

      /* Comparison operator */
      Bool Multinomial::operator ==(const Multinomial & other) const
      {
        if (this == &other) return true;
        return (n_ == other.n_) && (p_ == other.p_);
      }

      /* String converter */
      String Multinomial::__repr__() const
      {
        OSS oss;
        oss << "class=" << Multinomial::GetClassName()
            << " name=" << getName()
            << " dimension=" << getDimension()
            << " p=" << p_
            << " n=" << n_;
        return oss;
      }

      String Multinomial::__str__(const String & offset) const
      {
        OSS oss;
        oss << offset << getClassName() << "(n = " << n_ << ", p = " << p_.__str__() << ")";
        return oss;
      }

      /* Virtual constructor */
      Multinomial * Multinomial::clone() const
      {
        return new Multinomial(*this);
      }

      /* Get one realization of the distribution */
      Multinomial::NumericalPoint Multinomial::getRealization() const
      {
        const UnsignedLong dimension(getDimension());
        NumericalPoint realization(dimension);
        /* We use an elementary algorithm based on the definition of the Multinomial distribution:
         * the i-th component is generated using a Binomial distribution */
        UnsignedLong n(n_);
        NumericalScalar sum(1.0);
        for (UnsignedLong i = 0; i < dimension; ++i)
          {
            /* The current component follow a binomial distribution with parameters p_[i] and N */
            const UnsignedLong xI(DistFunc::rBinomial(n, p_[i] / sum));
            realization[i] = xI;
            n -= xI;
            sum -= p_[i];
          }
        return realization;
      }


      /* Get the PDF of the distribution */
      NumericalScalar Multinomial::computePDF(const NumericalPoint & point) const
      {
        const UnsignedLong dimension(getDimension());
        if (point.getDimension() != dimension) throw InvalidArgumentException(HERE) << "Error: the given point has a dimension not compatible with the distribution dimension";
        // First, check the validity of the input
        NumericalScalar sumX(0.0);
        for (UnsignedLong i = 0; i < dimension; ++i)
          {
            const NumericalScalar k(point[i]);
            // Early exit if the given point is not in the support of the distribution
            if ((fabs(k - round(k)) > DiscreteDistribution::SupportEpsilon) || (k < -DiscreteDistribution::SupportEpsilon) || (k > n_ + DiscreteDistribution::SupportEpsilon)) return 0.0;
            sumX += k;
          }
        if (sumX > n_ + DiscreteDistribution::SupportEpsilon) return 0.0;
        NumericalScalar pdf(SpecFunc::LnGamma(n_ + 1.0) - SpecFunc::LnGamma(n_ - sumX + 1.0) + (n_ - sumX) * log(1.0 - sumP_));
        for (UnsignedLong i = 0; i < dimension; ++i)
          {
            const NumericalScalar k(point[i]);
            pdf += k * log(p_[i]) - SpecFunc::LnGamma(k + 1.0);
          }
        return exp(pdf);
      }


      /* Get the CDF of the distribution
         Implementation using the representation given in Th. 1 of:
         Bruce Levin, "A representation for multinomial cumulative distribution functions",
         The annals of Statistics, 1981, Vol. 9, No. 5, 1123-1126
         with adaptations to our definition of the multinomial distribution.
         The convolution involved in the computation of the PDF of a sum of independent
         truncated Poisson distributions is done using Poisson's formula, see:
         J. Abate, W. Whitt, "The Fourier-series method for inverting transforms of probability distributions",
         Queuing systems, feb. 1991. */
      NumericalScalar Multinomial::computeCDF(const NumericalPoint & point,
                                              const Bool tail) const
      {
        const UnsignedLong dimension(getDimension());
        if (point.getDimension() != dimension) throw InvalidArgumentException(HERE) << "Error: the given point has a dimension not compatible with the distribution dimension";
        // Early exit for 1D case
        if (dimension == 1) return Binomial(n_, p_[0]).computeCDF(point, tail);
        // First, check the bording cases
        Indices indices(0);
        Bool allZero(true);
        // Trivial cases
        for (UnsignedLong i = 0; i < dimension; ++i)
          {
            // If the given point does not cover any point of the support, return 0.0
            if (point[i] < -DiscreteDistribution::SupportEpsilon) return 0.0;
            if (point[i] < n_ - DiscreteDistribution::SupportEpsilon) indices.add(i);
            allZero = allZero && (fabs(point[i]) < DiscreteDistribution::SupportEpsilon);
          }
        // If we are at the origin, CDF=PDF(0,...,0)
        if (allZero) return pow(1.0 - sumP_, n_);
        // If the point covers the whole support of the distribution, return 1.0
        const UnsignedLong size(indices.getSize());
        if (size == 0) return 1.0;
        // Now, reduce the dimension to the active indices
        NumericalScalar sum(0.0);
        NumericalScalar logCDF(0.0);
        Base::Type::Collection< TruncatedDistribution > atoms(0);
        for (UnsignedLong i = 0; i < size; ++i)
          {
            const NumericalScalar pI(p_[indices[i]]);
            const NumericalScalar nPI(n_ * pI);
            const NumericalScalar xI(point[indices[i]]);
            atoms.add(TruncatedDistribution(Poisson(nPI), xI, TruncatedDistribution::UPPER));
            logCDF += log(Poisson(nPI).computeCDF(xI));
            sum += pI;
          }
        const Poisson compoundDistribution(n_ * (1.0 - sum));
        // Convolution of truncated Poisson
        UnsignedLong m(2 * n_);
        const NumericalScalar r(pow(2.0, -106.0 / (3. * m)));
        logCDF += SpecFunc::LogGamma(n_ + 1) - n_ * log(r * n_) + n_ - log(m);
        NumericalScalar factor(0.0);
        // Evaluation of P(W=n) using Poisson's formula
        // First term of Poisson's formula
        {
          // Loop to compute the scaled log generating function
          NumericalScalar term(compoundDistribution.computeGeneratingFunction(r, true).real());
          for (UnsignedLong i = 0; i < size; ++i) term += atoms[i].computeGeneratingFunction(r, true);
          factor += exp(term);
        }
        // Interior terms of Poisson's formula
        int sign(-1);
        for (UnsignedLong k = 1; k < n_; ++k)
          {
            const NumericalComplex argument(r * exp(NumericalComplex(0.0, (M_PI * k) / n_)));
            // Loop to compute the scaled log generating function
            NumericalComplex term(compoundDistribution.computeGeneratingFunction(argument, true));
            for (UnsignedLong i = 0; i < size; ++i) term += atoms[i].computeGeneratingFunction(argument, true);
            factor += sign * 2.0 * exp(term).real();
            sign = -sign;
          } // Poisson's formula
        // Last term of Poisson's formula
        {
          // Loop to compute the scaled log generating function
          NumericalScalar term(compoundDistribution.computeGeneratingFunction(-r, true).real());
          for (UnsignedLong i = 0; i < size; ++i) term += atoms[i].computeGeneratingFunction(-r, true);
          factor += sign * exp(term);
        }
        // Poisson's formula based algorithms have an absolute precision of order the arithmetic precision at the power 2/3
        if (factor < pow(DefaultQuantileEpsilon, 2.0 / 3.0)) return getMarginal(indices)->computeCDFReferenceAlgorithm(point, tail);
        logCDF += log(factor);
        // Due to round-off errors, the computed CDF can be slightly over 1.
        return std::min(1.0, exp(logCDF));
      }

      /* Get the CDF of the distribution */
      NumericalScalar Multinomial::computeCDFReferenceAlgorithm(const NumericalPoint & point,
                                                                const Bool tail) const
      {
        const UnsignedLong dimension(getDimension());
        if (point.getDimension() != dimension) throw InvalidArgumentException(HERE) << "Error: the given point has a dimension not compatible with the distribution dimension";
        // Early exit for 1D case
        if (dimension == 1) return Binomial(n_, p_[0]).computeCDF(point, tail);
        // First, check the bording cases
        Bool outside(true);
        for (UnsignedLong i = 0; i < dimension; ++i)
          {
            // If the given point does not cover any point of the support, return 0.0
            if (point[i] < -DiscreteDistribution::SupportEpsilon) return 0.0;
            // If the current component is also over the bound
            outside &= (point[i] >= n_ - DiscreteDistribution::SupportEpsilon);
          }
        // If the point covers the whole support of the distribution, return 1.0
        if (outside) return 1.0;
        // Straightforward implementation: sum the PDF over all the integral points in [[0, point[0]]] x ... x [[0,point[dimension-1]]]
        NumericalScalar cdf(0.0);
        UnsignedLongCollection integerPoint(dimension);
        for (UnsignedLong i = 0; i < dimension; ++i) integerPoint[i] = UnsignedLong(floor(point[i] + DiscreteDistribution::SupportEpsilon));
        /* Size of the sample to be generated: levels[0] * ... * levels[dimension-1] */
        UnsignedLong size(integerPoint[0] + 1);
        for (UnsignedLong i = 1; i < dimension; ++i) size *= (integerPoint[i] + 1);
        /* Indices would have stored the indices of the nested loops if we were able to code "dimension" nested loops dynamically */
        UnsignedLongCollection indices(dimension, 0);
        for (UnsignedLong flatIndex = 0; flatIndex < size; ++flatIndex)
          {
            NumericalPoint currentPoint(dimension);
            for (UnsignedLong i = 0; i < dimension; ++i) currentPoint[i] = indices[i];
            cdf += computePDF(currentPoint);
            /* Update the indices */
            ++indices[0];
            /* Propagate the remainders */
            for (UnsignedLong i = 0; i < dimension - 1; ++i) indices[i + 1] += (indices[i] == (integerPoint[i] + 1));
            /* Correction of the indices. The last index cannot overflow. */
            for (UnsignedLong i = 0; i < dimension - 1; ++i) indices[i] = indices[i] % (integerPoint[i] + 1);
          }
        return cdf;
      }

      /* Compute the scalar quantile of the 1D multinomial distribution */
      NumericalScalar Multinomial::computeScalarQuantile(const NumericalScalar prob,
                                                         const Bool tail,
                                                         const NumericalScalar precision) const
      {
        return Binomial(n_, p_[0]).computeQuantile(prob, tail)[0];
      } // computeScalarQuantile

      /* Compute the PDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1)
         For Multinomial distribution, the conditional distribution is Binomial
      */
      NumericalScalar Multinomial::computeConditionalPDF(const NumericalScalar x,
                                                         const NumericalPoint & y) const
      {
        const UnsignedLong conditioningDimension(y.getDimension());
        if (conditioningDimension >= getDimension()) throw InvalidArgumentException(HERE) << "Error: cannot compute a conditional PDF with a conditioning point of dimension greater or equal to the distribution dimension.";
        // Special case for no conditioning or independent copula
        if (conditioningDimension == 0) return Binomial(n_, p_[0]).computePDF(x);
        // General case
        // Check that y is a valid conditioning vector
        UnsignedLongCollection intY(conditioningDimension);
        UnsignedLong sumY(0);
        NumericalScalar sumP(0.0);
        for (UnsignedLong i = 0; i < conditioningDimension; ++i)
          {
            const NumericalScalar yI(y[i]);
            const UnsignedLong intYI(static_cast<UnsignedLong>(round(yI)));
            if (fabs(yI - intYI) > DiscreteDistribution::SupportEpsilon) throw InvalidArgumentException(HERE) << "Error: the conditioning vector has non-integer values";
            sumY += yI;
            intY[i] = intYI;
            sumP += p_[i];
          }
        if (sumY > n_) throw InvalidArgumentException(HERE) << "Error: the conditioning vector has a sum of components greater than the allowed range.";
        return Binomial(n_ - sumY, p_[conditioningDimension] / sumP).computePDF(x);
      }

      /* Compute the CDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1) */
      NumericalScalar Multinomial::computeConditionalCDF(const NumericalScalar x,
                                                         const NumericalPoint & y) const
      {
        const UnsignedLong conditioningDimension(y.getDimension());
        if (conditioningDimension >= getDimension()) throw InvalidArgumentException(HERE) << "Error: cannot compute a conditional CDF with a conditioning point of dimension greater or equal to the distribution dimension.";
        // Special case for no conditioning or independent copula
        if (conditioningDimension == 0) return Binomial(n_, p_[0]).computeCDF(x);
        // General case
        // Check that y is a valid conditioning vector
        UnsignedLongCollection intY(conditioningDimension);
        UnsignedLong sumY(0);
        NumericalScalar sumP(0.0);
        for (UnsignedLong i = 0; i < conditioningDimension; ++i)
          {
            const NumericalScalar yI(y[i]);
            const UnsignedLong intYI(static_cast<UnsignedLong>(round(yI)));
            if (fabs(yI - intYI) > DiscreteDistribution::SupportEpsilon) throw InvalidArgumentException(HERE) << "Error: the conditioning vector has non-integer values";
            sumY += yI;
            intY[i] = intYI;
            sumP += p_[i];
          }
        if (sumY > n_) throw InvalidArgumentException(HERE) << "Error: the conditioning vector has a sum of components greater than the allowed range.";
        return Binomial(n_ - sumY, p_[conditioningDimension] / sumP).computeCDF(x);
      }

      /* Compute the quantile of Xi | X1, ..., Xi-1, i.e. x such that CDF(x|y) = q with x = Xi, y = (X1,...,Xi-1) */
      NumericalScalar Multinomial::computeConditionalQuantile(const NumericalScalar q,
                                                              const NumericalPoint & y) const
      {
        const UnsignedLong conditioningDimension(y.getDimension());
        if (conditioningDimension >= getDimension()) throw InvalidArgumentException(HERE) << "Error: cannot compute a conditional quantile with a conditioning point of dimension greater or equal to the distribution dimension.";
        if ((q < 0.0) || (q > 1.0)) throw InvalidArgumentException(HERE) << "Error: cannot compute a conditional quantile for a probability level outside of [0, 1]";
        // Special case when no contitioning or independent copula
        if (conditioningDimension == 0) return Binomial(n_, p_[0]).computeQuantile(q)[0];
        // General case
        // Check that y is a valid conditioning vector
        UnsignedLongCollection intY(conditioningDimension);
        UnsignedLong sumY(0);
        NumericalScalar sumP(0.0);
        for (UnsignedLong i = 0; i < conditioningDimension; ++i)
          {
            const NumericalScalar yI(y[i]);
            const UnsignedLong intYI(static_cast<UnsignedLong>(round(yI)));
            if (fabs(yI - intYI) > DiscreteDistribution::SupportEpsilon) throw InvalidArgumentException(HERE) << "Error: the conditioning vector has non-integer values";
            sumY += yI;
            intY[i] = intYI;
            sumP += p_[i];
          }
        if (sumY > n_) throw InvalidArgumentException(HERE) << "Error: the conditioning vector has a sum of components greater than the allowed range.";
        return Binomial(n_ - sumY, p_[conditioningDimension] / sumP).computeQuantile(q)[0];
      }

      /* Get the i-th marginal distribution */
      Multinomial * Multinomial::getMarginal(const UnsignedLong i) const /* throw(InvalidArgumentException) */
      {
        const UnsignedLong dimension(getDimension());
        if (i >= dimension) throw InvalidArgumentException(HERE) << "The index of a marginal distribution must be in the range [0, dim-1]";
        // Special case for dimension 1
        if (dimension == 1) return clone();
        // General case
        Multinomial result(n_, NumericalPoint(1, p_[i]));
        result.setName("Marginal of " + getName());
        result.setWeight(getWeight());
        return result.clone();
      }

      /* Get the distribution of the marginal distribution corresponding to indices dimensions */
      Multinomial * Multinomial::getMarginal(const Indices & indices) const /* throw(InvalidArgumentException) */
      {
        const UnsignedLong dimension(getDimension());
        if (!indices.check(dimension - 1)) throw InvalidArgumentException(HERE) << "The indices of a marginal distribution must be in the range [0, dim-1] and  must be different";
        // Special case for dimension 1
        if (dimension == 1) return clone();
        // General case
        const UnsignedLong outputDimension(indices.getSize());
        NumericalPoint marginalP(outputDimension);
        // Extract the correlation matrix, the marginal standard deviations and means
        for (UnsignedLong i = 0; i < outputDimension; ++i)
          {
            const UnsignedLong index_i(indices[i]);
            marginalP[i] = p_[index_i];
          }
        Multinomial result(n_, marginalP);
        result.setName("Marginal of " + getName());
        result.setWeight(getWeight());
        return result.clone();
      } // getMarginal(Indices)

      /* Get the support of a discrete distribution that intersect a given interval */
      Multinomial::NumericalSample Multinomial::getSupport(const Interval & interval) const
      {
        if (interval.getDimension() != getDimension()) throw InvalidArgumentException(HERE) << "Error: the given interval has a dimension that does not match the distribution dimension.";
        throw NotYetImplementedException(HERE);
      }

      /* Compute the mean of the distribution */
      void Multinomial::computeMean() const
      {
        mean_ = n_ * p_;
        isAlreadyComputedMean_ = true;
      }

      /* Compute the covariance of the distribution */
      void Multinomial::computeCovariance() const
      {
        const UnsignedLong dimension(getDimension());
        // To insure a zero initialization
        covariance_ = CovarianceMatrix(dimension);
        for (UnsignedLong i = 0; i < dimension; ++i)
          {
            const NumericalScalar pI(p_[i]);
            covariance_(i, i) = pI * (1.0 - pI) * n_;
            // Be careful! in these computations, n_ cannot be at the begining of the formula else -n_ will underflow the UnsignedLong range!
            for (UnsignedLong j = 0; j < i; ++j) covariance_(i, j) = -pI * p_[j] * n_;
          }
        isAlreadyComputedCovariance_ = true;
      }

      /* Parameters value and description accessor */
      Multinomial::NumericalPointWithDescriptionCollection Multinomial::getParametersCollection() const
      {
        const UnsignedLong dimension(getDimension());
        NumericalPointWithDescriptionCollection parameters((dimension == 1 ? 1 : dimension + 1));
        for (UnsignedLong i = 0; i < dimension; ++i)
          {
            NumericalPointWithDescription point(2);
            point[0] = n_;
            point[1] = p_[i];
            Description description(2);
            description[0] = "n";
            OSS oss;
            oss << "p_" << i;
            description[1] = oss;
            point.setDescription(description);
            point.setName(getDescription()[i]);
            parameters[i] = point;
          }
        if (dimension > 1)
          {
            NumericalPointWithDescription point(dimension + 1);
            Description description(dimension + 1);
            point[0] = n_;
            description[0] = "n";
            for (UnsignedLong i = 0; i < dimension; ++i)
              {
                point[i + 1] = p_[i];
                OSS oss;
                oss << "p_" << i;
                description[i + 1] = oss;
              }
            point.setDescription(description);
            point.setName("dependence");
            parameters[dimension] = point;
          }
        return parameters;
      }

      /* P accessor */
      void Multinomial::setP(const NumericalPoint & p)
      /* throw (InvalidArgumentException) */
      {
        // We check that the elements are all positive
	const UnsignedLong dimension(p.getDimension());
        if (dimension == 0) throw InvalidArgumentException(HERE) << "P must have a positive dimension.";
        NumericalScalar sum(0.0);
        for(UnsignedLong i = 0; i < dimension; ++i)
          {
            NumericalScalar pI(p[i]);
            if (pI <= 0.0) throw InvalidArgumentException(HERE) << "P elements MUST be positive";
            sum += pI;
            if (sum > 1.0) throw InvalidArgumentException(HERE) << "The sum of the probabilities MUST be <= 1.0";
          }
        p_ = p;
        sumP_ = sum;
	setDimension(dimension);
        isAlreadyComputedMean_ = false;
        isAlreadyComputedCovariance_ = false;
        isAlreadyCreatedGeneratingFunction_ = false;
      }

      /* P accessor */
      Multinomial::NumericalPoint Multinomial::getP() const
      {
        return p_;
      }

      /* N accessor */
      void Multinomial::setN(const UnsignedLong n)
      {
        if (n != n_)
          {
            n_ = n;
            isAlreadyComputedMean_ = false;
            isAlreadyComputedCovariance_ = false;
	    isAlreadyCreatedGeneratingFunction_ = false;
          }
      }

      UnsignedLong Multinomial::getN() const
      {
        return n_;
      }

      /* Method save() stores the object through the StorageManager */
      void Multinomial::save(StorageManager::Advocate & adv) const
      {
        DiscreteDistribution::save(adv);
        adv.saveAttribute( "n_", n_ );
        adv.saveAttribute( "p_", p_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void Multinomial::load(StorageManager::Advocate & adv)
      {
        DiscreteDistribution::load(adv);
        adv.loadAttribute( "n_", n_ );
        adv.loadAttribute( "p_", p_ );
        computeRange();
      }

    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
