//                                               -*- C++ -*-
/**
 *  @file  KernelMixture.hxx
 *  @brief The class that implements kernelMixtures
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: souchaud $
 *  @date:   $LastChangedDate: 2011-07-01 10:34:36 +0200 (Fri, 01 Jul 2011) $
 *  Id:      $Id: KernelMixture.hxx 1981 2011-07-01 08:34:36Z souchaud $
 */
#ifndef OPENTURNS_KERNELMIXTURE_HXX
#define OPENTURNS_KERNELMIXTURE_HXX

#include "Distribution.hxx"
#include "Exception.hxx"
#include "NonEllipticalDistribution.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {


      /**
       * @class KernelMixture
       *
       * The class describes the probabilistic concept of KernelMixture.
       */
      class KernelMixture
        : public Model::NonEllipticalDistribution
      {
        CLASSNAME;
      public:

        typedef Model::NonEllipticalDistribution                    NonEllipticalDistribution;  // required by SWIG
        typedef Model::Distribution                                 Distribution;
        typedef NonEllipticalDistribution::InvalidArgumentException InvalidArgumentException;
        typedef NonEllipticalDistribution::NumericalPoint           NumericalPoint;
        typedef NonEllipticalDistribution::NumericalSample          NumericalSample;
        typedef NonEllipticalDistribution::Implementation           Implementation;
        typedef NonEllipticalDistribution::Indices                  Indices;
        typedef NonEllipticalDistribution::CovarianceMatrix         CovarianceMatrix;
        typedef NonEllipticalDistribution::NotDefinedException      NotDefinedException;
        typedef NonEllipticalDistribution::NumericalPointWithDescriptionCollection NumericalPointWithDescriptionCollection;
        typedef NonEllipticalDistribution::StorageManager            StorageManager;

        /** Default constructor */
        explicit KernelMixture(const Distribution & kernel,
                               const NumericalPoint & bandwidth,
                               const NumericalSample & sample)
          /* throw (InvalidArgumentException) */;


        /** Comparison operator */
        Bool operator ==(const KernelMixture & other) const;

        /** String converter */
        String __repr__() const;
        String __str__(const String & offset = "") const;


        /** Kernel accessor */
        void setKernel(const Distribution & kernel)
          /* throw (InvalidArgumentException) */;
        Distribution getKernel() const;

        /** Bandwidth accessor */
        void setBandwidth(const NumericalPoint & bandwidth)
          /* throw (InvalidArgumentException) */;
        NumericalPoint getBandwidth() const;

        /** Sample accessor */
        void setSample(const NumericalSample & sample)
          /* throw (InvalidArgumentException) */;
        NumericalSample getSample() const;



        /* Here is the interface that all derived class must implement */

        /** Virtual constructor */
        virtual KernelMixture * clone() const;

        /** Get one realization of the KernelMixture */
        virtual NumericalPoint getRealization() const;

        /** Get the DDF of the KernelMixture */
        using NonEllipticalDistribution::computeDDF;
        virtual NumericalPoint computeDDF(const NumericalPoint & point) const;

        /** Get the PDF of the KernelMixture */
        using NonEllipticalDistribution::computePDF;
        virtual NumericalScalar computePDF(const NumericalPoint & point) const;

        /** Get the CDF of the KernelMixture */
        using NonEllipticalDistribution::computeCDF;
        virtual NumericalScalar computeCDF(const NumericalPoint & point, const Bool tail = false) const;

        /** Get the characteristic function of the distribution, i.e. phi(u) = E(exp(I*u*X)) */
        NumericalComplex computeCharacteristicFunction(const NumericalScalar x,
                                                       const Bool logScale = false) const;

        /** Get the PDF gradient of the distribution */
        virtual NumericalPoint computePDFGradient(const NumericalPoint & point) const;

        /** Get the CDF gradient of the distribution */
        virtual NumericalPoint computeCDFGradient(const NumericalPoint & point) const;

        /** Get the i-th marginal distribution */
        KernelMixture * getMarginal(const UnsignedLong i) const /* throw(InvalidArgumentException) */;

        /** Get the distribution of the marginal distribution corresponding to indices dimensions */
        KernelMixture * getMarginal(const Indices & indices) const /* throw(InvalidArgumentException) */;

        /** Get the standard deviation of the distribution */
        NumericalPoint getStandardDeviation() const /* throw(NotDefinedException) */;

        /** Get the skewness of the distribution */
        NumericalPoint getSkewness() const /* throw(NotDefinedException) */;

        /** Get the kurtosis of the distribution */
        NumericalPoint getKurtosis() const /* throw(NotDefinedException) */;

        /** Parameters value and description accessor */
        virtual NumericalPointWithDescriptionCollection getParametersCollection() const;

        /** Check if the distribution is elliptical */
        Bool isElliptical() const;

        /** Check if the distribution is constinuous */
        Bool isContinuous() const;

        /** Tell if the distribution has elliptical copula */
        Bool hasEllipticalCopula() const;

        /** Tell if the distribution has independent copula */
        Bool hasIndependentCopula() const;

        /** Method save() stores the object through the StorageManager */
        virtual void save(StorageManager::Advocate & adv) const;

        /** Method load() reloads the object from the StorageManager */
        virtual void load(StorageManager::Advocate & adv);

      protected:


      private:

        /** Compute the numerical range of the distribution given the parameters values */
        void computeRange();

        /** For save/load mechanism */
        KernelMixture() {};
        friend class Base::Common::Factory<KernelMixture>;

        /** Get the mean of a kernelMixture */
        void computeMean() const /* throw(NotDefinedException) */;

        /** Get the covariance of a kernelMixture */
        void computeCovariance() const /* throw(NotDefinedException) */;

        /** Compute the normalization factor */
        void computeNormalizationFactor();

        /** The kernel of the kernelMixture */
        Distribution kernel_;

        /** The bandwidth of the KernelMixture */
        NumericalPoint bandwidth_;

        /** The componentwise bandwidth inverse of the KernelMixture */
        NumericalPoint bandwidthInverse_;

        /** The (n\product_{k=1}^{dim}h_k)^{-1} normalization factor */
        NumericalScalar normalizationFactor_;

        /** The sample of the kernelMixture */
        NumericalSample sample_;

      }; /* class KernelMixture */


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_KERNELMIXTURE_HXX */
