//                                               -*- C++ -*-
/**
 *  @file  Dirichlet.hxx
 *  @brief The Dirichlet distribution
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-07-12 15:45:44 +0200 (lun. 12 juil. 2010) $
 *  Id:      $Id: Dirichlet.hxx 1581 2010-07-12 13:45:44Z dutka $
 */
#ifndef OPENTURNS_DIRICHLET_HXX
#define OPENTURNS_DIRICHLET_HXX

#include "NonEllipticalDistribution.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {


      /**
       * @class Dirichlet
       *
       * The Dirichlet distribution.
       */
      class Dirichlet
        : public Model::NonEllipticalDistribution
      {
        // Maximum number of samples in the evaluation of the CDF
        static const UnsignedLong DefaultSamplingSize; // 400000
        // Number of integration nodes per dimension
        static const UnsignedLong DefaultIntegrationSize; // 50
        CLASSNAME;
      public:

        typedef Base::Common::InvalidArgumentException               InvalidArgumentException;
        typedef Model::NonEllipticalDistribution                     NonEllipticalDistribution;  // required by SWIG
        typedef NonEllipticalDistribution::Implementation            Implementation;
        typedef NonEllipticalDistribution::NumericalPoint            NumericalPoint;
        typedef NonEllipticalDistribution::NumericalSample           NumericalSample;
        typedef NonEllipticalDistribution::CovarianceMatrix          CovarianceMatrix;
        typedef NonEllipticalDistribution::NumericalPointWithDescriptionCollection  NumericalPointWithDescriptionCollection;
        typedef NonEllipticalDistribution::NotDefinedException       NotDefinedException;
        typedef NonEllipticalDistribution::StorageManager            StorageManager;

        /** Default constructor */
        Dirichlet();

        /** Parameters constructor */
        Dirichlet(const NumericalPoint & theta);

        /** Comparison operator */
        Bool operator ==(const Dirichlet & other) const;

        /** String converter */
        String __repr__() const;
        String __str__(const String & offset = "") const;



        /* Interface inherited from Distribution */

        /** Virtual constructor */
        Dirichlet * clone() const;

        /** Get one realization of the distribution */
        NumericalPoint getRealization() const;

        /** Get the PDF of the distribution */
        using NonEllipticalDistribution::computePDF;
        NumericalScalar computePDF(const NumericalPoint & point) const;

        /** Get the CDF of the distribution */
        using NonEllipticalDistribution::computeCDF;
        NumericalScalar computeCDF(const NumericalPoint & point, const Bool tail = false) const;

        /** Get the standard deviation of the distribution */
        NumericalPoint getStandardDeviation() const /* throw(NotDefinedException) */;

        /** Get the skewness of the distribution */
        NumericalPoint getSkewness() const /* throw(NotDefinedException) */;

        /** Get the kurtosis of the distribution */
        NumericalPoint getKurtosis() const /* throw(NotDefinedException) */;

        /* Interface specific to Dirichlet */

        /** Theta accessor */
        void setTheta(const NumericalPoint & theta);
        NumericalPoint getTheta() const;

        /** Compute the PDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1) */
        NumericalScalar computeConditionalPDF(const NumericalScalar x, const NumericalPoint & y) const;

        /** Compute the CDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1) */
        NumericalScalar computeConditionalCDF(const NumericalScalar x, const NumericalPoint & y) const;

        /** Compute the quantile of Xi | X1, ..., Xi-1, i.e. x such that CDF(x|y) = q with x = Xi, y = (X1,...,Xi-1) */
        NumericalScalar computeConditionalQuantile(const NumericalScalar q, const NumericalPoint & y) const;

        /** Get the i-th marginal distribution */
        Dirichlet * getMarginal(const UnsignedLong i) const /* throw(InvalidArgumentException) */;

        /** Get the distribution of the marginal distribution corresponding to indices dimensions */
        Dirichlet * getMarginal(const Indices & indices) const /* throw(InvalidArgumentException) */;

        /** Method save() stores the object through the StorageManager */
        void save(StorageManager::Advocate & adv) const;

        /** Method load() reloads the object from the StorageManager */
        void load(StorageManager::Advocate & adv);


      protected:


      private:

        /** Compute the mean of the distribution */
        void computeMean() const;

        /** Compute the covariance of the distribution */
        void computeCovariance() const;

        /** Get the quantile of the distribution */
        NumericalScalar computeScalarQuantile(const NumericalScalar prob,
                                              const Bool tail = false,
                                              const NumericalScalar precision = DefaultQuantileEpsilon) const;

        /** Compute the numerical range of the distribution given the parameters values */
        void computeRange();

        /** Update the derivative attributes */
        void update();

        /** The main parameter set of the distribution */
        NumericalPoint theta_;
        NumericalScalar sumTheta_;
        NumericalScalar normalizationFactor_;
        mutable Bool isInitializedCDF_;
        mutable NumericalPointCollection integrationNodes_;
        mutable NumericalPointCollection integrationWeights_;
      }; /* class Dirichlet */


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_DIRICHLET_HXX */
