//                                               -*- C++ -*-
/**
 *  @file  ChiFactory.cxx
 *  @brief Factory for Chi distribution
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2007-05-10 16:43:31 +0200 (jeu. 10 mai 2007) $
 *  Id:      $Id: ChiFactory.cxx 434 2007-05-10 14:43:31Z dutka $
 */
#include "ChiFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      /* Default constructor */
      ChiFactory::ChiFactory():
        DistributionImplementationFactory()
      {
        // Nothing to do
      }

      /* Virtual constructor */
      ChiFactory * ChiFactory::clone() const
      {
        return new ChiFactory(*this);
      }


      /* Here is the interface that all derived class must implement */
      Chi * ChiFactory::build(const NumericalSample & sample) const /* throw(InvalidArgumentException, InternalException) */
      {
        const UnsignedLong size(sample.getSize());
        if (size == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a Chi distribution from an empty sample";
        if (sample.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: can build a Chi distribution only from a sample of dimension 1, here dimension=" << sample.getDimension();
        NumericalScalar sumSquares(0.0);
        for (UnsignedLong i = 0; i < size; ++i)
          {
            const NumericalScalar xI(sample[i][0]);
            sumSquares += xI * xI;
          }
        try
          {
            return Chi( sumSquares / size ).clone();
          }
        catch (InvalidArgumentException & ex)
          {
            throw InternalException(HERE) << "Error: cannot estimate parameters of a Chi distribution from the given sample";
          }
      }

      Chi * ChiFactory::build(const NumericalPointCollection & parameters) const /* throw(InvalidArgumentException, InternalException) */
      {
        try {
          Chi distribution;
          distribution.setParametersCollection(parameters);
          return distribution.clone();
        }
        catch (InvalidArgumentException & ex)
          {
            throw InternalException(HERE) << "Error: cannot build a Chi distribution from the given parameters";
          }
      }

      Chi * ChiFactory::build() const /* throw(InvalidArgumentException, InternalException) */
      {
        return Chi().clone();
      }


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
