//                                               -*- C++ -*-
/**
 *  @file  QuasiMonteCarlo.cxx
 *  @brief Monte Carlo simulation using low-discrepancy sequences
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: QuasiMonteCarlo.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include "QuasiMonteCarlo.hxx"
#include "ComparisonOperatorImplementation.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {

      typedef Base::Type::NumericalPoint                NumericalPoint;

      /*
       * @class QuasiMonteCarlo
       */
      CLASSNAMEINIT(QuasiMonteCarlo);


      /* Constructor with parameters */
      QuasiMonteCarlo::QuasiMonteCarlo(const Simulation::Event & event,
                                       const LowDiscrepancySequence & lowDiscrepancySequence) :
        Simulation(event),
        dimension_(event.getImplementation()->getAntecedent()->getDimension()),
        lowDiscrepancySequence_(lowDiscrepancySequence)
      {
        if (!event.getImplementation()->getAntecedent()->getDistribution().hasIndependentCopula()) throw InvalidArgumentException(HERE) << "Error: cannot use the QuasiMonteCarlo algorithm with an input distribution whose components are not independent.";
        // retrieve the marginal laws
        for (UnsignedLong index = 0; index < dimension_; ++index) marginals_.add(event.getImplementation()->getAntecedent()->getDistribution().getMarginal(index));

        // initialize the low-discrepancy sequence
        lowDiscrepancySequence_.initialize(dimension_);

        result_ = QuasiMonteCarloResult();
      }


      /* Virtual constructor */
      QuasiMonteCarlo * QuasiMonteCarlo::clone() const
      {
        return new QuasiMonteCarlo(*this);
      }


      /* Compute the block sample */
      QuasiMonteCarlo::NumericalSample QuasiMonteCarlo::computeBlockSample(NumericalSample & realizedEventSample)
      {
        // Size of a block
        const UnsignedLong blockSize(getBlockSize());

        // allocate the input sample
        NumericalSample inputSample(lowDiscrepancySequence_.generate(blockSize));

        // for each point of the sample
        for(UnsignedLong index = 0; index < blockSize; ++index)
          {
            // for each component
            for(UnsignedLong component = 0; component < dimension_; ++component)
              // use marginal laws to compute quantile from the low-discrepancy value to build the input sample
              inputSample[index][component] = marginals_[component].computeQuantile(inputSample[index][component])[0];
          } // For index

        // Then, evaluate the function on this sample
        NumericalSample blockSample(event_.getImplementation()->getFunction()(inputSample));
        realizedEventSample = NumericalSample(blockSize_, inputSample.getDimension());
        for (UnsignedLong i = 0; i < blockSize_; ++i)
          {
            // store input and output values
            inputStrategy_.store(inputSample[i]);
            outputStrategy_.store(blockSample[i]);
            const Bool isRealized(getEvent().getOperator()(blockSample[i][0], event_.getThreshold()));
            if (isRealized)
              {
                blockSample[i][0] = 1.0;
                realizedEventSample[i] = inputSample[i];
              }
            else blockSample[i][0] = 0.0;
          }
        return blockSample;
      }


      /* String converter */
      String QuasiMonteCarlo::__repr__() const
      {
        OSS oss;
        oss << "class=" << QuasiMonteCarlo::GetClassName()
            << " derived from " << Simulation::__repr__();
        return oss;
      }


    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
