//                                               -*- C++ -*-
/**
 * @file  SVMKernelRegressionHessian.cxx
 * @brief Implementation of SVM regression hessian
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2008-05-05 08:50:15 +0200 (lun, 05 mai 2008) $
 *  Id:      $Id: SVMKernelRegressionHessian.cxx 805 2008-05-05 06:50:15Z schueller $
 */

#include "SVMKernelRegressionHessian.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {

      CLASSNAMEINIT(SVMKernelRegressionHessian);


      /* Default constructor */
      SVMKernelRegressionHessian::SVMKernelRegressionHessian() :
        Base::Func::NumericalMathHessianImplementation()
      {
        // nothing to do
      }


      /* Constructor with parameters */
      SVMKernelRegressionHessian::SVMKernelRegressionHessian(const SVMKernel & kernel,
                                                             const NumericalPoint & lagrangeMultiplier,
                                                             const NumericalSample & dataIn,
                                                             const NumericalScalar constant) :
        kernel_(kernel),
        lagrangeMultiplier_(lagrangeMultiplier),
        dataIn_(dataIn),
        constant_(constant)
      {
        // Nothing to do
      }


      /* Constructor from SVMKernelRegressionEvaluation */
      SVMKernelRegressionHessian::SVMKernelRegressionHessian(const SVMKernelRegressionEvaluation & evaluation) :
        Base::Func::NumericalMathHessianImplementation()
        //  :
        //      kernel_(kernel),
        //      dataIn_(dataIn),
        //      constant_(constant)
      {
        // Nothing to do
      }


      /* Virtual constructor */
      SVMKernelRegressionHessian * SVMKernelRegressionHessian::clone() const
      {
        return new SVMKernelRegressionHessian(*this);
      }

      /* Comparison operator */
      Bool SVMKernelRegressionHessian::operator==(const SVMKernelRegressionHessian & other) const
      {
        return true;
      }

      /* String converter */
      String SVMKernelRegressionHessian::__repr__() const {
        OSS oss;
        oss << "class=" << SVMKernelRegressionHessian::GetClassName()
            << " name=" << getName()
            << " kernel=" << kernel_
            << " lagrange multipliers=" << lagrangeMultiplier_
            << " data in=" << dataIn_
            << " constant=" << constant_;
        return oss;
      }

      /* Test for actual implementation */
      Bool SVMKernelRegressionHessian::isActualImplementation() const
      {
        return true;
      }

      /* Hessian method */
      SVMKernelRegressionHessian::SymmetricTensor SVMKernelRegressionHessian::hessian(const NumericalPoint & inP) const
      /* throw(InvalidArgumentException,InternalException) */
      {
        ++ callsNumber_;

        UnsignedLong dimension = inP.getDimension();
        if(dimension != dataIn_.getDimension())
          throw InvalidArgumentException(HERE) << "Invalid input dimension=" << dimension;

        // compute the sum of the partial Hessians
        UnsignedLong size = dataIn_.getSize();
        SymmetricMatrix partialHessian(dimension);
        for(UnsignedLong i = 0; i < size; ++ i)
          {
            if(lagrangeMultiplier_[i] != 0.0)
              partialHessian = partialHessian + lagrangeMultiplier_[i] * kernel_.partialHessian(inP, dataIn_[i]);
          }

        // return the result into a symmetric tensor
        SymmetricTensor hessian(dimension, 1);
        for(UnsignedLong i = 0; i < dimension; ++ i)
          {
            for(UnsignedLong j = 0; j <= i; ++ j)
              {
                hessian(i, j, 0) = partialHessian(i, j);
              }
          }
        return hessian;
      }

      /* Accessor for input point dimension */
      UnsignedLong SVMKernelRegressionHessian::getInputDimension() const
      /* throw(InternalException) */
      {
        return dataIn_.getDimension();
      }

      /* Accessor for output point dimension */
      UnsignedLong SVMKernelRegressionHessian::getOutputDimension() const
      /* throw(InternalException) */
      {
        return 1;
      }


    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
