//                                               -*- C++ -*-
/**
 *  @file  InverseNatafIndependentCopulaEvaluation.cxx
 *  @brief Class for the InverseNataf transformation evaluation for independent
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-04-11 12:32:27 +0200 (Mon, 11 Apr 2011) $
 *  Id:      $Id: InverseNatafIndependentCopulaEvaluation.cxx 1866 2011-04-11 10:32:27Z schueller $
 */
#include "InverseNatafIndependentCopulaEvaluation.hxx"
#include "DistFunc.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {

      /*
       * @class InverseNatafIndependentCopulaEvaluation
       *
       * This class offers an interface for the InverseNataf function for independent copula
       */

      typedef Distribution::DistFunc DistFunc;

      CLASSNAMEINIT(InverseNatafIndependentCopulaEvaluation);

      /* Parameter constructor */
      InverseNatafIndependentCopulaEvaluation::InverseNatafIndependentCopulaEvaluation(const UnsignedLong dimension):
        NumericalMathEvaluationImplementation(),
        dimension_(dimension)
      {
        Description description;
        for (UnsignedLong i = 0; i < dimension_; ++i)
          {
            OSS oss;
            oss << "x" << i;
            description.add(oss);
          }
        for (UnsignedLong i = 0; i < dimension_; ++i)
          {
            OSS oss;
            oss << "y" << i;
            description.add(oss);
          }
        setDescription(description);
      }

      /* Virtual constructor */
      InverseNatafIndependentCopulaEvaluation * InverseNatafIndependentCopulaEvaluation::clone() const
      {
        return new InverseNatafIndependentCopulaEvaluation(*this);
      }

      /* String converter */
      String InverseNatafIndependentCopulaEvaluation::__repr__() const
      {
        OSS oss;
        oss << "class=" << InverseNatafIndependentCopulaEvaluation::GetClassName()
            << " description=" << getDescription()
            << " dimension=" << dimension_;

        return oss;
      }

      /*
       * Evaluation
       * The inverse Nataf transform S reads:
       * Si(u) = Phi(ui), where Phi is the CDF of the standard normal distribution
       */
      InverseNatafIndependentCopulaEvaluation::NumericalPoint InverseNatafIndependentCopulaEvaluation::operator () (const NumericalPoint & inP) const
      /* throw (InvalidArgumentException, InternalException) */
      {
        NumericalPoint result(dimension_);
        for (UnsignedLong i = 0; i < dimension_; ++i)
          {
            result[i] = DistFunc::pNormal(inP[i]);
          }
        return result;
      }

      /* Gradient according to the marginal parameters. Currently, the dependence parameter are not taken into account. */

      InverseNatafIndependentCopulaEvaluation::Matrix InverseNatafIndependentCopulaEvaluation::parametersGradient(const NumericalPoint & inP) const
      {
        return Matrix(0, dimension_);
      }

      /* Accessor for input point dimension */
      UnsignedLong InverseNatafIndependentCopulaEvaluation::getInputDimension() const
      /* throw(InternalException) */
      {
        return dimension_;
      }

      /* Accessor for output point dimension */
      UnsignedLong InverseNatafIndependentCopulaEvaluation::getOutputDimension() const
      /* throw(InternalException) */
      {
        return dimension_;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
