//                                               -*- C++ -*-
/**
 *  @file  Pairs.cxx
 *  @brief Pairs class for contourchart plots
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: lebrun $
 *  @date:   $LastChangedDate: 2011-02-26 21:12:17 +0100 (sam. 26 févr. 2011) $
 *  Id:      $Id: Pairs.cxx 1813 2011-02-26 20:12:17Z lebrun $
 */
#include <cstdio>
#include <cstdlib>


#include "PersistentObjectFactory.hxx"
#include "Log.hxx"
#include "ResourceMap.hxx"
#include "Pairs.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Graph {

      CLASSNAMEINIT(Pairs);

      typedef Common::Log         Log;
      typedef Common::ResourceMap ResourceMap;

      static Common::Factory<Pairs> RegisteredFactory("Pairs");

      /* Default constructor */
      Pairs::Pairs(const NumericalSample & data,
                   const String & title)
        /* throw(InvalidArgumentException) */
        :DrawableImplementation(data, NoSpecifiedLabel)
      {
        if(!isValidData(data)) throw InvalidArgumentException(HERE) << "Error: the data dimension must be greater or equal to 2";
        labels_ = data_.getDescription();
        if(labels_.getSize() == 0) buildDefaultLabels();
      }

      /* Constructor with parameters*/
      Pairs::Pairs(const NumericalSample & data,
                   const String & title,
                   const Description labels,
                   const String color,
                   const String pointStyle)
        /* throw(InvalidArgumentException) */
        :DrawableImplementation(data, NoSpecifiedLabel)
      {
        if(!isValidData(data)) throw InvalidArgumentException(HERE) << "Error: the data dimension must be greater or equal to 2";
        setTitle(title);
        setColor(color);
        setLabels(labels);
        setPointStyle(pointStyle);
      }



      /* String converter */
      String Pairs::__repr__() const
      {
        OSS oss;
        oss << " class=" << Pairs::GetClassName()
            << " name=" << getName()
            << " data=" << data_
            << " derived from " << DrawableImplementation::__repr__();
        return oss;
      }

      /* Accessor for first coordinate */
      Pairs::NumericalSample Pairs::getData() const
      {
        return data_;
      }

      void Pairs::setData(const NumericalSample & data)
      {
        if(!isValidData(data)) throw InvalidArgumentException(HERE) << "Error: the data dimension must be greater or equal to 2";
        data_ = data;
      }

      /* Accessor for labels */
      Pairs::Description Pairs::getLabels() const
      {
        return labels_;
      }

      void Pairs::setLabels(const Description & labels)
      /* throw(InvalidArgumentException) */
      {
        if (labels.getSize() != data_.getDimension()) throw InvalidArgumentException(HERE) << "Error: the labels size must be equal equal to the data dimension";
        labels_ = labels;
      }

      /* Clean all the temporary data created by draw() method */
      void Pairs::clean() const
      {
        if ((dataFileName_ != "") && (remove(dataFileName_.c_str()) == -1)) LOGWARN(OSS() << "GraphImplementation: error trying to remove file " << dataFileName_);
      }

      /* Draw method */
      String Pairs::draw() const
      {
        dataFileName_ = "";
        OSS oss;
        const String code((OSS() << getPointCode(pointStyle_)));
        // Stores the data in a temporary file
        // The specific R command for drawing
        oss << "dim_ <- " << data_.getDimension() << "\n" ;
        oss << "size_ <- " << data_.getSize() << "\n";
        oss << DrawableImplementation::draw() << "\n";
        oss << "description=c(";
        const UnsignedLong length(data_.getDimension());
        for(UnsignedLong i = 0; i < length - 1; ++i) oss << "\"" << labels_[i] <<"\"" << ",";
        oss << "\"" << labels_[length -1] <<"\"" << ") \n";
        // DataFrame organisation
        oss << "data = data.frame(data) \n";
        oss << "names(data) <- description \n";
        oss << "points(pairs(data "
            << ",pch=" << (code == "dot" ? "\".\"" : code)
            << ",col=\"" << color_ << "\""
            << ",main=\""<< getTitle() << "\"))";
        return oss;
      }

      /* Clone method */
      Pairs * Pairs::clone() const
      {
        return new Pairs(*this);
      }

      String Pairs::getTitle() const
      {
        return title_;
      }
      void Pairs::setTitle(const String & title)
      {
        title_ = title;
      }
      /* throw(InvalidArgumentException) */;


      /* Check for data validity */
      Bool Pairs::isValidData(const NumericalSample & data) const
      {
        return (data.getDimension() >= 2);
      }

      /* Build default labels by taking the level values */
      void Pairs::buildDefaultLabels()
      {
        const UnsignedLong number(data_.getDimension());
        labels_ = Description(number);
        for (UnsignedLong i = 0; i < number; ++i) labels_[i] = OSS() << "V" << i + 1;
      }

      /* Method save() stores the object through the StorageManager */
      void Pairs::save(StorageManager::Advocate & adv) const
      {
        DrawableImplementation::save(adv);
        adv.saveAttribute( "data_", data_ );
        adv.saveAttribute( "labels_", labels_ );
        adv.saveAttribute( "color_", color_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void Pairs::load(StorageManager::Advocate & adv)
      {
        DrawableImplementation::load(adv);
        adv.loadAttribute( "data_", data_ );
        adv.loadAttribute( "labels_", labels_ );
        adv.loadAttribute( "color_", color_ );
      }

    }/* namespace Graph */

  }/*namespace Base */

}/* namespace OpenTURNS */
