//                                               -*- C++ -*-
/**
 *  @file  Graph.hxx
 *  @brief Graph implements graphic devices for plotting through R
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-07-28 18:31:43 +0200 (Thu, 28 Jul 2011) $
 *  Id:      $Id: Graph.hxx 2046 2011-07-28 16:31:43Z schueller $
 */
#ifndef OPENTURNS_GRAPH_HXX
#define OPENTURNS_GRAPH_HXX

#include "TypedInterfaceObject.hxx"
#include "GraphImplementation.hxx"
#include "DrawableImplementation.hxx"
#include "Collection.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Graph
    {
      /**
       * @class Graph
       *
       * Graph implements graphic devices for plotting through R,
       * and manages drawables to be plotted on the same window
       */

      class Graph :
        public Common::TypedInterfaceObject<GraphImplementation>
      {

        CLASSNAME;

      public:

        typedef GraphImplementation::InvalidArgumentException InvalidArgumentException;
        typedef GraphImplementation::InternalException        InternalException;
        typedef GraphImplementation::BoundingBox              BoundingBox;
        typedef GraphImplementation::Interval                 Interval;
        typedef GraphImplementation::Description              Description;
        typedef GraphImplementation::DrawableCollection       DrawableCollection;
        typedef GraphImplementation::Format                   Format;
        typedef GraphImplementation::LogScale                 LogScale;
        typedef Type::Collection<Base::Graph::Graph>          GraphCollection;

        /** Default constructor */
        explicit Graph();

        /** Constructor from implementation */
        Graph(const GraphImplementation & implementation);

        /** Constructor with parameters */
        Graph(const String & title,
              const String & xTitle,
              const String & yTitle,
              const Bool showAxes,
              const String & legendPosition = GraphImplementation::NoSpecifiedLabel,
              const NumericalScalar legendFontSize = GraphImplementation::DefaultLegendFontSize,
	      const LogScale logScale = GraphImplementation::NONE) /* throw(InvalidArgumentException) */;


        /** String converter */
        String __repr__() const;

        /** String converter */
        String __str__(const String & offset = "") const;

        /** Adds a drawable instance to the collection of drawables contained in Graph */
        void add(const Drawable & aDrawable);

        /** Adds a collection of drawable instances to the collection of drawables contained in GraphImplementation */
        void add(const DrawableCollection & drawableCollection);

        /** Adds a drawable instance to the collection of drawables contained in Graph
         * @deprecated */
        void addDrawable(const Drawable & aDrawable) DEPRECATED;

        /** Adds a collection of drawable instances to the collection of drawables contained in GraphImplementation
         * @deprecated */
        void addDrawables(const DrawableCollection & drawableCollection) DEPRECATED;

        /** Drawables accessor */
        DrawableCollection getDrawables() const;
        void setDrawables(const DrawableCollection & drawableCollection);

        /** Individual drawable accessor */
        Drawable getDrawable(const UnsignedLong index) const;
        void setDrawable(const Drawable & drawable,
                         const UnsignedLong index);

        /** Hide or show x and y axes */
        void setAxes(const Bool showAxes);

        /** Accessor for showAxes_ */
        Bool getAxes() const;

        /** Set log scale for x, y both or none axes */
        void setLogScale(const LogScale logScale);

        /** Accessor for logScale_ */
        LogScale getLogScale() const;

        /** Hide or show grid */
        void setGrid(const Bool showGrid);

        /** Accessor for showGrid_ */
        Bool getGrid() const;

        /** Accesor for xTitle */
        String getXTitle() const;

        /** Accessor for xTitle */
        void setXTitle(const String & title);

        /** Accessor for yTitle */
        String getYTitle() const;

        /** Accessor for yTitle */
        void setYTitle(const String & title);

        /** Accesor for title */
        String getTitle() const;

        /** Accesor for title */
        void setTitle(const String & title);

        /** Accessor for path */
        String getPath() const;

        /** Accessor for file name */
        String getFileName() const;

        /** The method that generates the graphic files */
        void draw(const String & path,
                  const String & file,
                  const NumericalScalar width,
                  const NumericalScalar height,
                  const Format format = GraphImplementation::ALL) /* throw(InternalException) */;

        /** The method that generates the graphic files */
        void draw(const String & file,
                  const NumericalScalar width = GraphImplementation::DefaultWidth,
                  const NumericalScalar height = GraphImplementation::DefaultHeight,
                  const Format format = GraphImplementation::ALL) /* throw(InternalException) */;

        /** Get the R command corresponding to the graph */
        String getRCommand() const;

        /** Clean temporary files */
        void clean();

        /** The method returning absolute path of the bitmap graphic file */
        String getBitmap() const;

        /** The method returning absolute path of the postscript graphic file */
        String getPostscript() const;

        /** The method returning absolute path of the vectorial graphic file */
        String getVectorial() const;

        /** Get the bounding box of the whole plot */
        BoundingBox getBoundingBox() const;

        /** Set the bounding box of the whole plot */
        void setBoundingBox(const BoundingBox & boundingBox);
        void setBoundingBox(const Interval & boundingBox);

        /** Automatic bounding box accessor */
        Bool getAutomaticBoundingBox() const;
        void setAutomaticBoundingBox(const Bool automaticBoundingBox);

        /** Get the legend position */
        String getLegendPosition() const;

        /** Set the legend position */
        void setLegendPosition(const String & position) /* throw(InvalidArgumentException) */;

        /** Get the legend font size */
        NumericalScalar getLegendFontSize() const;

        /** Set the legend font size */
        void setLegendFontSize(const NumericalScalar legendFontSize) /* throw(InvalidArgumentException) */;

        /** Check for legend position validity */
        Bool isValidLegendPosition(const String & position) const;

        /** Initialize valid legend positions **/
        void initializeValidLegendPositions();

        /** Gives all the valid legend positions */
        Description getValidLegendPositions();

        /** Gives all the valid legend positions */
        static Description GetValidLegendPositions();

      }; /* class Graph */

    }/* namespace Graph */

  }/* namespace Base */

}/* namespace OpenTURNS */

#endif /* OPENTURNS_GRAPH_HXX */
