//                                               -*- C++ -*-
/**
 *  @file  Graph.cxx
 *  @brief Graph implements graphic devices for plotting through R
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-07-28 18:31:43 +0200 (Thu, 28 Jul 2011) $
 *  Id:      $Id: Graph.cxx 2046 2011-07-28 16:31:43Z schueller $
 */
#include "Graph.hxx"
#include "Log.hxx"

namespace OpenTURNS
{
  namespace Base
  {
    namespace Graph
    {
      CLASSNAMEINIT(Graph);

      typedef Common::Log Log;

      /* Initialize valid legend positions */
      void Graph::initializeValidLegendPositions()
      {
        getImplementation()->initializeValidLegendPositions();
      }

      /* Default constructor */
      Graph::Graph():
        Common::TypedInterfaceObject<GraphImplementation>(new GraphImplementation())
      {
        // Nothing to do
      }

      /* Constructor from implementation */
      Graph::Graph(const GraphImplementation & implementation)
        : Common::TypedInterfaceObject<GraphImplementation>(implementation.clone())
      {
        // Nothing to do
      }

      /* Constructor with parameters */
      Graph::Graph(const String & title,
                   const String & xTitle,
                   const String & yTitle,
                   const Bool showAxes,
                   const String & legendPosition,
                   const NumericalScalar legendFontSize,
		   const LogScale logScale) /* throw(InvalidArgumentException) */
        : Common::TypedInterfaceObject<GraphImplementation>(new GraphImplementation(title, xTitle, yTitle, showAxes, legendPosition, legendFontSize, logScale))
      {
        // Nothing to do
      }

      /* String converter */
      String Graph::__repr__() const
      {
        return OSS() << "class=" << Graph::GetClassName()
                     << " name=" << getName()
                     << " implementation=" << getImplementation()->__repr__();
      }

      /* String converter */
      String Graph::__str__(const String & offset) const
      {
        return __repr__();
      }

      /* Adds a drawable instance to the collection of drawables contained in Graph */
      void Graph::add(const Drawable & aDrawable)
      {
        copyOnWrite();
        getImplementation()->add(aDrawable);
      }

      /* Adds a collection of drawable instances to the collection of drawables contained in GraphImplementation */
      void Graph::add(const DrawableCollection & drawableCollection)
      {
        copyOnWrite();
        getImplementation()->add(drawableCollection);
      }

      /* Adds a drawable instance to the collection of drawables contained in Graph */
      void Graph::addDrawable(const Drawable & aDrawable)
      {
        LOGWARN(OSS() << "The method addDrawable(const Drawable & aDrawable) is deprecated and should not be used anymore. Use add(const Drawable & aDrawable) instead.");
        add(aDrawable);
      }

      /* Adds a collection of drawable instances to the collection of drawables contained in GraphImplementation */
      void Graph::addDrawables(const DrawableCollection & drawableCollection)
      {
        LOGWARN(OSS() << "The method addDrawable(const DrawableCollection & drawableCollection) is deprecated and should not be used anymore. Use add(const DrawableCollection & drawableCollection) instead.");
        add(drawableCollection);
      }

      /* Drawables accessor */
      Graph::DrawableCollection Graph::getDrawables() const
      {
        return getImplementation()->getDrawables();
      }

      void Graph::setDrawables(const DrawableCollection & drawableCollection)
      {
        copyOnWrite();
        getImplementation()->setDrawables(drawableCollection);
      }
      /* Individual drawable accessor */
      Drawable Graph::getDrawable(const UnsignedLong index) const
      {
        return getImplementation()->getDrawable(index);
      }

      void Graph::setDrawable(const Drawable & drawable,
                              const UnsignedLong index)
      {
        copyOnWrite();
        getImplementation()->setDrawable(drawable, index);
      }


      /* Hide or show x and y axes */
      void Graph::setAxes(const Bool showAxes)
      {
        copyOnWrite();
        getImplementation()->setAxes(showAxes);
      }

      /* Accessor for showAxes_ */
      Bool Graph::getAxes() const
      {
        return getImplementation()->getAxes();
      }

      /* Set log scale for x, y both or none axes */
      void Graph::setLogScale(const LogScale logScale)
      {
        copyOnWrite();
        getImplementation()->setLogScale(logScale);
      }

      /* Accessor for showAxes_ */
      Graph::LogScale Graph::getLogScale() const
      {
        return getImplementation()->getLogScale();
      }

      /* Hide or show grid */
      void Graph::setGrid(const Bool showGrid)
      {
        copyOnWrite();
        getImplementation()->setGrid(showGrid);
      }

      /* Accessor for showAxes_ */
      Bool Graph::getGrid() const
      {
        return getImplementation()->getGrid();
      }

      /* Accesor for xTitle */
      String Graph::getXTitle() const
      {
        return getImplementation()->getXTitle();
      }

      /* Accessor for xTitle */
      void Graph::setXTitle(const String & title)
      {
        copyOnWrite();
        getImplementation()->setXTitle(title);
      }

      /* Accessor for yTitle */
      String Graph::getYTitle() const
      {
        return getImplementation()->getYTitle();
      }

      /* Accessor for yTitle */
      void Graph::setYTitle(const String & title)
      {
        copyOnWrite();
        getImplementation()->setYTitle(title);
      }

      /* Accesor for title */
      String Graph::getTitle() const
      {
        return getImplementation()->getTitle();
      }

      /* Accesor for title */
      void Graph::setTitle(const String & title)
      {
        copyOnWrite();
        getImplementation()->setTitle(title);
      }

      /* Accessor for path */
      String Graph::getPath() const
      {
        return getImplementation()->getPath();
      }

      /* Accessor for file name */
      String Graph::getFileName() const
      {
        return getImplementation()->getFileName();
      }

      /* The method that generates the graphic files */
      void Graph::draw(const String & path,
                       const String & file,
                       const NumericalScalar width,
                       const NumericalScalar height,
                       const Format format) /* throw(InternalException) */
      {
        getImplementation()->draw(path, file, width, height, format);
      }

      /* The method that generates the graphic files */
      void Graph::draw(const String & file,
                       const NumericalScalar width,
                       const NumericalScalar height,
                       const Format format) /* throw(InternalException) */
      {
        getImplementation()->draw(file, width, height, format);
      }

      /* Get the R command corresponding to the graph */
      String Graph::getRCommand() const
      {
        return getImplementation()->getRCommand();
      }

      /* Clean temporary files */
      void Graph::clean()
      {
        getImplementation()->clean();
      }

      /* The method returning absolute path of the bitmap graphic file */
      String Graph::getBitmap() const
      {
        return getImplementation()->getBitmap();
      }

      /* The method returning absolute path of the postscript graphic file */
      String Graph::getPostscript() const
      {
        return getImplementation()->getPostscript();
      }

      /* The method returning absolute path of the vectorial graphic file */
      String Graph::getVectorial() const
      {
        return getImplementation()->getVectorial();
      }

      /* Get the bounding box of the whole plot */
      Graph::BoundingBox Graph::getBoundingBox() const
      {
        return getImplementation()->getBoundingBox();
      }

      /* Set the bounding box of the whole plot */
      void Graph::setBoundingBox(const BoundingBox & boundingBox)
      {
        copyOnWrite();
        getImplementation()->setBoundingBox(boundingBox);
      }

      void Graph::setBoundingBox(const Interval & boundingBox)
      {
        copyOnWrite();
        getImplementation()->setBoundingBox(boundingBox);
      }

      /* Automatic bounding box accessor */
      Bool Graph::getAutomaticBoundingBox() const
      {
        return getImplementation()->getAutomaticBoundingBox();
      }

      void Graph::setAutomaticBoundingBox(const Bool automaticBoundingBox)
      {
        copyOnWrite();
        getImplementation()->setAutomaticBoundingBox(automaticBoundingBox);
      }

      /* Get the legend position */
      String Graph::getLegendPosition() const
      {
        return getImplementation()->getLegendPosition();
      }

      /* Set the legend position */
      void Graph::setLegendPosition(const String & position) /* throw(InvalidArgumentException) */
      {
        copyOnWrite();
        return getImplementation()->setLegendPosition(position);
      }

      /* Gives all the valid legend positions */
      Graph::Description Graph::getValidLegendPositions()
      {
        return getImplementation()->GetValidLegendPositions();
      }

      /* Get the legend font size */
      NumericalScalar Graph::getLegendFontSize() const
      {
        return getImplementation()->getLegendFontSize();
      }

      /* Set the legend font size */
      void Graph::setLegendFontSize(const NumericalScalar legendFontSize) /* throw(InvalidArgumentException) */
      {
        copyOnWrite();
        return getImplementation()->setLegendFontSize(legendFontSize);
      }

      /* check for legend position validity */
      Bool Graph::isValidLegendPosition(const String & position) const
      {
        return getImplementation()->isValidLegendPosition(position);
      }

      /* Gives all the valid legend positions */
      Graph::Description Graph::GetValidLegendPositions()
      {
        return GraphImplementation::GetValidLegendPositions();
      }

    } /* namespace Graph */

  }/* namespace Base */

}/* namespace OpenTURNS */
