//                                               -*- C++ -*-
/**
 *  @file  Contour.hxx
 *  @brief Contour class for piechart plots
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-04-11 12:32:27 +0200 (Mon, 11 Apr 2011) $
 *  Id:      $Id: Contour.hxx 1866 2011-04-11 10:32:27Z schueller $
 */
#ifndef OPENTURNS_CONTOUR_HXX
#define OPENTURNS_CONTOUR_HXX

#include "DrawableImplementation.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Graph {

      /**
       * @class Contour
       *
       * The class describing a contour chart
       * Instance of Drawable
       */


      class Contour : public DrawableImplementation
      {
        CLASSNAME;

      public:

        static const UnsignedLong DefaultLevelsNumber;

        typedef DrawableImplementation::NumericalPoint           NumericalPoint;
        typedef DrawableImplementation::Description              Description;
        typedef DrawableImplementation::NumericalSample          NumericalSample;
        typedef DrawableImplementation::InvalidArgumentException InvalidArgumentException;

        /** Default constructor */
        Contour(const UnsignedLong dimX,
                const UnsignedLong dimY,
                const NumericalSample & data,
                const String & legend = NoSpecifiedLabel)
          /* throw(InvalidArgumentException) */;

        /** Constructor with parameters */
        Contour(const NumericalSample & x,
                const NumericalSample & y,
                const NumericalSample & data,
                const NumericalPoint & levels,
                const Description & labels,
                const Bool drawLabels = true,
                const String & legend = NoSpecifiedLabel)
          /* throw(InvalidArgumentException) */;

        /** Constructor with parameters
            Contour(const NumericalSample & xy,
            const NumericalSample & data,
            const NumericalPoint & levels,
            const Description & labels,
            const Bool drawLabels = True,
            const String & legend = NoSpecifiedLabel); */
        /* throw(InvalidArgumentException) */

        /** String converter */
        String __repr__() const;

        /** Accessor for first coordinate */
        NumericalSample getX() const;
        void setX(const NumericalSample & x);

        /** Accessor for second coordinate */
        NumericalSample getY() const;
        void setY(const NumericalSample & y);

        /** Accessor for levels */
        NumericalPoint getLevels() const;
        void setLevels(const NumericalPoint & levels);

        /** Accessor for labels */
        Description getLabels() const;
        void setLabels(const Description & labels)
          /* throw(InvalidArgumentException) */;

        /** Accessor for drawLabels */
        Bool getDrawLabels() const;
        void setDrawLabels(const Bool & drawLabels);

        /** Check for data validity */
        Bool isValidData(const NumericalSample & data) const;

        /** Accessor for boundingbox */
        BoundingBox getBoundingBox() const;

        /** Draw method */
        String draw() const;

        /** Clone method */
        Contour * clone() const;

        /** Build default levels by sampling uniformly the range [min(data), max(data)] into DefaultLevelsNumber points
            starting from min(data) + 0.5 * (max(data) - min(data)) / DefaultLevelsNumber
        */
        void buildDefaultLevels(const UnsignedLong number = DefaultLevelsNumber);

        /** Build default labels by taking the level values */
        void buildDefaultLabels();

        /** Method save() stores the object through the StorageManager */
        void save(StorageManager::Advocate & adv) const;

        /** Method load() stores the object through the StorageManager */
        void load(StorageManager::Advocate & adv);

        /** Clean all the temporary data created by draw() method */
        virtual void clean() const;

      private:

        Contour() {};
        friend class Common::Factory<Contour>;

        /** Sample of first coordinate */
        NumericalSample x_;

        /** Sample of second coordinate */
        NumericalSample y_;

        /** Values of the level sets */
        NumericalPoint levels_;

        /** Labels of the level sets. If none is given, it defaults to the level values. */
        Description labels_;

        /** Flag for drawing the labels */
        Bool drawLabels_;

        /** Temporary file name for the exchange with R */
        mutable String xFileName_;

        /** Temporary file name for the exchange with R */
        mutable String yFileName_;

      }; /* class Contour */

    }/* namespace Graph */

  }/*namespace Base */

}/* namespace OpenTURNS */

#endif /* OPENTURNS_CONTOUR_HXX */
