//                                               -*- C++ -*-
/**
 *  @file  LinearCombinationHessianImplementation.cxx
 *  @brief The hessian part of linear combination of polynomials
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include "LinearCombinationHessianImplementation.hxx"
#include "OSS.hxx"
#include "PersistentObjectFactory.hxx"
#include "SymmetricMatrix.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Func
    {

      CLASSNAMEINIT(LinearCombinationHessianImplementation);

      static Base::Common::Factory<LinearCombinationHessianImplementation> RegisteredFactory("LinearCombinationHessianImplementation");

      typedef Type::SymmetricMatrix SymmetricMatrix;

      /* Default constructor */
      LinearCombinationHessianImplementation::LinearCombinationHessianImplementation()
        : NumericalMathHessianImplementation(),
          evaluation_()
      {
        // Nothing to do
      }


      /* Parameters constructor */
      LinearCombinationHessianImplementation::LinearCombinationHessianImplementation(const LinearCombinationEvaluationImplementation & evaluation)
        : NumericalMathHessianImplementation(),
          evaluation_(evaluation)
      {
        // Nothing to do
      }

      /* Virtual constructor */
      LinearCombinationHessianImplementation * LinearCombinationHessianImplementation::clone() const
      {
        return new LinearCombinationHessianImplementation(*this);
      }


      /* Hessian method */
      LinearCombinationHessianImplementation::SymmetricTensor LinearCombinationHessianImplementation::hessian(const NumericalPoint & inP) const
      /* throw(InvalidArgumentException, InternalException) */
      {
        const UnsignedLong inputDimension(getInputDimension());
        if (inP.getDimension() != inputDimension) throw InvalidArgumentException(HERE) << "Error: the given point has an invalid dimension. Expect a dimension " << inputDimension << ", got " << inP.getDimension();
        const UnsignedLong size(evaluation_.functionsCollection_.getSize());
        const UnsignedLong sheetSize(evaluation_.getOutputDimension());
        SymmetricTensor result(evaluation_.getInputDimension(), sheetSize);
        // We work on a sheet basis because there is no tensor arithmetic
        for (UnsignedLong i = 0; i < size; ++i)
          {
            const SymmetricTensor currentTensor(evaluation_.functionsCollection_[i].hessian(inP));
            const NumericalScalar factor(evaluation_.coefficients_[i]);
            for (UnsignedLong k = 0; k < sheetSize; ++k)
              {
                SymmetricMatrix sheet(result.getSheet(k) + factor * currentTensor.getSheet(k));
                result.setSheet(k, sheet);
              }
          }
        return result;
      }

      /* Accessor for input point dimension */
      UnsignedLong LinearCombinationHessianImplementation::getInputDimension() const
      /* throw(InternalException) */
      {
        return evaluation_.getInputDimension();
      }

      /* Accessor for output point dimension */
      UnsignedLong LinearCombinationHessianImplementation::getOutputDimension() const
      /* throw(InternalException) */
      {
        return evaluation_.getOutputDimension();
      }

      /* String converter */
      String LinearCombinationHessianImplementation::__repr__() const
      {
        return OSS() << "class=" << GetClassName()
                     << " evaluation=" << evaluation_;
      }


      /* Method save() stores the object through the StorageManager */
      void LinearCombinationHessianImplementation::save(StorageManager::Advocate & adv) const
      {
        PersistentObject::save(adv);
        adv.saveAttribute( "evaluation_", evaluation_ );
      }


      /* Method load() reloads the object from the StorageManager */
      void LinearCombinationHessianImplementation::load(StorageManager::Advocate & adv)
      {
        PersistentObject::load(adv);
        adv.loadAttribute( "evaluation_", evaluation_ );
      }



    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
