//                                               -*- C++ -*-
/**
 * @file  FunctionCache.cxx
 * @brief Cache of NumericalMathFunction
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 * \author $LastChangedBy: schueller $
 * \date   $LastChangedDate: 2011-01-26 14:50:44 +0100 (Wed, 26 Jan 2011) $
 */
#include "FunctionCache.hxx"
#include "PersistentObjectFactory.hxx"


namespace OpenTURNS {

  namespace Base {

    namespace Type {
      /* These methods are implemented here for the needs of Cache */
      /* We should be careful because they may interfere with other definitions placed elsewhere */
      TEMPLATE_CLASSNAMEINIT(PersistentCollection<UnsignedLong>);
      //static Common::Factory<PersistentCollection<UnsignedLong> > RegisteredFactory_alt1("PersistentCollection<UnsignedLong>");
      TEMPLATE_CLASSNAMEINIT(PersistentCollection<PersistentCollection<NumericalScalar> >);
      //static Common::Factory<PersistentCollection<PersistentCollection<NumericalScalar> > > RegisteredFactory_alt2("PersistentCollection<PersistentCollection<NumericalScalar> >");

      typedef Cache<PersistentCollection<NumericalScalar>, PersistentCollection<NumericalScalar> > FunctionCacheBaseType;
      TEMPLATE_CLASSNAMEINIT(FunctionCacheBaseType );

    } // Type

    namespace Func {
      CLASSNAMEINIT(FunctionCache);
      static Common::Factory<FunctionCache> RegisteredFactory("FunctionCache");

      typedef Base::Type::PersistentCollection<NumericalScalar>                          SimplePoint;
      typedef std::map<SimplePoint,std::pair<SimplePoint,UnsignedLong> >::const_iterator const_iterator;
      typedef std::map<SimplePoint,std::pair<SimplePoint,UnsignedLong> >::iterator       iterator;
      typedef SimplePoint::const_iterator                                                const_internalIterator;
      typedef Base::Type::NumericalPoint                                                 NumericalPoint;
      typedef Base::Common::InvalidDimensionException                                    InvalidDimensionException;


      /* Default constructor */
      FunctionCache::FunctionCache()
        : Base::Type::Cache<KeyType, ValueType >()
      {
        //nothing to do
      }

      /* parameter constructor */
      FunctionCache::FunctionCache(const UnsignedLong size)
        : Type::Cache<KeyType, ValueType>(size)
      {
        //nothing to do
      }

      /* String converter */
      String FunctionCache::__repr__() const
      {
        OSS oss;
        oss << "class=" << FunctionCache::GetClassName()
            << " inherited from "
            << Base::Type::Cache<KeyType, ValueType >::__repr__();
        return oss;
      }

      String FunctionCache::__str__() const
      {
        return __repr__();
      }

      /* export the cache to NumericalSample */
      FunctionCache::NumericalSample FunctionCache::getInputNumericalSample() const
      {
        NumericalSample result;
        if (points_.size() > 0 )
          {
            result = NumericalSample( 0, points_.begin()->first.getSize() );
            for ( const_iterator it = points_.begin(); it != points_.end(); ++ it )
              {
                result.add( it->first );
              }
          }
        return result;
      }

      /* export the cache to NumericalSample */
      FunctionCache::NumericalSample FunctionCache::getOutputNumericalSample() const
      {
        NumericalSample result;
        if ( points_.size() > 0 )
          {
            result = NumericalSample( 0, points_.begin()->second.first.getSize() );
            for ( const_iterator it = points_.begin(); it != points_.end() ; ++ it )
              {
                result.add( it->second.first );
              }
          }
        return result;
      }


      /* Import cache from a NumericalSample */
      void FunctionCache::importFromNumericalSample(const NumericalSample & input,
                                                    const NumericalSample & output)
      {
        if ( input.getSize() != output.getSize() )
          throw InvalidDimensionException(HERE) << "input and output sample sizes do not match (in="<<input.getSize()<<" out="<<output.getSize()<<")";

        const UnsignedLong size( input.getSize() );
        for ( UnsignedLong i = 0; i < size; ++ i )
          {
            add( input[i], output[i] );
          }
      }

    }
  }
}
