//                                               -*- C++ -*-
/**
 *  @file  ConstantNumericalMathGradientImplementation.cxx
 *  @brief Class for a constant numerical math gradient implementation
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-04-11 12:32:27 +0200 (Mon, 11 Apr 2011) $
 *  Id:      $Id: ConstantNumericalMathGradientImplementation.cxx 1866 2011-04-11 10:32:27Z schueller $
 */
#include "ConstantNumericalMathGradientImplementation.hxx"
#include "PersistentObjectFactory.hxx"
#include "Os.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {

      CLASSNAMEINIT(ConstantNumericalMathGradientImplementation);

      static Common::Factory<ConstantNumericalMathGradientImplementation> RegisteredFactory("ConstantNumericalMathGradientImplementation");

      typedef Common::Os Os;

      /* Default constructor */
      ConstantNumericalMathGradientImplementation::ConstantNumericalMathGradientImplementation()
        : NumericalMathGradientImplementation()
      {
        // Nothing to do
      }

      /* Parameter constructor */
      ConstantNumericalMathGradientImplementation::ConstantNumericalMathGradientImplementation(const Matrix & constant)
        /* throw(InvalidDimensionException) */
        : NumericalMathGradientImplementation(),
          constant_(constant)
      {
        /* Check if the dimensions of the constant term is compatible with the linear term */
      }

      /* Virtual constructor */
      ConstantNumericalMathGradientImplementation * ConstantNumericalMathGradientImplementation::clone() const
      {
        return new ConstantNumericalMathGradientImplementation(*this);
      }

      /* Comparison operator */
      Bool ConstantNumericalMathGradientImplementation::operator ==(const ConstantNumericalMathGradientImplementation & other) const
      {
        return (constant_ == other.constant_);
      }

      /* String converter */
      String ConstantNumericalMathGradientImplementation::__repr__() const {
        OSS oss;
        oss << "class=" << ConstantNumericalMathGradientImplementation::GetClassName()
            << " name=" << getName()
            << " constant=" << constant_;
        return oss;
      }

      /* String converter */
      String ConstantNumericalMathGradientImplementation::__str__(const String & offset) const
      {
        OSS oss(false);
        const UnsignedLong inputDimension(getInputDimension());
        Description description(getDescription());
        if (description.getSize() == 0) for (UnsignedLong i = 0; i < inputDimension; ++i) description.add(OSS() << "x" << i);
        oss << offset << getName() << ":" << Os::GetEndOfLine();
        for (UnsignedLong i = 0; i < inputDimension; ++i)
          {
            if (i == 0) oss << offset << "  (";
            oss << description[i];
            if (i == inputDimension - 1) oss << ")";
            else oss << ", ";
          }
        oss << " ->" << Os::GetEndOfLine();
        oss << constant_.__str__(offset + "  ");
        return oss;
      }

      /* Accessor for the constant term */
      ConstantNumericalMathGradientImplementation::Matrix ConstantNumericalMathGradientImplementation::getConstant() const
      {
        return constant_;
      }

      /* Here is the interface that all derived class must implement */

      /* Operator () */
      ConstantNumericalMathGradientImplementation::Matrix ConstantNumericalMathGradientImplementation::gradient(const NumericalPoint & inP) const
      /* throw(InvalidArgumentException, InternalException) */
      {
        if (inP.getDimension() != constant_.getNbRows()) throw InvalidArgumentException(HERE) << "Invalid input dimension";
        ++callsNumber_;
        return constant_;
      }

      /* Accessor for input point dimension */
      UnsignedLong ConstantNumericalMathGradientImplementation::getInputDimension() const
      /* throw(InternalException) */
      {
        return constant_.getNbRows();
      }

      /* Accessor for output point dimension */
      UnsignedLong ConstantNumericalMathGradientImplementation::getOutputDimension() const
      /* throw(InternalException) */
      {
        return constant_.getNbColumns();
      }

      /* Method save() stores the object through the StorageManager */
      void ConstantNumericalMathGradientImplementation::save(StorageManager::Advocate & adv) const
      {
        NumericalMathGradientImplementation::save(adv);
        adv.saveAttribute( "constant_", constant_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void ConstantNumericalMathGradientImplementation::load(StorageManager::Advocate & adv)
      {
        NumericalMathGradientImplementation::load(adv);
        adv.loadAttribute( "constant_", constant_ );
      }

    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
