//                                               -*- C++ -*-
/**
 *  @file  ComposedNumericalMathGradientImplementation.cxx
 *  @brief Abstract top-level class for all distributions
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: souchaud $
 *  @date:   $LastChangedDate: 2011-07-01 10:34:36 +0200 (Fri, 01 Jul 2011) $
 *  Id:      $Id: ComposedNumericalMathGradientImplementation.cxx 1981 2011-07-01 08:34:36Z souchaud $
 */
#include "ComposedNumericalMathGradientImplementation.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {

      CLASSNAMEINIT(ComposedNumericalMathGradientImplementation);

      static Common::Factory<ComposedNumericalMathGradientImplementation> RegisteredFactory("ComposedNumericalMathGradientImplementation");

      /* Default constructor */
      ComposedNumericalMathGradientImplementation::ComposedNumericalMathGradientImplementation(const GradientImplementation & p_leftGradient,
                                                                                               const EvaluationImplementation & p_rightFunction,
                                                                                               const GradientImplementation & p_rightGradient) /* throw(InvalidArgumentException) */
        : NumericalMathGradientImplementation(),
          p_leftGradient_(p_leftGradient),
          p_rightFunction_(p_rightFunction),
          p_rightGradient_(p_rightGradient)
      {
        // We check that the dimensions of the input parameters are compatible
        // First, check the compatibility between the right function and the right gradient
        if ((p_rightFunction->getInputDimension() != p_rightGradient->getInputDimension()) || (p_rightFunction->getOutputDimension() != p_rightGradient->getOutputDimension())) throw InvalidArgumentException(HERE) << "Error: the right function and the right gradient have incompatible input or output dimensions.";
        // Second, check the left gradient and the right function
        if (p_leftGradient->getInputDimension() != p_rightFunction->getOutputDimension()) throw InvalidArgumentException(HERE) << "Error: the left gradient and the right function have incompatible input or output dimensions.";
      }

      /* Virtual constructor */
      ComposedNumericalMathGradientImplementation * ComposedNumericalMathGradientImplementation::clone() const
      {
        return new ComposedNumericalMathGradientImplementation(*this);
      }

      /* Comparison operator */
      Bool ComposedNumericalMathGradientImplementation::operator ==(const ComposedNumericalMathGradientImplementation & other) const
      {
        return true;
      }

      /* String converter */
      String ComposedNumericalMathGradientImplementation::__repr__() const {
        OSS oss;
        oss << "class=" << ComposedNumericalMathGradientImplementation::GetClassName()
            << " name=" << getName()
            << " leftGradient=" << p_leftGradient_->__repr__()
            << " rightFunction=" << p_rightFunction_->__repr__()
            << " rightGradient=" << p_rightGradient_->__repr__();
        return oss;
      }

      /* Method gradient() returns the Jacobian transposed matrix of the function at point
       * f goes from Rn to Rp and g from Rq to Rp, so h = f o g goes from Rq to Rn. Its
       * jacobian goes from Rq to L(Rq, Rn), thus its gradient goes from Rq to L(Rn, Rq):
       * it is an n by q matrix                                                            */
      ComposedNumericalMathGradientImplementation::Matrix ComposedNumericalMathGradientImplementation::gradient(const NumericalPoint & inP) const
      /* throw(InvalidArgumentException,InternalException) */
      {
        const UnsignedLong inputDimension(getInputDimension());
        if (inP.getDimension() != inputDimension) throw InvalidArgumentException(HERE) << "Error: the given point has an invalid dimension. Expect a dimension " << inputDimension << ", got " << inP.getDimension();
        ++callsNumber_;
        return  p_rightGradient_->gradient(inP) * p_leftGradient_->gradient(p_rightFunction_->operator()(inP));
      }

      /* Accessor for input point dimension */
      UnsignedLong ComposedNumericalMathGradientImplementation::getInputDimension() const
      /* throw(InternalException) */
      {
        return p_rightGradient_->getInputDimension();
      }

      /* Accessor for output point dimension */
      UnsignedLong ComposedNumericalMathGradientImplementation::getOutputDimension() const
      /* throw(InternalException) */
      {
        return p_leftGradient_->getOutputDimension();

      }

      /* Method save() stores the object through the StorageManager */
      void ComposedNumericalMathGradientImplementation::save(StorageManager::Advocate & adv) const
      {
        NumericalMathGradientImplementation::save(adv);
        adv.saveAttribute( "leftGradient_", *p_leftGradient_ );
        adv.saveAttribute( "rightFunction_", *p_rightFunction_ );
        adv.saveAttribute( "rightGradient_", *p_rightGradient_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void ComposedNumericalMathGradientImplementation::load(StorageManager::Advocate & adv)
      {
        Common::TypedInterfaceObject<NumericalMathEvaluationImplementation> evaluationValue;
        Common::TypedInterfaceObject<NumericalMathGradientImplementation> gradientValue;
        NumericalMathGradientImplementation::load(adv);
        adv.loadAttribute( "leftGradient_", gradientValue );
        p_leftGradient_ = gradientValue.getImplementation();
        adv.loadAttribute( "rightFunction_", evaluationValue );
        p_rightFunction_ = evaluationValue.getImplementation();
        adv.loadAttribute( "rightGradient_", gradientValue );
        p_rightGradient_ = gradientValue.getImplementation();
      }

    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
