//                                               -*- C++ -*-
/**
 *  @file  t_IsoProbabilisticTransformation_EllipticalDistribution.cxx
 *  @brief The test file of class IsoProbabilisticTransformation for standard methods
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: t_IsoProbabilisticTransformation_EllipticalDistribution.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include <iostream>
#include <sstream>
#include <cmath>
#include "OT.hxx"
#include "OTtestcode.hxx"
#include "OStream.hxx"
#include "Normal.hxx"
#include "NumericalPoint.hxx"
#include "NumericalSample.hxx"
#include "CorrelationMatrix.hxx"
#include "Distribution.hxx"
#include "CenteredFiniteDifferenceGradient.hxx"
#include "CenteredFiniteDifferenceHessian.hxx"

using namespace OT;
using namespace OT::Test;
using namespace OT::Base::Common;
using namespace OT::Base::Common;
using namespace OT::Base::Type;
using namespace OT::Base::Stat;
using namespace OT::Base::Diff;
using namespace OT::Uncertainty::Model;
using namespace OT::Uncertainty::Distribution;

typedef Distribution::IsoProbabilisticTransformation IsoProbabilisticTransformation;
typedef Distribution::InverseIsoProbabilisticTransformation InverseIsoProbabilisticTransformation;

NumericalPoint clean(NumericalPoint in)
{
  UnsignedLong dim(in.getDimension());
  for(UnsignedLong i = 0; i < dim; i++)
    if (fabs(in[i]) < 1.e-10) in[i] = 0.0;
  return in;
}

Matrix clean(Matrix in)
{
  UnsignedLong rowDim(in.getNbRows());
  UnsignedLong colDim(in.getNbColumns());
  for(UnsignedLong i = 0; i < rowDim; i++)
    for(UnsignedLong j = 0; j < colDim; j++)
      if (fabs(in(i, j)) < 1.e-6) in(i, j) = 0.0;
  return in;
}

SymmetricTensor clean(SymmetricTensor in)
{
  UnsignedLong rowDim(in.getNbRows());
  UnsignedLong colDim(in.getNbColumns());
  UnsignedLong sheetDim(in.getNbSheets());
  for(UnsignedLong i = 0; i < rowDim; i++)
    for(UnsignedLong j = 0; j < colDim; j++)
      for (UnsignedLong k = 0; k < sheetDim; k++)
        if (fabs(in(i, j, k)) < 1.e-6) in(i, j, k) = 0.0;
  return in;
}

int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  OStream fullprint(std::cout);
  setRandomGenerator();

  try {
    // Instanciate one distribution object
    UnsignedLong dim(3);
    NumericalPoint meanPoint(dim, 1.0);
    meanPoint[0] = 0.5;
    meanPoint[1] = -0.5;
    NumericalPoint sigma(dim, 1.0);
    sigma[0] = 2.0;
    sigma[1] = 3.0;
    CorrelationMatrix R(dim);
    for (UnsignedLong i = 1; i < dim; i++)
      {
	R(i, i - 1) = 0.5;
      }

    Normal distribution(meanPoint, sigma, R);

    // Test for sampling
    UnsignedLong size = 10000;
    NumericalSample sample = distribution.getNumericalSample( size );
    fullprint << "sample first=" << sample[0] << " last=" << sample[size - 1] << std::endl;
    fullprint << "sample mean=" << sample.computeMean() << std::endl;
    fullprint << "sample covariance=" << sample.computeCovariance() << std::endl;

    IsoProbabilisticTransformation transform(distribution.getIsoProbabilisticTransformation());
    fullprint << "isoprobabilistic transformation=" << transform << std::endl;
    NumericalSample transformedSample(transform(sample));
    fullprint << "transformed sample first=" << transformedSample[0] << " last=" << transformedSample[size - 1] << std::endl;
    fullprint << "transformed sample mean=" << transformedSample.computeMean() << std::endl;
    fullprint << "transformed sample covariance=" << transformedSample.computeCovariance() << std::endl;

    // Test for evaluation
    InverseIsoProbabilisticTransformation inverseTransform(distribution.getInverseIsoProbabilisticTransformation());
    fullprint << "inverse isoprobabilistic transformation=" << inverseTransform << std::endl;
    NumericalSample transformedBackSample(inverseTransform(transformedSample));
    fullprint << "transformed back sample first=" << transformedBackSample[0] << " last=" << transformedBackSample[size - 1] << std::endl;
    fullprint << "transformed back sample mean=" << transformedBackSample.computeMean() << std::endl;
    fullprint << "transformed back sample covariance=" << transformedBackSample.computeCovariance() << std::endl;
    NumericalPoint point(dim, 1.0);
    fullprint << "point=" << point << std::endl;
    NumericalPoint transformedPoint(transform(point));
    fullprint << "transform value at point        =" << transformedPoint << std::endl;
    fullprint << "transform gradient at point     =" << clean(transform.gradient(point)) << std::endl;
    fullprint << "transform gradient at point (FD)=" << clean(CenteredFiniteDifferenceGradient(1.0e-5, transform.getEvaluationImplementation()).gradient(point)) << std::endl;
    fullprint << "transform hessian at point      =" << clean(transform.hessian(point)) << std::endl;
    fullprint << "transform hessian at point (FD) =" << clean(CenteredFiniteDifferenceHessian(1.0e-4, transform.getEvaluationImplementation()).hessian(point)) << std::endl;
    fullprint << "inverse transform value at transformed point        =" << inverseTransform(transformedPoint) << std::endl;
    fullprint << "inverse transform gradient at transformed point (FD)=" << clean(inverseTransform.gradient(transformedPoint)) << std::endl;
    fullprint << "inverse transform gradient at transformed point     =" << clean(CenteredFiniteDifferenceGradient(1.0e-5, inverseTransform.getEvaluationImplementation()).gradient(transformedPoint)) << std::endl;
    fullprint << "inverse transform hessian at transformed point      =" << clean(inverseTransform.hessian(transformedPoint)) << std::endl;
    fullprint << "inverse transform hessian at transformed point (FD) =" << clean(CenteredFiniteDifferenceHessian(1.0e-4, inverseTransform.getEvaluationImplementation()).hessian(transformedPoint)) << std::endl;

    // Test for parameters
    fullprint << "parameters gradient at point=" << clean(transform.parametersGradient(point)) << std::endl;
  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
