//                                               -*- C++ -*-
/**
 * @file  SensitivityAnalysis.cxx
 * @brief SensitivityAnalysis implements the sensitivity analysis methods based on Sobol coefficients
 *
 * (C) Copyright 2005-2010 EADS
 *
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 *
 *
 * @author $LastChangedBy:  $
 * @date   $LastChangedDate:  $
 */
#include <cstdlib>
#include <fstream>

#include "SensitivityAnalysis.hxx"
#include "Path.hxx"
#include "ResourceMap.hxx"
#include "Log.hxx"
#include "Exception.hxx"
#include "Os.hxx"
#include "OTconfig.hxx"


namespace OpenTURNS
{
  namespace Base
  {
    namespace Stat
    {

      typedef Common::Path                      Path;
      typedef Common::ResourceMap               ResourceMap;
      typedef Common::Log                       Log;
      typedef Common::Os                        Os;
      typedef Common::InvalidArgumentException  InvalidArgumentException;
      typedef Common::InvalidDimensionException InvalidDimensionException;
      typedef Common::InternalException         InternalException;
      typedef Common::NotYetImplementedException NotYetImplementedException;

      const UnsignedLong SensitivityAnalysis::DefaultSampleSize = 1000;//strtoul(ResourceMap::GetInstance().get("SensitivityAnalysis-DefaultSampleSize").c_str(), NULL, 0);

      /* Constructor with input samples and model */
      SensitivityAnalysis::SensitivityAnalysis(const NumericalSample & inputSample1,
                                               const NumericalSample & inputSample2,
                                               const NumericalMathFunction & model):
        inputSample1_(inputSample1),
        inputSample2_(inputSample2),
        model_(model),
        sensitivityIndicesSobol_(0,0),
        isAlreadyComputed_(false)
      {
        isComputationPossible();
      }



      /* Compute all the Sobol indices */
      void SensitivityAnalysis::computeSobolIndices() const
      {
        const String dataOTFileName1(Path::BuildTemporaryFileName("OTData1.csv.XXXXXX"));
        const String dataOTFileName2(Path::BuildTemporaryFileName("OTData2.csv.XXXXXX"));
        const String dataRInputSampleFileName(Path::BuildTemporaryFileName("RInputSample.csv.XXXXXX"));
        const String dataOTOutputSampleFileName(Path::BuildTemporaryFileName("OTOutputSample.csv.XXXXXX"));
        const String dataRResultsFileName(Path::BuildTemporaryFileName("RResults.csv.XXXXXX"));

        const String commandFileName1(Path::BuildTemporaryFileName("RCmd1.R.XXXXXX"));
        std::ofstream cmdFile1(commandFileName1.c_str(), std::ios::out);

        inputSample1_.exportToCSVFile(dataOTFileName1);
        inputSample2_.exportToCSVFile(dataOTFileName2);

        // Fill-in the command file 1
        cmdFile1 << "library(rotRPackage)" << std::endl;
        cmdFile1 << "require(sensitivity)" << std::endl;
        cmdFile1 << "options(digits=17)" << std::endl;
        cmdFile1 << "options(warn=-1)" << std::endl;
        cmdFile1 << "sample1 <- data.matrix(read.table(\"" << dataOTFileName1 << "\", sep=\"" << ResourceMap::GetInstance().get("csv-file-separator") << "\"))" << std::endl;
        cmdFile1 << "sample2 <- data.matrix(read.table(\"" << dataOTFileName2 << "\", sep=\"" << ResourceMap::GetInstance().get("csv-file-separator") << "\"))" << std::endl;
        cmdFile1 << "sasobolsaltelli <- sobol.saltelli02(x1=sample1, x2=sample2)" << std::endl;
        cmdFile1 << "write.table(sasobolsaltelli$x, \"" << dataRInputSampleFileName << "\", sep=\"" << ResourceMap::GetInstance().get("csv-file-separator") << "\",col.names=FALSE,row.names=FALSE)" << std::endl;
        cmdFile1.close();

        OSS systemCommand1;
#ifdef R_EXECUTABLE_PATH
        systemCommand1 << ResourceMap::GetInstance().get("R-executable-command") << " --no-save --silent < \"" << commandFileName1 << "\"" << Os::GetDeleteCommandOutput();
#else
        throw NotYetImplementedException(HERE) << "SensitivityAnalysis::computeSobolIndices() need R";
#endif
        int returnCode1(system(String(systemCommand1).c_str()));
        if (returnCode1 != 0) throw InternalException(HERE) << "Error: unable to execute the system command " << String(systemCommand1) << " returned code is " << returnCode1;

        const NumericalSample inputSample(NumericalSample::ImportFromCSVFile(dataRInputSampleFileName));

        const NumericalSample outputSample(model_(inputSample));
        outputSample.exportToCSVFile(dataOTOutputSampleFileName);

        const String commandFileName2(Path::BuildTemporaryFileName("RCmd2.R.XXXXXX"));
        std::ofstream cmdFile2(commandFileName2.c_str(), std::ios::out);

        // Fill-in the command file 2
        cmdFile2 << "library(rotRPackage)" << std::endl;
        cmdFile2 << "require(sensitivity)" << std::endl;
        cmdFile2 << "options(digits=17)" << std::endl;
        cmdFile2 << "options(warn=-1)" << std::endl;
        cmdFile2 << "sample1 <- data.matrix(read.table(\"" << dataOTFileName1 << "\", sep=\"" << ResourceMap::GetInstance().get("csv-file-separator") << "\"))" << std::endl;
        cmdFile2 << "sample2 <- data.matrix(read.table(\"" << dataOTFileName2 << "\", sep=\"" << ResourceMap::GetInstance().get("csv-file-separator") << "\"))" << std::endl;
        cmdFile2 << "sasobolsaltelli <- sobol.saltelli02(x1=sample1, x2=sample2)" << std::endl;
        cmdFile2 << "outputSample <- data.matrix(read.table(\"" << dataOTOutputSampleFileName << "\", sep=\"" << ResourceMap::GetInstance().get("csv-file-separator") << "\"))" << std::endl;
        cmdFile2 << "sasobolsaltelli <- tell.sobol.saltelli02(sa=sasobolsaltelli, y=outputSample)" << std::endl;
        cmdFile2 << "write.table(rbind(t(sasobolsaltelli$S1),t(sasobolsaltelli$St)),\"" << dataRResultsFileName << "\", sep=\"" << ResourceMap::GetInstance().get("csv-file-separator") << "\",col.names=FALSE,row.names=FALSE)" << std::endl;
        cmdFile2.close();

        OSS systemCommand2;
#ifdef R_EXECUTABLE_PATH
        systemCommand2 << ResourceMap::GetInstance().get("R-executable-command") << " --no-save --silent < \"" << commandFileName2 << "\"" << Os::GetDeleteCommandOutput();
#else
        throw NotYetImplementedException(HERE) << "SensitivityAnalysis::computeSobolIndices() need R";
#endif
        int returnCode2(system(String(systemCommand2).c_str()));
        if (returnCode2 != 0) throw InternalException(HERE) << "Error: unable to execute the system command " << String(systemCommand2) << " returned code is " << returnCode2;

        sensitivityIndicesSobol_ = NumericalSample::ImportFromCSVFile(dataRResultsFileName);

        isAlreadyComputed_ = true;

        // Clean-up everything
        if (remove(dataOTFileName1.c_str()) == -1) Log::Warn(OSS() << "Warning: cannot remove file " << dataOTFileName1);
        if (remove(dataOTFileName2.c_str()) == -1) Log::Warn(OSS() << "Warning: cannot remove file " << dataOTFileName2);
        if (remove(dataRInputSampleFileName.c_str()) == -1) Log::Warn(OSS() << "Warning: cannot remove file " << dataRInputSampleFileName);
        if (remove(dataOTOutputSampleFileName.c_str()) == -1) Log::Warn(OSS() << "Warning: cannot remove file " << dataOTOutputSampleFileName);
        if (remove(dataRResultsFileName.c_str()) == -1) Log::Warn(OSS() << "Warning: cannot remove file " << dataRResultsFileName);
        if (remove(commandFileName1.c_str()) == -1) Log::Warn(OSS() << "Warning: cannot remove file " << commandFileName1);
        if (remove(commandFileName2.c_str()) == -1) Log::Warn(OSS() << "Warning: cannot remove file " << commandFileName2);

      }

      /* First Order indices accessor */
      SensitivityAnalysis::NumericalPoint SensitivityAnalysis::getFirstOrderIndices() const
      {
        if (!isAlreadyComputed_) computeSobolIndices();
        return sensitivityIndicesSobol_[0];
      }

      /* Total Order indices accessor */
      SensitivityAnalysis::NumericalPoint SensitivityAnalysis::getTotalOrderIndices() const
      {
        if (!isAlreadyComputed_) computeSobolIndices();
        return sensitivityIndicesSobol_[1];
      }

      /* Check if computation of the Sobol indices is possible */
      void SensitivityAnalysis::isComputationPossible()
      {
        if (inputSample1_.getDimension() != inputSample2_.getDimension()) throw InvalidDimensionException(HERE) << "Error: input samples must have the same dimension";
        if (inputSample1_.getSize() != inputSample2_.getSize()) throw InvalidArgumentException(HERE) << "Error: input samples must have the same size";
        if (inputSample1_.getDimension() != model_.getInputDimension()) throw InvalidDimensionException(HERE) << "Error: input samples must have the same dimension as the model input dimension";
        if (model_.getOutputDimension() != 1) throw InvalidDimensionException(HERE) << "Error: model output must be 1D";
      }


    } // namespace Stat
  } // namespace Base
} // namespace OpenTURNS


