//                                               -*- C++ -*-
/**
 *  @file  WrapperObject.cxx
 *  @brief A WrapperObject binds a dynamic library symbol to itself
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: WrapperObject.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include <sstream>               // for std::istringstream
#include "WrapperObject.hxx"
#include "LibraryLoader.hxx"
#include "OSS.hxx"
#include "Log.hxx"
#include "ResourceMap.hxx"
#include "WrapperInterface.h"
#include "WrapperCommon.h"
#include "WrapperCommonFunctions.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Func
    {

      //const UnsignedLong WrapperObject::NBCPUS_ = Common::ResourceMap::GetInstance().getAsUnsignedLong("parallel-threads");


      using Common::DynamicLibraryException;
      using Common::Log;
      
      CLASSNAMEINIT(WrapperObject);

      // Sub-classes definitions
      WrapperObject::ObjectPrefix::ObjectPrefix()
	: std::vector<String>(WrapperObject::ObjectPrefixSize)
      {
	ObjectPrefix & ref_me = *this;
	ref_me[WrapperObject::FUNCTION]        = "func_";
	ref_me[WrapperObject::GRADIENT]        = "grad_";
	ref_me[WrapperObject::HESSIAN]         = "hess_";	
      }

      WrapperObject::FunctionPrefix::FunctionPrefix()
	: std::vector<String>(WrapperObject::FunctionPrefixSize)
      {
	FunctionPrefix & ref_me = *this;
	ref_me[WrapperObject::STATE_CREATION]   = "createState_";
	ref_me[WrapperObject::STATE_DELETION]   = "deleteState_";
	ref_me[WrapperObject::INFORMATION]      = "getInfo_";
	ref_me[WrapperObject::INITIALIZATION]   = "init_";
	ref_me[WrapperObject::EXECUTION]        = "exec_";
	ref_me[WrapperObject::EXECUTION_SAMPLE] = "exec_sample_";
	ref_me[WrapperObject::FINALIZATION]     = "finalize_";
      }

      // Initialization of class members
      const WrapperObject::ObjectPrefix   WrapperObject::ObjectPrefix_;
      const WrapperObject::FunctionPrefix WrapperObject::FunctionPrefix_;








      WrapperObject::ExchangedData::ExchangedData(const WrapperData & data)
	: p_exchangedData_(0)
      {
	// Create and initialize the structure of data exchanged with the wrapper
	p_exchangedData_.reset( new WrapperExchangedData );

	p_exchangedData_->fileList_     = data.getNewFileListForCInterface();
	p_exchangedData_->variableList_ = data.getNewVariableListForCInterface();
	p_exchangedData_->parameters_   = data.getNewParametersForCInterface();
	p_exchangedData_->platform_     = new struct PlatformConfiguration;
	p_exchangedData_->framework_    = data.getNewFrameworkForCInterface();

	String tempDir = Common::ResourceMap::GetInstance().get("temporary-directory");
	p_exchangedData_->platform_->generalTemporaryDirectory_ = new char[tempDir.size()+1];
	strncpy(p_exchangedData_->platform_->generalTemporaryDirectory_, tempDir.c_str(), tempDir.size()+1);

	String realRegexpShortcut = Common::ResourceMap::GetInstance().get("real-regexp-shortcut");
	p_exchangedData_->platform_->realRegexpShortcut_ = new char[realRegexpShortcut.size()+1];
	strncpy(p_exchangedData_->platform_->realRegexpShortcut_, realRegexpShortcut.c_str(), realRegexpShortcut.size()+1);

	String integerRegexpShortcut = Common::ResourceMap::GetInstance().get("integer-regexp-shortcut");
	p_exchangedData_->platform_->integerRegexpShortcut_ = new char[integerRegexpShortcut.size()+1];
	strncpy(p_exchangedData_->platform_->integerRegexpShortcut_, integerRegexpShortcut.c_str(), integerRegexpShortcut.size()+1);

	String separatorRegexpShortcut = Common::ResourceMap::GetInstance().get("separator-regexp-shortcut");
	p_exchangedData_->platform_->separatorRegexpShortcut_ = new char[separatorRegexpShortcut.size()+1];
	strncpy(p_exchangedData_->platform_->separatorRegexpShortcut_, separatorRegexpShortcut.c_str(), separatorRegexpShortcut.size()+1);

	p_exchangedData_->platform_->nbCpus_ = Common::ResourceMap::GetInstance().getAsUnsignedLong("parallel-threads");
      }


      WrapperObject::ExchangedData::~ExchangedData()
      {
	// At last we clean up everything that was allocated
	// ... files
	WrapperData::FreeFileListForCInterface(p_exchangedData_->fileList_);
	
	// ... variables
	WrapperData::FreeVariableListForCInterface(p_exchangedData_->variableList_);
	
	// ... parameters
	WrapperData::FreeParametersForCInterface(p_exchangedData_->parameters_);
	
	// ... platform
	delete [] p_exchangedData_->platform_->generalTemporaryDirectory_;
	delete [] p_exchangedData_->platform_->realRegexpShortcut_;
	delete [] p_exchangedData_->platform_->integerRegexpShortcut_;
	delete [] p_exchangedData_->platform_->separatorRegexpShortcut_;
	delete p_exchangedData_->platform_;
	
	// ... framework
	WrapperData::FreeFrameworkForCInterface(p_exchangedData_->framework_);
      }

      const struct WrapperExchangedData * WrapperObject::ExchangedData::get() const
      {
	return p_exchangedData_.get();
      }

      struct WrapperExchangedData*  WrapperObject::ExchangedData::get()
      {
	return p_exchangedData_.get();
      }







      WrapperObject::Error::Error()
	: p_error_(new WrapperError)
      {
	struct WrapperError * p_wrapperError = static_cast<struct WrapperError *>( p_error_.get() );
	pthread_mutex_init( &(p_wrapperError->mutex), NULL );
	p_wrapperError->length  = 0;
	p_wrapperError->message = NULL;
      }

      WrapperObject::Error::~Error()
      {
	struct WrapperError * p_wrapperError = static_cast<struct WrapperError *>( p_error_.get() );
	clearWrapperError( p_wrapperError );
	pthread_mutex_destroy( &(p_wrapperError->mutex) );
      }

      const struct WrapperError * WrapperObject::Error::get() const
      {
	return p_error_.get();
      }

      struct WrapperError * WrapperObject::Error::get()
      {
	return p_error_.get();
      }





      static enum WrapperErrorCode optionalWrapperFunction(...)
      {
	const char FUNCTIONNAME[] = "optionalWrapperFunction";
	printEntrance(FUNCTIONNAME);
	printExit(FUNCTIONNAME);

	return WRAPPER_OK;
      }

      static enum WrapperErrorCode defaultWrapperGetInfoFunction(void * p_state,
								 struct WrapperInformation * p_info,
								 const struct WrapperExchangedData * p_exchangedData,
								 void * p_error)
      {
	const char FUNCTIONNAME[] = "defaultWrapperGetInfoFunction";
	printEntrance(FUNCTIONNAME);
	printState(FUNCTIONNAME, p_state);

	p_info->inSize_  = getNumberOfVariables(p_exchangedData, WRAPPER_IN);
	p_info->outSize_ = getNumberOfVariables(p_exchangedData, WRAPPER_OUT);

	printWrapperInformation(FUNCTIONNAME, p_info);
	printExit(FUNCTIONNAME);

	return WRAPPER_OK;
      }

      static enum WrapperErrorCode defaultWrapperCreateStateFunction(void ** p_p_state,
								     const struct WrapperExchangedData * p_exchangedData,
								     void * p_error)
      {
	const char FUNCTIONNAME[] = "defaultWrapperCreateStateFunction";
	printEntrance(FUNCTIONNAME);
	if (!p_p_state) {
	  setWrapperError( p_error, "No storage available for internal state" );
	  return WRAPPER_CANNOT_CREATE_STATE;
	}
	printState(FUNCTIONNAME, *p_p_state);

	if (!p_exchangedData) {
	  setWrapperError( p_error, "No exchanged data sent by the platform" );
	  return WRAPPER_INTERNAL_ERROR;
	}
	printWrapperExchangedData(FUNCTIONNAME, p_exchangedData);

	printExit(FUNCTIONNAME);

	return WRAPPER_OK;
      }

      static enum WrapperErrorCode defaultWrapperDeleteStateFunction(void * p_state,
								     const struct WrapperExchangedData * p_exchangedData,
								     void * p_error)
      {
	const char FUNCTIONNAME[] = "defaultWrapperDeleteStateFunction";
	printEntrance(FUNCTIONNAME);

	printState(FUNCTIONNAME, p_state);
	printExit(FUNCTIONNAME);

	return WRAPPER_OK;
      }

      static enum WrapperErrorCode notDefinedFunction(...)
      {
	const char FUNCTIONNAME[] = "notDefinedFunction";
	printEntrance(FUNCTIONNAME);
	printExit(FUNCTIONNAME);
	return WRAPPER_NOT_IMPLEMENTED;
      }





      struct WrapperSymbols : public Common::Object
      {
	GetWrapperInformationFunctionPointer getInfoSymbol_;
	InitializationFunctionPointer initSymbol_;
	ExecutionFunctionPointer execSymbol_;
	ExecutionSampleFunctionPointer execSampleSymbol_;
	GradientFunctionPointer gradSymbol_;
	HessianFunctionPointer hessSymbol_;
	FinalizationFunctionPointer finalizeSymbol_;
	StateCreationFunctionPointer stateCreationSymbol_;
	StateDeletionFunctionPointer stateDeletionSymbol_;
	WrapperSymbols()
	  : getInfoSymbol_(0),
	    initSymbol_(0),
	    execSymbol_(0),
	    execSampleSymbol_(0),
	    gradSymbol_(0),
	    hessSymbol_(0),
	    finalizeSymbol_(0),
	    stateCreationSymbol_(0),
	    stateDeletionSymbol_(0)
	{}
	~WrapperSymbols() throw() {}
	CLASSNAME;
      };

      CLASSNAMEINIT(WrapperSymbols);





      /* Default constructor */
      WrapperObject::WrapperObject(const FileName & libraryPath,
				   const String & symbolName,
				   const WrapperData & data,
				   ObjectType o
				   )
      /* throw (WrapperInternalException) */
	: PersistentObject(symbolName),
	  handle_(LibraryLoader::GetInstance().load(libraryPath)),
	  data_(data),
	  exchangedData_(data),
	  type_(o),
	  error_(),
	  wrapperSymbols_(new WrapperSymbols),
	  wrapperInfo_(0)
      {
	if (symbolName.empty()) {
	  String name;
	  switch (o) {
	  case WrapperObject::FUNCTION: name="function"; break;
	  case WrapperObject::GRADIENT: name="gradient"; break;
	  case WrapperObject::HESSIAN:  name="hessian";  break;
	  default: name="unknown";
	  };
	  throw WrapperInternalException(HERE) << "No symbol name for " << name << " in library " << libraryPath;
	}

	const Bool optional = true;
	const LibrarySymbol optWrpFunc = REINTERPRET_CAST( LibrarySymbol, &optionalWrapperFunction           );
	const LibrarySymbol defWrpFunc = REINTERPRET_CAST( LibrarySymbol, &defaultWrapperGetInfoFunction     );
	const LibrarySymbol notDefFunc = REINTERPRET_CAST( LibrarySymbol, &notDefinedFunction                );
	const LibrarySymbol createFunc = REINTERPRET_CAST( LibrarySymbol, &defaultWrapperCreateStateFunction );
	const LibrarySymbol deleteFunc = REINTERPRET_CAST( LibrarySymbol, &defaultWrapperDeleteStateFunction );


	wrapperSymbols_->getInfoSymbol_       = REINTERPRET_CAST(GetWrapperInformationFunctionPointer,
						getSymbol(getFunctionName(INFORMATION), optional, defWrpFunc) );
	wrapperSymbols_->stateCreationSymbol_ = REINTERPRET_CAST(StateCreationFunctionPointer,
						getSymbol(getFunctionName(STATE_CREATION), optional, createFunc) );
	wrapperSymbols_->stateDeletionSymbol_ = REINTERPRET_CAST(StateDeletionFunctionPointer,
						getSymbol(getFunctionName(STATE_DELETION), optional, deleteFunc) );
	wrapperSymbols_->initSymbol_          = REINTERPRET_CAST(InitializationFunctionPointer,
						getSymbol(getFunctionName(INITIALIZATION), optional, optWrpFunc) );
	wrapperSymbols_->finalizeSymbol_      = REINTERPRET_CAST(FinalizationFunctionPointer,
						getSymbol(getFunctionName(FINALIZATION), optional, optWrpFunc) );
	wrapperSymbols_->execSymbol_          = REINTERPRET_CAST(ExecutionFunctionPointer,
						getSymbol(getFunctionName(EXECUTION))) ;
	wrapperSymbols_->gradSymbol_          = REINTERPRET_CAST(GradientFunctionPointer,
						getSymbol(getFunctionName(EXECUTION)) );
	wrapperSymbols_->hessSymbol_          = REINTERPRET_CAST(HessianFunctionPointer,
						getSymbol(getFunctionName(EXECUTION)) );
	wrapperSymbols_->execSampleSymbol_    = REINTERPRET_CAST(ExecutionSampleFunctionPointer,
						getSymbol(getFunctionName(EXECUTION_SAMPLE), optional, notDefFunc) );
      }

      /* Virtual constructor */
      WrapperObject * WrapperObject::clone() const
      {
	return new WrapperObject(*this);
      }


      /* Comparison operator */
      /* NOT IMPLEMENTED */

      /* String converter */
      String WrapperObject::__repr__() const
      {
	return OSS() << "class=" << WrapperObject::GetClassName()
		     << " name=" << getName()
		     << " handle=" << handle_
		     << " type_=" << type_;
      }



      /* Method getFunctionName returns the name of the symbol
       * in the shared library.
       */
      String WrapperObject::getFunctionName(FunctionType f) const
      {
	return String().append(ObjectPrefix_[type_]).append(FunctionPrefix_[f]).append(getName());
      }


      /* Method getInNumericalPointDimension returns the dimension of the in point */
      UnsignedLong WrapperObject::getInNumericalPointDimension(void * p_state) const
      /* throw (WrapperInternalException) */
      {
	if (wrapperInfo_.isNull() ) {
	  wrapperInfo_.reset(new WrapperInformation);
	  WrapperErrorCode returnCode = (*(wrapperSymbols_->getInfoSymbol_))( p_state, wrapperInfo_.get(), exchangedData_.get(), error_.get() );
	  if (returnCode != WRAPPER_OK) 
	    throw WrapperInternalException(HERE)
	      << "Wrapper function '" << getFunctionName(INFORMATION)
	      << "' returned error message: " << getErrorAsString(returnCode)
	      << ". Reason: " << getWrapperError( error_.get() );
	}
	return wrapperInfo_->inSize_;
      }

      /* Method getDescription returns the description of the input and output variables of the functions */
      WrapperObject::Description WrapperObject::getDescription() const
      /* throw (WrapperInternalException) */
      {
	Description description;
	WrapperData::VariableListType variableList(data_.getVariableList());
	for (UnsignedLong i = 0; i < variableList.getSize(); ++i) description.add(variableList[i].id_);
	return description;
      }


      /* Method getInNumericalPointDimension returns the dimension of the out point */
      UnsignedLong WrapperObject::getOutNumericalPointDimension(void * p_state) const
      /* throw (WrapperInternalException) */
      {
	if (wrapperInfo_.isNull() ) {
	  wrapperInfo_.reset(new WrapperInformation);
	  WrapperErrorCode returnCode = (*(wrapperSymbols_->getInfoSymbol_))( p_state, wrapperInfo_.get(), exchangedData_.get(), error_.get() );
	  if (returnCode != WRAPPER_OK) 
	    throw WrapperInternalException(HERE)
	      << "Wrapper function '" << getFunctionName(INFORMATION)
	      << "' returned error message: " << getErrorAsString(returnCode)
	      << ". Reason: " << getWrapperError( error_.get() );
	}
	return wrapperInfo_->outSize_;
      }


      /* Method initialize calls the initializationSymbol of the library */
      void WrapperObject::initialize(void * p_state) const
      /* throw (WrapperInternalException) */
      {
	// We initialize the wrapper
	WrapperErrorCode returnCode = (*(wrapperSymbols_->initSymbol_))( p_state, exchangedData_.get(), error_.get() );
	if (returnCode != WRAPPER_OK)
	  throw WrapperInternalException(HERE)
	    << "Wrapper function '" << getFunctionName(INITIALIZATION)
	    << "' returned error message: " << getErrorAsString(returnCode)
	    << ". Reason: " << getWrapperError( error_.get() );
      }


      // TODO : make this method simpler !!!
      /* Method execute actually realizes the desired treatment */
      /* TODO: MAKE THIS ROUTINE THREAD-SAFE */
      WrapperObject::NumericalPoint WrapperObject::execute(void * p_state,
							   const NumericalPoint & in) const 
      /* throw(InvalidArgumentException,WrapperInternalException) */
      {
	// Check the in argument
	if (in.getDimension() != getInNumericalPointDimension(p_state))
	  throw InvalidArgumentException(HERE)
	    << "Argument 'in' has incorrect size (" << in.getDimension()
	    << "). Expected = " << getInNumericalPointDimension(p_state);



	// We create a point structure to embed the in NumericalPoint passed as argument
	struct point inPoint;
	inPoint.size_ = in.getDimension();
	inPoint.data_ = const_cast<NumericalScalar *>(&in[0]);

	// We create a point structure to embed the returned out NumericalPoint
	struct point outPoint;
	outPoint.size_ = getOutNumericalPointDimension(p_state);
	NumericalPoint out(outPoint.size_);
	outPoint.data_ = const_cast<NumericalScalar *>(&out[0]);

#ifdef DEBUG
	{
	  OSS oss;
	  oss << "In struct point {size= " << inPoint.size_ << ", data=[";
	  const char * separator = "";
	  for(unsigned int i=0; i < inPoint.size_; i++, separator=",") oss << separator << inPoint.data_[i];
	  oss << "]}";
	  Log::Debug( oss );
	}
#endif

	// Then we call the wrapper execution function
	WrapperErrorCode returnCode = (*(wrapperSymbols_->execSymbol_))( p_state, & inPoint, & outPoint, exchangedData_.get(), error_.get() );
	if (returnCode != WRAPPER_OK)
	  throw WrapperInternalException(HERE)
	    << "Wrapper function '" << getFunctionName(EXECUTION)
	    << "' returned error message: " << getErrorAsString(returnCode)
	    << ". Reason: " << getWrapperError( error_.get() );
	

#ifdef DEBUG
	{
	  OSS oss;
	  oss << "Out struct point {size= " << outPoint.size_ << ", data=[";
	  const char * separator = "";
	  for(unsigned int i=0; i < outPoint.size_; i++, separator=",") oss << separator << outPoint.data_[i];
	  oss << "]}";
	  Log::Debug( oss );
	}
#endif
	return out;
      } // WrapperObject::execute(void * p_state, const NumericalPoint & in) const



      // TODO : make this method simpler !!!
      /* Method execute actually realizes the desired treatment */
      /* TODO: MAKE THIS ROUTINE THREAD-SAFE */
      WrapperObject::NumericalSample WrapperObject::execute(void * p_state,
					                    const NumericalSample & in) const 
      /* throw(InvalidArgumentException,WrapperInternalException) */
      {
	const UnsignedLong inDimension(getInNumericalPointDimension(p_state));
	// Check the in argument
	if (in.getDimension() != inDimension)
	  throw InvalidArgumentException(HERE)
	    << "Argument 'in' has incorrect size (" << in.getDimension()
	    << "). Expected = " << getInNumericalPointDimension(p_state);

	const UnsignedLong size(in.getSize());

	// We create a point structure to embed the in NumericalPoint passed as argument
	struct sample inSample;
	inSample.size_ = size;
	inSample.data_ = new struct point[size];
	// We pass the input sample by reference
	for(UnsignedLong i = 0; i < size; i++)
	  {
	    inSample.data_[i].size_ = inDimension;
	    inSample.data_[i].data_ = const_cast<NumericalScalar *>(&in[i][0]);
	  }

	// We create the output sample
	const UnsignedLong outDimension(getOutNumericalPointDimension(p_state));
	NumericalSample out(size, outDimension);
	// We create a point structure to embed the returned out NumericalSample
	struct sample outSample;
	outSample.size_ = size;
	outSample.data_ = new struct point[size];
	// We pass the output sample by reference
	for(UnsignedLong i = 0; i < size; i++)
	  {
	    outSample.data_[i].size_ = outDimension;
	    outSample.data_[i].data_ = &out[i][0];
	  }

	// Then we call the wrapper execution function for a sample
	Bool switchToPointEvaluation = false;
	try
	  {
	    WrapperErrorCode returnCode = (*(wrapperSymbols_->execSampleSymbol_))( p_state, & inSample, & outSample, exchangedData_.get(), error_.get() );
	    if (returnCode == WRAPPER_NOT_IMPLEMENTED) {
	      Log::Info("Sample evaluation function NOT provided. Switching to point evaluation");
	      switchToPointEvaluation = true;

	    } else if (returnCode != WRAPPER_OK)
	      throw WrapperInternalException(HERE)
		<< "Wrapper function '" << getFunctionName(EXECUTION_SAMPLE)
		<< "' returned error message: " << getErrorAsString(returnCode)
		<< ". Reason: " << getWrapperError( error_.get() );
	  }
	// If it fails, try to do it by calling the wrapper execution function for a point
	catch(WrapperInternalException & ex)
	  {
	    Log::Info("Sample evaluation function returned an error. Switching to point evaluation");
	    switchToPointEvaluation = true;
	  }
	catch(DynamicLibraryException & ex)
	  {
	    Log::Info("Sample evaluation function failure. Switching to point evaluation");
	    switchToPointEvaluation = true;
	  }
	if(switchToPointEvaluation)
	  {
	    WrapperErrorCode returnCode = WRAPPER_OK;
	    for(UnsignedLong i = 0; i < size; i++)
	      {
        	returnCode = (*(wrapperSymbols_->execSymbol_))( p_state, & inSample.data_[i], & outSample.data_[i], exchangedData_.get(), error_.get() );
	      }
	    if (returnCode != WRAPPER_OK)
	      throw WrapperInternalException(HERE)
		<< "Wrapper function '" << getFunctionName(EXECUTION)
		<< "' returned error message: " << getErrorAsString(returnCode)
		<< ". Reason: " << getWrapperError( error_.get() );
	  }

	delete [] inSample.data_;
	delete [] outSample.data_;

	return out;
      } // execute(void * p_state, const NumericalSample & in) const 


      /* Method gradient actually realizes the desired treatment */
      WrapperObject::Matrix WrapperObject::gradient(void * p_state, const NumericalPoint & in) const 
      /* throw(InvalidArgumentException,WrapperInternalException) */
      {
	// Check the in argument
	if (in.getDimension() != getInNumericalPointDimension(p_state))
	  throw InvalidArgumentException(HERE)
	    << "Argument 'in' has incorrect size (" << in.getDimension()
	    << "). Expected = " << getInNumericalPointDimension(p_state);
	


	// We create a point structure to embed the in NumericalPoint passed as argument
	struct point inPoint;
	inPoint.size_ = in.getDimension();
	inPoint.data_ = new double[inPoint.size_];
	for(unsigned int i=0; i < inPoint.size_; i++) inPoint.data_[i] = in[i];

	// We create a point structure to embed the returned out NumericalPoint
	struct matrix outMatrix;
	outMatrix.nbrows_ = getInNumericalPointDimension(p_state);
	outMatrix.nbcols_ = getOutNumericalPointDimension(p_state);
	outMatrix.data_   = new double[outMatrix.nbrows_*outMatrix.nbcols_];

#ifdef DEBUG
	{
	  OSS oss;
	  oss << "In struct point {size= " << inPoint.size_ << ", data=[";
	  const char * separator = "";
	  for(unsigned int i=0; i < inPoint.size_; i++, separator=",") oss << separator << inPoint.data_[i];
	  oss << "]}";	
	  Log::Debug( oss );
	}
#endif

	// Then we call the wrapper execution function
	WrapperErrorCode returnCode = (*(wrapperSymbols_->gradSymbol_))( p_state, & inPoint, & outMatrix, exchangedData_.get(), error_.get() );
	if (returnCode != WRAPPER_OK)
	  throw WrapperInternalException(HERE)
	    << "Wrapper function '" << getFunctionName(EXECUTION)
	    << "' returned error message: " << getErrorAsString(returnCode)
	    << ". Reason: " << getWrapperError( error_.get() );
	



#ifdef DEBUG
	{
	  OSS oss;
	  oss << "Out struct matrix {rows= " << outMatrix.nbrows_ 
	      << ", cols=" << outMatrix.nbcols_ << ", data=[[";
	  const char * separator = "";
	  for(unsigned int i=0; i < outMatrix.nbrows_; i++, separator="],[")
	    for(unsigned int j=0; j < outMatrix.nbcols_; j++, separator=",")
	      oss << separator << outMatrix.data_[i*outMatrix.nbcols_ + j];
	  oss << "]]}";	
	  Log::Debug( oss );
	}
#endif

	// Should be improved ! No need to double-copy data
	// OT::Base::Type::Collection<double> temporaryCollection(outMatrix.data_, outMatrix.data_+(outMatrix.nbrows_*outMatrix.nbcols_));
	Matrix out(outMatrix.nbrows_, outMatrix.nbcols_, outMatrix.data_, outMatrix.data_+(outMatrix.nbrows_*outMatrix.nbcols_) );

	delete [] inPoint.data_;
	delete [] outMatrix.data_;

	return out;
      }


      /* Method hessian actually realizes the desired treatment */
      WrapperObject::SymmetricTensor WrapperObject::hessian(void * p_state, const NumericalPoint & in) const 
      /* throw(InvalidArgumentException,WrapperInternalException) */
      {
	// Check the in argument
	if (in.getDimension() != getInNumericalPointDimension(p_state))
	  throw InvalidArgumentException(HERE)
	    << "Argument 'in' has incorrect size (" << in.getDimension()
	    << "). Expected = " << getInNumericalPointDimension(p_state);


	// We create a point structure to embed the in NumericalPoint passed as argument
	struct point inPoint;
	inPoint.size_ = in.getDimension();
	inPoint.data_ = new double[inPoint.size_];
	for(unsigned int i=0; i < inPoint.size_; i++) inPoint.data_[i] = in[i];

	// We create a point structure to embed the returned out NumericalPoint
	struct tensor outTensor;
	outTensor.nbrows_   = getInNumericalPointDimension(p_state);
	outTensor.nbcols_   = getInNumericalPointDimension(p_state);
	outTensor.nbsheets_ = getOutNumericalPointDimension(p_state);
	outTensor.data_     = new double[outTensor.nbrows_ * outTensor.nbcols_ * outTensor.nbsheets_];

#ifdef DEBUG
	{
	  OSS oss;
	  oss << "In struct point {size= " << inPoint.size_ << ", data=[";
	  const char * separator = "";
	  for(unsigned int i=0; i < inPoint.size_; i++, separator=",") oss << separator << inPoint.data_[i];
	  oss << "]}";	
	  Log::Debug( oss );
	}
#endif

	// Then we call the wrapper execution function
	WrapperErrorCode returnCode = (*(wrapperSymbols_->hessSymbol_))( p_state, & inPoint, & outTensor, exchangedData_.get(), error_.get() );
	if (returnCode != WRAPPER_OK)
	  throw WrapperInternalException(HERE)
	    << "Wrapper function '" << getFunctionName(EXECUTION)
	    << "' returned error message: " << getErrorAsString(returnCode)
	    << ". Reason: " << getWrapperError( error_.get() );


#ifdef DEBUG
	{
	  OSS oss;
	  oss << "Out struct tensor {rows= " << outTensor.nbrows_ 
	      << ", cols=" << outTensor.nbcols_ 
	      << ", sheets=" << outTensor.nbsheets_ 
	      << ", data=[[[";
	  const char * separator = "";
	  for(unsigned int i=0; i < outTensor.nbrows_; i++, separator="],[")
	    for(unsigned int j=0; j < outTensor.nbcols_; j++, separator="],[")
	      for(unsigned int k=0; k < outTensor.nbsheets_; k++, separator=",")
		oss << separator << outTensor.data_[(i*outTensor.nbcols_ + j)*outTensor.nbsheets_ + k];
	  oss << "]]]}";
	  Log::Debug( oss );
	}
#endif
	SymmetricTensor out(outTensor.nbrows_,
			    /* outTensor.nbcols_, */
			    outTensor.nbsheets_,
			    outTensor.data_,
			    outTensor.data_+(outTensor.nbrows_ *
					     outTensor.nbcols_ *
					     outTensor.nbsheets_));
	
	delete [] inPoint.data_;
	delete [] outTensor.data_;

	return out;
      }




      /* Method finalize calls the finalizationSymbol of the library */
      void WrapperObject::finalize(void * p_state) const
      /* throw (WrapperInternalException) */
      {
	WrapperErrorCode returnCode = (*(wrapperSymbols_->finalizeSymbol_))( p_state, exchangedData_.get(), error_.get() );
	if (returnCode != WRAPPER_OK)
	  throw WrapperInternalException(HERE)
	    << "Wrapper function '" << getFunctionName(FINALIZATION)
	    << "' returned error message: " << getErrorAsString(returnCode)
	    << ". Reason: " << getWrapperError( error_.get() );
      }




      /* State managment */
      void * WrapperObject::createNewState()
      /* throw (WrapperInternalException) */
      {
	void * p_state = 0;

	WrapperErrorCode returnCode = (*(wrapperSymbols_->stateCreationSymbol_))( & p_state, exchangedData_.get(), error_.get() );
	if (returnCode != WRAPPER_OK) 
	  throw WrapperInternalException(HERE)
	    << "Wrapper function '" << getFunctionName(STATE_CREATION)
	    << "' returned error message: " << getErrorAsString(returnCode)
	    << ". Reason: " << getWrapperError( error_.get() );
	
	return p_state;
      }


      void WrapperObject::deleteState(void * p_state)
      /* throw (WrapperInternalException) */
      {
	WrapperErrorCode returnCode = (*(wrapperSymbols_->stateDeletionSymbol_))( p_state, exchangedData_.get(), error_.get() );
	if (returnCode != WRAPPER_OK)
	  throw WrapperInternalException(HERE)
	    << "Wrapper function '" << getFunctionName(STATE_DELETION)
	    << "' returned error message: " << getErrorAsString(returnCode)
	    << ". Reason: " << getWrapperError( error_.get() );
	
      }



      /* Symbol accessor */
      LibrarySymbol WrapperObject::getSymbol(const String & name, Bool optional, LibrarySymbol alternateSymbol) const
      /* throw (WrapperInternalException) */
      {
	try {
	  return handle_.getSymbol(name);

	} catch (DynamicLibraryException & ex) {
	  if (optional) {
	    Log::Info(OSS() << "Library symbol '" << name << "' linked to an internal symbol");
	    return alternateSymbol;
	  } else throw WrapperInternalException(HERE) << ex;
	}
      }


    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
