/* $Id: register.h,v 1.56 2005/05/24 19:38:25 graziano Exp $ */

#ifndef REGISTER_H
#define REGISTER_H

#include "hosts.h"

#ifdef __cplusplus
extern "C" {
#endif

/* This package defines the interface for updating NWS name server records.  */

/* Definitions of name server attributes, objects, and sets of objects. */
typedef char *NwsAttribute;
#define NO_ATTRIBUTE NULL
typedef char *Object;
#define NO_OBJECT NULL
typedef char *ObjectSet;
#define NO_OBJECT_SET NULL

/* definitions of various TERMININATORS (don't think of arnold) */
#define ATTRIBUTE_TERMINATOR "\t" 
#define ATTRIBUTE_TERMINATOR_LEN 1
#define NAME_TERMINATOR ":"
#define NAME_TERMINATOR_LEN 1 
#define OBJECT_TERMINATOR "\t\t"
#define OBJECT_TERMINATOR_LEN 2

/* we track at most MAX_ADDRESSES network addresses for each machine */
#define MAX_ADDRESSES (20)

/*
 * the cookie contains the information for a host.
 */
struct host_cookie {
	char name[MAX_MACHINE_NAME];
	unsigned short port;
	Socket sd;
};

/**
 * Adds an attribute consisting of the name/value pair #addName#/#addValue# to
 * the object pointed to by #toObject#. The attributed is copied so you
 * can free addName /addValue after.
 */
void
AddNwsAttribute(Object *toObject,
             const char *addName,
             const char *addValue);


/**
 * Pulls each name:value pair out of #options# and adds it as an
 * attribute of #toObject#. The name:pair are ATTRIBUTE_TERMINATOR
 * separated.
 */
void
AddOptionToObject(	Object *toObject,
			const char* options);


/*
 * Delete an attribute from an object. Returns 1 if the attribute has
 * been found and succesfully deleted, 0 otherwise.
 */
int 
DeleteNwsAttribute(	Object *fromObject,
			const char *attribute);

/*
 * Checks if 2 Objects are equivalent: it checks if the 2 objects have
 * the same values for the following attributes:
 * 	name
 * 	objectclass
 * 	activity
 * 	owner
 * 	hostType
 *	controlName
 *	skillName
 *
 * Returns 1 if they are equivalent, 0 otherwise.
 */
int
AreObjectsEquivalent(	const Object lhs,
			const Object rhs);



/**
 * Inserts a copy of #addObject# into #toSet#. Only the first object in 
 * #addObject# is copied. Returns 1 on success 0 otherwise.
 */
int
AddObject(ObjectSet *toSet,
          const Object addObject);


/** 
 * Deletes #delObject# from #fromSet#. The objects are compared using
 * AreObjectsEquivalent to determine which objects needs to be deleted.
 * Returns 1 if an object has been deleted, 0 otherwise.
 */
int
DeleteObject(	ObjectSet *fromSet,
		const Object delObject);

/* 
 * This version of NwsAttributeName returns the attribute name contained
 * in #ofNwsAttribute in a newly allocated chunk of memory to be freed by
 * the caller. 
 */
char *
NwsAttributeName_r(const NwsAttribute ofNwsAttribute);


/* 
 * This version of NwsAttributeValue returns the value in a newly allocated
 * chunk of memory to be freed by the caller. 
 */
char *
NwsAttributeValue_r(const NwsAttribute ofNwsAttribute);


/*
** Returns the first attribute in #ofObject# that with name #name#, or
** NO_ATTRIBUTE if no attribute in #ofObject# has that name.
*/
NwsAttribute
FindNwsAttribute(const Object ofObject,
              const char *name);


/**
 * The following functions, free the memory for Object and ObjectSet
 * respectevily. Set the variable to NO_OBJECT(_SET).
 */
void
FreeObject(Object *toBeFreed);
void
FreeObjectSet(ObjectSet *toBeFreed);


/**
 * Returns a new object(set) containing no attributes, which should
 * eventually be passed to FreeObject(set).
 */
Object
NewObject(void);
ObjectSet
NewObjectSet(void);


/**
 * Returns the attribute in #ofObject# that follows #preceding#, or
 * NO_ATTRIBUTE if #preceding# is the last attribute in #ofObject#.
 * #preceding# may be NO_ATTRIBUTE, in which case the first attribute in
 * #ofObject# is returned.
 */
NwsAttribute
NextNwsAttribute(const Object ofObject,
              const NwsAttribute preceding);


/**
 * Returns the object in #ofSet# that follows #preceding#, or NO_OBJECT if
 * #preceding# is the last object in #ofSet#.  #preceding# may be NO_OBJECT, in
 * which case the first object in #ofSet# is returned.
 */
Object
NextObject(const ObjectSet ofSet,
           const Object preceding);


/**
 * A convenience routine to set all the fields of #host_c#.
 */
void
MakeHostCookie(	const char *host_name,
		short host_port,
		struct host_cookie *host_c);

/**
 * Functions which translate between host structures and printable strings.
 * The values returned by HostCImage() is volatile;
 * subsequent calls to the function will overwrite any prior value.
 * 
 * HostCIMage is NOT thread safe because they return a pointer to a
 * common statically allocate variable.
 */
const char *
HostCImage(const struct host_cookie *host_c);
int
Host2Cookie(const char *image,
            unsigned short defaultPort,
            struct host_cookie *cookie);


/**
 * Establishes a connection to the host described in #host_c# and
 * places a copy of the cookie socket in #sd#.  Returns 1 if successful,
 * else 0.
 */
int
ConnectToHost(struct host_cookie *host_c,
              Socket *sd);


/** 
 * Establishes a connection to the host described in #registration#.
 * trying  *all* the IP addresses in the registration. If succesfull
 * returns #cookie# filled with IP, port and socket, otherwise socket is
 * -1. If there is an already open socket in cookie, the socket is going
 * to be checked against the IP address in #registration# to be sure the
 * socket is a good one. If not, the value is ignored and a new socket is
 * used.
 * Returns 1 if succesfull (the cookie is good), otherwise 0. 
 */
int
ConnectToObject(	Object registration,
			struct host_cookie *cookie);


/*
** Registers #object# with the name server connected to #withWho#.  The
** registration will last for #forHowLong# seconds, or forever if #forHowLong#
** is 0.  Returns 1 if successful, else 0.
*/
#define NEVER_EXPIRE 0.0
int
Register(struct host_cookie *withWho,
         const Object object,
         double forHowLong);


/**
 * Retrieves from #whoFrom# all objects that have attributes that match
 * #filter#.  #filter# has an LDAP-like format of attribute name/value
 * pairs; the value may contain wildcard characters. Uses #timeout#
 * seconds during communication (0 disable timeouts and -1 set the
 * automatic dicovery of timeout). If successful, returns 1 and
 * sets #retrieved# to a set containing the retrieved objects (which should
 * eventually be passed to FreeObjectSet()); else returns 0.
 */
int
RetrieveObjects(struct host_cookie *whoFrom,
                const char *filter,
                ObjectSet *retrieved,
		int timeout);

/*
** Directs the name server connected to #withWho# to purge all registrations
** matching #filter#, a LDAP-like filter like that passed to RetrieveObjets().
** Returns 1 if successful, else 0.
*/
int
Unregister(struct host_cookie *withWho,
           const char *filter);

/* This is an internal structure used by both the memory and the
 * nameserver. It keeps a list of registrations tied together.
 * SearchForName and CheckRegistration can be used to operate on the
 * structure: the first one look for the registration given a name while
 * the second ensure that there is enough room to add a registration. The
 * idea here is to keep the registration ordered by name! It's up to the
 * client to do that. 
 * 
 * vals[i] with i < howMany shouldn't be NULL
 *
 * USE: howMany keeps the number of registration (expired or not!)
 * dirty keeps the number of free slots.
 *
 */
typedef struct {
	int howMany;			/* the number of registartion we have */
	int dirty;			/* how many unused entry we have */
	char **vals;			/* the registrations are here */
	unsigned long *expirations;	/* and their expiration */
} registrations;

/*
 * Initialize #r#. Returns 0 on failure.
 */
int
InitRegistrations(	registrations **r);


/*
 * It looks for #name# in the values of the 'name' attribute in the
 * registrations #r#. It returns the positions where the #name# belongs
 * in #r# in #ind# (it can returns r->howMany!): it returns 1 if at
 * least an objects was found (and the higher index if more than one was
 * present) or 0 if the objects would need to be inserted.  If
 * #matchExact# is not set, returns success when matching up to
 * lenght of #name#, not the length of the registration's name.
 */
int
SearchForName(	registrations *r,
		const char *name,
		int matchExact,
		int *ind);

/*
 * Similar to SearchForName, it uses AreObjectEquivalent to look for
 * equivalent object. Return 1 if succesful (and #ind# is the position of
 * the object), 0 otherwise (and #ind# is the position at which the
 * object should be inserted). #ind# is -1 in case of errors.
 */
int
SearchForObject(	registrations *r,
			Object obj,
			int *ind);

/* 
 * Insert #registration# that have #expiration# in #r# at position #ind#. 
 * Room is made in #r# to accomodate for an extra element.
 *
 * Returns 1 on success, 0 on failure (out of memory or erratic object).
 */
int
InsertRegistration(	registrations *r,
			const Object obj,
			unsigned long expiration,
			int ind);



/* 
 * Delete the registration at #ind#. Returns 0 on failure. 
 */
int
DeleteRegistration(	registrations *r,
			int ind);

/*
 * Check that there is enough room to insert a new element: if not makes
 * enough room for the registration.
 */
void
CheckRegistrations(	registrations *r);

/* Generate a tab-separated list of names from the registrations
 * structure. The returned string needs to be freed */
char *
GenerateList(const registrations *r);



#ifdef __cplusplus
}
#endif

#endif
