// file kernel/n/x86-64/mul_n2.S: O(n^2) multiplication of natural integers
/*-----------------------------------------------------------------------+
 |  Copyright 2005-2006, Michel Quercia (michel.quercia@prepas.org)      |
 |                                                                       |
 |  This file is part of Numerix. Numerix is free software; you can      |
 |  redistribute it and/or modify it under the terms of the GNU Lesser   |
 |  General Public License as published by the Free Software Foundation; |
 |  either version 2.1 of the License, or (at your option) any later     |
 |  version.                                                             |
 |                                                                       |
 |  The Numerix Library is distributed in the hope that it will be       |
 |  useful, but WITHOUT ANY WARRANTY; without even the implied warranty  |
 |  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU  |
 |  Lesser General Public License for more details.                      |
 |                                                                       |
 |  You should have received a copy of the GNU Lesser General Public     |
 |  License along with the GNU MP Library; see the file COPYING. If not, |
 |  write to the Free Software Foundation, Inc., 59 Temple Place -       |
 |  Suite 330, Boston, MA 02111-1307, USA.                               |
 +-----------------------------------------------------------------------+
 |                                                                       |
 |                        Multiplication quadratique                     |
 |                                                                       |
 +-----------------------------------------------------------------------*/

                      # +------------------------------+
                      # |  Multiplication par un long  |
                      # +------------------------------+

# unsigned long xn(mul_1)(chiffre *a, long la, unsigned long b, chiffre *c)
#
# entre :
# a = naturel de longueur la
# b = long >= 0
# c = naturel de longueur la, peut tre confondu avec a
#
# sortie :
# c <- a*b
# retourne la retenue

#ifdef assembly_sn_mul_1
QUICKENTER(sn_mul_1)

	leaq   (%rdi,%rsi,8), %rdi
	leaq   (%rcx,%rsi,8), %rcx
	movq   %rdx,  %r8
	xorq   %rdx,  %rdx
	negq   %rsi
	jz     2f

	ALIGN(8)
1:
	movq   (%rdi,%rsi,8), %rax
	movq   %rdx, %r9
	mulq   %r8
	addq   %r9,  %rax
	adcq   $0,   %rdx
	movq   %rax, (%rcx,%rsi,8) 
	incq   %rsi
	jne    1b

2:
	movq   %rdx, %rax
	ret
	
	
#endif /* assembly_sn_mul_1 */

                 # +---------------------------------------+
                 # |  Multiplication et addition droule  |
                 # +---------------------------------------+
# entre :
# rbx = 0
# rcx = -8*ceil(l/8)
# rdx = poids fort retenue entrante
# rsi = &a[l-1]
# rdi = &c[l]
# rbp = m
# r8  = r9 = poids faible retenue entrante
#
# sortie :
# rax <- ind.
# rbx <- retenue pour rdx
# rcx <- 0
# rdx <- poids fort retenue sortante
# r8  <- ind.
# r9  <- poids faible retenue sortante
# c <- c + m*a

	ALIGN(32)
.Lsn_mul_add_loop:

	# corps de boucle  drouler. taille du code = 23 octets
	# on code les instructions en hexadcimal pour tre sr qu'elles
	# totaliseront 23 octets pour un chiffre (les versions rcentes
	# de GAS liminent les dplacement nuls ce qui racourcit le code
	# associ au premier chiffre).
#undef BODY
#if 0
#define BODY(u,v) \
	movq  u(%rsi,%rcx,8), %rax ;\
	leaq  (%rdx,%rbx,1), %r8   ;\
        mulq  %rbp                 ;\
	addq  %r9,  u(%rdi,%rcx,8) ;\
	adcq  %rax, %r8            ;\
	setc  %bl                  ;\
                                   ;\
	movq  v(%rsi,%rcx,8), %rax ;\
	leaq  (%rdx,%rbx,1), %r9   ;\
        mulq  %rbp                 ;\
	addq  %r8,  v(%rdi,%rcx,8) ;\
	adcq  %rax, %r9            ;\
	setc  %bl                  
#else
#define BODY(u,v) \
	.byte 0x48,0x8b,0x44,0xce,u  ;\
 	.byte 0x4c,0x8d,0x04,0x1a    ;\
        .byte 0x48,0xf7,0xe5         ;\
	.byte 0x4c,0x01,0x4c,0xcf,u  ;\
	.byte 0x49,0x11,0xc0         ;\
	.byte 0x0f,0x92,0xc3         ;\
                                     ;\
	.byte 0x48,0x8b,0x44,0xce,v  ;\
 	.byte 0x4c,0x8d,0x0c,0x1a    ;\
        .byte 0x48,0xf7,0xe5         ;\
	.byte 0x4c,0x01,0x44,0xcf,v  ;\
	.byte 0x49,0x11,0xc1         ;\
	.byte 0x0f,0x92,0xc3
#endif

	# boucle droule pour 8 chiffres
	BODY( 0, 8)
	BODY(16,24)
	BODY(32,40)
	BODY(48,56)
	addq $8,%rcx
	jnz .Lsn_mul_add_loop
	ret
	
               # +-------------------------------------------+
               # |  Multiplication et soustraction droule  |
               # +-------------------------------------------+
# entre :
# rbx = 0
# rcx = -8*ceil(l/8)
# rdx = poids fort retenue entrante
# rsi = &a[l-1]
# rdi = &c[l]
# rbp = m
# r8  = r9 = poids faible retenue entrante
#
# sortie :
# rax <- ind.
# rbx <- retenue pour rdx
# rcx <- 0
# rdx <- poids fort retenue sortante
# r8  <- ind.
# r9  <- poids faible retenue sortante
# c <- c - m*a

	ALIGN(32)
.Lsn_mul_sub_loop:

	# corps de boucle  drouler. taille du code = 23 octets
	# on code les instructions en hexadcimal pour tre sr qu'elles
	# totaliseront 23 octets pour un chiffre (les versions rcentes
	# de GAS liminent les dplacement nuls ce qui racourcit le code
	# associ au premier chiffre).
#undef BODY
#if 0
#define BODY(u,v) \
	movq  u(%rsi,%rcx,8), %rax ;\
	leaq  (%rdx,%rbx,1), %r8   ;\
        mulq  %rbp                 ;\
	subq  %r9,  u(%rdi,%rcx,8) ;\
	adcq  %rax, %r8            ;\
	setc  %bl                  ;\
                                   ;\
	movq  v(%rsi,%rcx,8), %rax ;\
	leaq  (%rdx,%rbx,1), %r9   ;\
        mulq  %rbp                 ;\
	subq  %r8,  v(%rdi,%rcx,8) ;\
	adcq  %rax, %r9            ;\
	setc  %bl                  
#else
#define BODY(u,v) \
	.byte 0x48,0x8b,0x44,0xce,u  ;\
 	.byte 0x4c,0x8d,0x04,0x1a    ;\
        .byte 0x48,0xf7,0xe5         ;\
	.byte 0x4c,0x29,0x4c,0xcf,u  ;\
	.byte 0x49,0x11,0xc0         ;\
	.byte 0x0f,0x92,0xc3         ;\
                                     ;\
	.byte 0x48,0x8b,0x44,0xce,v  ;\
 	.byte 0x4c,0x8d,0x0c,0x1a    ;\
        .byte 0x48,0xf7,0xe5         ;\
	.byte 0x4c,0x29,0x44,0xcf,v  ;\
	.byte 0x49,0x11,0xc1         ;\
	.byte 0x0f,0x92,0xc3
#endif

	# boucle droule pour 8 chiffres
	BODY( 0, 8)
	BODY(16,24)
	BODY(32,40)
	BODY(48,56)
	addq $8,%rcx
	jnz .Lsn_mul_sub_loop
	ret
	

                      # +------------------------------+
                      # |  Multiplication quadratique  |
                      # +------------------------------+

# entre :
#   a = naturel de longueur la     rsi = &a, rdx = la
#   b = naturel de longueur lb     rbx = &b, rcx = lb
#   c = naturel de longueur la+lb  rdi = &c
# contraintes : 0 < lb <= la
#               si lb > 1, c ne doit pas recouvrir ni a ni b
#               si lb = 1, c peut recouvrir a ou b
#
# sortie :
#   c <- a * b
#
# registres modifis :
#   rax <- ind.
#   rbx <- ind.
#   rcx <- 0
#   rdx <- ind.
#   rsi <- &a[la]
#   rdi <- &c[la+lb-1]
#   rbp,r8,r9,r10,r11,r12,r13 <- ind.
#   r14, r15 inchangs
        
#ifdef assembly_sn_mul_n2
#undef L
#define L(x) .Lsn_fmul_n2_##x
        ALIGN(32)
#ifdef debug_mul_n2
.Lsn_fmul_n2_buggy:
#else
.Lsn_fmul_n2:
#endif

	# initialise les registres
	movq   (%rbx), %rbp         # rbp <- b[0]
	leaq   (%rbx,%rcx,8), %r10  # r10 <- &b[lb]
	negq   %rcx
	movq   %rcx,   %r12         # r12 <- -lb
	leaq   (%rsi,%rdx,8),  %rsi # a += la
	leaq   -8(%rdi,%rdx,8),%rdi # c += la-1
	negq   %rdx
	movq   %rdx,   %rcx
	leaq   1(%rdx),%r11         # r11 <- 1-la

	# prcalcule le droulement de la boucle interne
	leaq   .Lsn_mul_add_loop(%rip), %r13
	movq   %r11,   %rax
	andq   $7,     %rax         # rax <- (1 - la) mod 8
	leaq   (%rax,%rax,2), %rdx  # r13 += 23*rax
	leaq   (%r13,%rdx,8), %r13
	subq   %rax,   %r13

	# c <- b[0]*a
	xorq   %rbx,   %rbx         # init retenues
	xorq   %rdx,   %rdx
	ALIGN(8)
L(loop_0):
	movq   (%rsi,%rcx,8), %rax  # rax <- a[j]
	movq   %rdx,   %r8          # r8  <- pfort prcdent
	mulq   %rbp                 # rdx:rax <- b[0]*a[j]
	addq   %r8,    %rax         # ajoute la retenue
	adcq   %rbx,   %rdx
	incq   %rcx                 # j++
	movq   %rax, (%rdi,%rcx,8)  # sauve le poids faible dans c[j]
	jnz    L(loop_0)
	jmp    L(next)

	# boucle sur les chiffres suivants de b
	ALIGN(16)
L(loop_b):
	movq   (%r10,%r12,8),  %rbp # rbp <- b[i]
	movq   -8(%rsi,%r11,8),%rax # rax <- a[0]
	mulq   %rbp                 # calcule le premier produit
	movq   %rax,   %r8          # init retenues
	movq   %rax,   %r9
	movq   %r11,   %rcx         # rcx <- -8*ceil((la-1)/8)
	andq   $-8,    %rcx
	setc   %bl                  # rbx <- 0
	call   *%r13                # c += a*b[i]
	addq   %r9,  (%rdi)         # sauve les deux derniers chiffres
	adcq   %rbx, %rdx
L(next):
	addq   $8,     %rdi
	movq   %rdx,  (%rdi)
	incq   %r12                 # lb--
	jnz    L(loop_b)
	ret


                              # +---------------+
                              # |  interface C  |
                              # +---------------+

#  void xn(mul_n2)(chiffre *a, long la, chiffre *b, long lb, chiffre *c)
#
#  entre :
#  a = naturel de longueur la
#  b = naturel de longueur lb
#  c = naturel de longueur la+lb, non confondu avec a ou b
#  contraintes : 0 < lb <= la
#
#  sortie :
#  c <- a*b

#ifdef debug_mul_n2
ENTER(sn_mul_n2_buggy)
#else
ENTER(sn_mul_n2)
#endif

	movq   %rdx,   %rbx
	movq   %rsi,   %rdx
	movq   %rdi,   %rsi
	movq   %r8,    %rdi
#ifdef debug_mul_n2
        call   .Lsn_fmul_n2_buggy
#else
        call   .Lsn_fmul_n2
#endif
        RETURN_WITH_SP
#endif /* assembly_sn_mul_n2 */

        # cas o la version assembleur est dsactive ou dbogue :
        # sn_fmul_n2 renvoie vers la version C
        
#if !defined(assembly_sn_mul_n2) || defined(debug_mul_n2)
        ALIGN(32)
.Lsn_fmul_n2:

	movq   %rdi,   %r8
	movq   %rsi,   %rdi
	movq   %rdx,   %rsi
	movq   %rbx,   %rdx
	
	movq   %rdi,   %r15
	movq   %rsi,   %r14
	movq   %r8,    %r13
	movq   %rcx,   %r12
	
        call   SUBR(sn_mul_n2)
	
	xorq   %rcx,   %rcx
	leaq   (%r15,%r14,8), %rsi
	leaq   (%r13,%r14,8), %rdi
	leaq -8(%rdi,%r12,8), %rdi
	ret
        
#endif /* !defined(assembly_sn_mul_n2) || defined(debug_mul_n2) */

       
                                 # +---------+
                                 # |  Carr  |
                                 # +---------+

# entre :
#   a = naturel de longueur la     rsi = &a, rdx = la
#   c = naturel de longueur 2*la   rdi = &c
#
# contraintes : 0 < la
#               si la > 1, c ne doit pas recouvrir a
#               si la = 1, c peut recouvrir a
#
# sortie :
#   c <- a^2
#
# registres modifis :
#   rax <- ind.
#   rbx <- ind.x
#   rcx <- 0
#   rdx <- ind.
#   rsi <- &a[la]
#   rdi <- &c[2*la-2]
#   rbp,r8,r9,r10,r11 <- ind.
#   r12,r13,r14,r15 inchangs

#ifdef assembly_sn_sqr_n2
#undef L
#define L(x) .Lsn_fsqr_n2_##x
        
#ifdef debug_mul_n2
.Lsn_fsqr_n2_buggy:
#else
.Lsn_fsqr_n2:
#endif

	# initialise les registres
	movq   (%rsi), %rbp         # rbp <- 2*a[0]
	addq   %rbp, %rbp
	leaq   (%rsi,%rdx,8), %rsi  # a += la
	leaq  -8(%rdi,%rdx,8),%rdi  # c += la-1
	leaq   8(%rdi,%rdx,8),%r9   # r9 <- &c[2*la]
	negq   %rdx                 # la <- -la
	leaq   (%rdx,%rdx,1), %rcx  # rcx <- -2*la
	incq   %rdx
	movq   %rdx,   %r10         # r10 <- 1 - la
	xorq   %rbx,   %rbx         # initialise les retenues
	xorq   %r8,    %r8

	# prcalcule le droulement de la boucle interne
	leaq   .Lsn_mul_add_loop(%rip), %r11
	andq   $7,     %rdx         # rdx <- (1 - la) mod 8
	leaq   (%rdx,%rdx,2), %rax  # r11 += 23*rdx
	leaq   (%r11,%rax,8), %r11
	subq   %rdx, %r11

        # initialise c avec les a[i]^2
	ALIGN(8)
L(squares):
	movq   (%rsi,%rcx,4), %rax
	mulq   %rax
	movq   %rax,  (%r9,%rcx,8)
	movq   %rdx, 8(%r9,%rcx,8)
	addq   $2, %rcx
	jnz    L(squares)

	movq   %r10,   %rcx         # rcx <- -8*ceil((la-1)/8)
	andq   $-8,    %rcx
	jnz    L(loop_a)
	ret

        # ajoute les doubles produits
        ALIGN(8)
L(loop_a):
	movq   %r8,    %r9          # init retenues
        xorq   %rdx,   %rdx
	call   *%r11                # c += 2a[i]*a[i+1..la-1]
	addq   %r9,    (%rdi)       # sauve les deux derniers chiffres
	adcq   %rbx,   %rdx
	addq   $8,     %rdi         # c++
	movb   $0,     %bl
	addq   %rdx,   (%rdi)
	adcq   %rbx,  8(%rdi)

	movq   (%rsi,%r10,8), %rbp  # rbp <- a[i+1]
        btq    $63, -8(%rsi,%r10,8) # CF <- msb(a[i])
	cmovc  %rbp,   %r8          # r8 <- retenue tour suivant
	cmovnc %rbx,   %r8
	adcq   %rbp,   %rbp         # rbp <- 2*a[i+1]) + msb(a[i])

	incq   %r10                 # la--
	addq   $23,    %r11         # mise  jour adresse de saut
	leaq   .Lsn_mul_add_loop(%rip), %rax
	testq  $7,     %r10
	cmovz  %rax,   %r11
	movq   %r10,   %rcx         # rcx <- -8*ceil((la-1)/8)
	andq   $-8,    %rcx
	jnz    L(loop_a)
	
        # dernire retenue
	addq   %r8,    (%rdi)
	adcq   %rbx,  8(%rdi)
        ret



                              # +---------------+
                              # |  interface C  |
                              # +---------------+

#  void xn(sqr_n2)(chiffre *a, long la, chiffre *b)
#
#  entre :
#  a = naturel de longueur la
#  b = naturel de longueur 2*la, non confondu avec a
#  contraintes : 0 < la
#
#  sortie :
#  b <- a^2

#ifdef debug_mul_n2
ENTER(sn_sqr_n2_buggy)
#else
ENTER(sn_sqr_n2)
#endif

	movq   %rdx,   %rax
	movq   %rsi,   %rdx
	movq   %rdi,   %rsi
	movq   %rax,   %rdi
#ifdef debug_mul_n2
        call   .Lsn_fsqr_n2_buggy
#else
        call   .Lsn_fsqr_n2      
#endif
        RETURN_WITH_SP
#endif /* assembly_sn_sqr_n2 */

        # cas o la version assembleur est dsactive ou dbogue :
        # sn_fsqr_n2 renvoie vers la version C
        
#if !defined(assembly_sn_sqr_n2) || defined(debug_mul_n2)
        ALIGN(32)
.Lsn_fsqr_n2:

	movq   %rdi,   %rax
	movq   %rsi,   %rdi
	movq   %rdx,   %rsi
	movq   %rax,   %rdx

	movq   %rdi,   %r15
	movq   %rsi,   %r14
	movq   %rdx,   %r13
	
        call   SUBR(sn_sqr_n2)
	
	xorq   %rcx,   %rcx
	leaq   (%r15,%r14,8), %rsi
	leaq   (%r13,%r14,8), %rdi
	leaq -16(%rdi,%r14,8), %rdi
	ret
        
#endif /* !defined(assembly_sn_sqr_n2) || defined(debug_mul_n2) */




