/*
 * Nautilus Actions
 * A Nautilus extension which offers configurable context menu actions.
 *
 * Copyright (C) 2005 The GNOME Foundation
 * Copyright (C) 2006, 2007, 2008 Frederic Ruaudel and others (see AUTHORS)
 * Copyright (C) 2009, 2010 Pierre Wieser and others (see AUTHORS)
 *
 * This Program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This Program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this Library; see the file COPYING.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *   Frederic Ruaudel <grumz@grumz.net>
 *   Rodrigo Moya <rodrigo@gnome-db.org>
 *   Pierre Wieser <pwieser@trychlos.org>
 *   ... and many others (see AUTHORS)
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>

#include <api/na-object-api.h>

#include "na-importer.h"

extern gboolean iimporter_initialized;		/* defined in na-iimporter.c */
extern gboolean iimporter_finalized;		/* defined in na-iimporter.c */

static NAObjectItem *is_importing_already_exists( const NAObjectItem *importing, const NAIImporterListParms *parms );

/**
 * na_importer_import_from_uri:
 * @pivot: the #NAPivot pivot for this application.
 * @parms: a #NAIImporterUriParms structure.
 *
 * Returns: the import operation code.
 */
guint
na_importer_import_from_uri( const NAPivot *pivot, NAIImporterUriParms *parms )
{
	static const gchar *thisfn = "na_importer_import_from_uri";
	GList *modules, *im;
	guint code;

	g_return_val_if_fail( NA_IS_PIVOT( pivot ), IMPORTER_CODE_PROGRAM_ERROR );

	g_debug( "%s: pivot=%p, parms=%p", thisfn, ( void * ) pivot, ( void * ) parms );

	code = IMPORTER_CODE_PROGRAM_ERROR;

	if( iimporter_initialized && !iimporter_finalized ){

		code = IMPORTER_CODE_NOT_WILLING_TO;
		modules = na_pivot_get_providers( pivot, NA_IIMPORTER_TYPE );

		for( im = modules ; im && code == IMPORTER_CODE_NOT_WILLING_TO ; im = im->next ){

			if( NA_IIMPORTER_GET_INTERFACE( NA_IIMPORTER( im->data ))->from_uri ){
				code = NA_IIMPORTER_GET_INTERFACE( NA_IIMPORTER( im->data ))->from_uri( NA_IIMPORTER( im->data ), parms );
			}
		}

		na_pivot_free_providers( modules );
	}

	return( code );
}

/**
 * na_importer_import_from_list:
 * @pivot: the #NAPivot pivot for this application.
 * @parms: a #NAIImporterListParms structure.
 *
 * Imports a list of URIs.
 *
 * Each successfully imported #NAObjectItem-derived object is added to
 * the 'imported' list of the structure, in the same order than the URIs.
 *
 * Messages which may be generated by each individual import operation
 * are added to the 'messages' member of @parms.
 *
 * Returns: the last import operation code.
 */
guint
na_importer_import_from_list( const NAPivot *pivot, NAIImporterListParms *parms )
{
	static const gchar *thisfn = "na_importer_import_from_list";
	GSList *iuri;
	NAIImporterUriParms uri_parms;
	guint code;

	g_return_val_if_fail( NA_IS_PIVOT( pivot ), IMPORTER_CODE_PROGRAM_ERROR );

	g_debug( "%s: pivot=%p, parms=%p", thisfn, ( void * ) pivot, ( void * ) parms );

	code = IMPORTER_CODE_PROGRAM_ERROR;
	parms->imported = NULL;

	if( iimporter_initialized && !iimporter_finalized ){

		code = IMPORTER_CODE_NOT_WILLING_TO;

		uri_parms.version = 1;
		uri_parms.mode = parms->mode;
		uri_parms.window = parms->window;
		uri_parms.check_fn = ( NAIImporterCheckFn ) is_importing_already_exists;
		uri_parms.check_fn_data = parms;
		uri_parms.messages = parms->messages;

		for( iuri = parms->uris ; iuri ; iuri = iuri->next ){

			uri_parms.uri = ( gchar * ) iuri->data;
			uri_parms.imported = NULL;

			code = na_importer_import_from_uri( pivot, &uri_parms );

			if( uri_parms.imported ){
				parms->imported = g_list_prepend( parms->imported, uri_parms.imported );
			}
		}

		parms->imported = g_list_reverse( parms->imported );
	}

	return( code );
}

static NAObjectItem *
is_importing_already_exists( const NAObjectItem *importing, const NAIImporterListParms *parms )
{
	NAObjectItem *exists;
	GList *ip;

	exists = NULL;
	gchar *importing_id = na_object_get_id( importing );
	g_debug( "na_importer_is_importing_already_exists: importing_id=%s", importing_id );

	/* is the importing item already in the current importation list ?
	 */
	for( ip = parms->imported ; ip && !exists ; ip = ip->next ){
		gchar *id = na_object_get_id( ip->data );
		if( !strcmp( importing_id, id )){
			exists = NA_OBJECT_ITEM( ip->data );
		}
		g_free( id );
	}

	g_free( importing_id );

	/* if not found in our current importation list,
	 * then check the via provided function and data
	 */
	if( !exists ){
		exists = parms->check_fn( importing, parms->check_fn_data );
	}

	return( exists );
}
