#ifndef OBJECTVISITORS_H
#define OBJECTVISITORS_H

#include "ObjectFwd.h"

//----------------------------------------------------------------------------
/**
 * This is a base class for all visitors for the ObjectBase hierarchy.
 */
class ObjectVisitor
{
  public:
    virtual ~ObjectVisitor() = 0;

    inline void visit(Barrier *b) { do_visit(b); }
    inline void visit(BlackHole *b) { do_visit(b); }
    inline void visit(Crate *c) { do_visit(c); }
    inline void visit(Grenade *g) { do_visit(g); }
    inline void visit(Grinder *g) { do_visit(g); }
    inline void visit(MagnetBase *m) { do_visit(m); }
    inline void visit(Missile *m) { do_visit(m); }
    inline void visit(MortarBase *m) { do_visit(m); }
    inline void visit(ParticleBase *p) { do_visit(p); }
    inline void visit(ParticleFountainBase *f) { do_visit(f); }
    inline void visit(Platform *p) { do_visit(p); }
    inline void visit(ProjectileBase *p) { do_visit(p); }
    inline void visit(SAMBatteryBase *s) { do_visit(s); }
    inline void visit(Ship *s) { do_visit(s); }
    inline void visit(SwitchBase *s) { do_visit(s); }
    inline void visit(Tank *t) { do_visit(t); }
    inline void visit(Thorn *t) { do_visit(t); }
    inline void visit(TurretBase *t) { do_visit(t); }

    /**
     * Called from ObjectRepository::acceptObjects().
     *
     * @return false, to abort the for loop in the ObjectRepository.
     */
    virtual bool continueForLoopInVisitObjectList() const { return true; }

  protected:
    ObjectVisitor() {}

  private:
    virtual void do_visit(Barrier *b) {}
    virtual void do_visit(BlackHole *b) {}
    virtual void do_visit(Crate *c) {}
    virtual void do_visit(Grenade *g) {}
    virtual void do_visit(Grinder *g) {}
    virtual void do_visit(MagnetBase *m) {}
    virtual void do_visit(Missile *m) {}
    virtual void do_visit(MortarBase *m) {}
    virtual void do_visit(ParticleBase *p) {}
    virtual void do_visit(ParticleFountainBase *f) {}
    virtual void do_visit(Platform *p) {}
    virtual void do_visit(ProjectileBase *p) {}
    virtual void do_visit(SAMBatteryBase *s) {}
    virtual void do_visit(Ship *s) {}
    virtual void do_visit(SwitchBase *s) {}
    virtual void do_visit(Tank *t) {}
    virtual void do_visit(Thorn *t) {}
    virtual void do_visit(TurretBase *t) {}
};


//----------------------------------------------------------------------------
/**
 * This is a base class for all const visitors for the ObjectBase hierarchy.
 */
class ObjectConstVisitor
{
  public:
    virtual ~ObjectConstVisitor() = 0;

    inline void visit(const Barrier *b) { do_visit(b); }
    inline void visit(const BlackHole *b) { do_visit(b); }
    inline void visit(const Crate *c) { do_visit(c); }
    inline void visit(const Grenade *g) { do_visit(g); }
    inline void visit(const Grinder *g) { do_visit(g); }
    inline void visit(const MagnetBase *m) { do_visit(m); }
    inline void visit(const Missile *m) { do_visit(m); }
    inline void visit(const MortarBase *m) { do_visit(m); }
    inline void visit(const ParticleBase *p) { do_visit(p); }
    inline void visit(const ParticleFountainBase *f) { do_visit(f); }
    inline void visit(const Platform *p) { do_visit(p); }
    inline void visit(const ProjectileBase *p) { do_visit(p); }
    inline void visit(const SAMBatteryBase *s) { do_visit(s); }
    inline void visit(const Ship *s) { do_visit(s); }
    inline void visit(const SwitchBase *s) { do_visit(s); }
    inline void visit(const Tank *t) { do_visit(t); }
    inline void visit(const Thorn *t) { do_visit(t); }
    inline void visit(const TurretBase *t) { do_visit(t); }

    /**
     * Called from ObjectRepository::acceptObjects().
     *
     * @return false, to abort the for loop in the ObjectRepository.
     */
    virtual bool continueForLoopInVisitObjectList() const { return true; }

  protected:
    ObjectConstVisitor() {}

  private:
    virtual void do_visit(const Barrier *b) {}
    virtual void do_visit(const BlackHole *b) {}
    virtual void do_visit(const Crate *c) {}
    virtual void do_visit(const Grenade *g) {}
    virtual void do_visit(const Grinder *g) {}
    virtual void do_visit(const MagnetBase *m) {}
    virtual void do_visit(const Missile *m) {}
    virtual void do_visit(const MortarBase *m) {}
    virtual void do_visit(const ParticleBase *p) {}
    virtual void do_visit(const ParticleFountainBase *f) {}
    virtual void do_visit(const Platform *p) {}
    virtual void do_visit(const ProjectileBase *p) {}
    virtual void do_visit(const SAMBatteryBase *s) {}
    virtual void do_visit(const Ship *s) {}
    virtual void do_visit(const SwitchBase *s) {}
    virtual void do_visit(const Tank *t) {}
    virtual void do_visit(const Thorn *t) {}
    virtual void do_visit(const TurretBase *t) {}
};


//----------------------------------------------------------------------------
/**
 * This const visitor walks through an ObjectList
 * and tries to find a Platform, where the given ship
 * lies directly on top of it (has landed on it).
 */
class FindLandingPlatformConstVisitor : public ObjectConstVisitor
{
  public:
    FindLandingPlatformConstVisitor(const Ship *ship)
        : m_ship(ship), m_platform(NULL) {}
    ~FindLandingPlatformConstVisitor() { m_ship = NULL; }

    /**
     * @return The found platform, or NULL.
     */
    inline const Platform *getPlatform() const { return m_platform; }

  private:
    //------------------------------------------------------------------------
    void do_visit(const Platform *p);

    bool continueForLoopInVisitObjectList() const
    {
        return m_platform == NULL;
    }

    //------------------------------------------------------------------------
    const Ship *m_ship;
    const Platform *m_platform;
};


//----------------------------------------------------------------------------
/**
 * This const visitor is used, when the ship landed on a platform.
 * It will walk through an ObjectList and search for a Crate,
 * which collides with the ship.
 */
class HasCrateForShipConstVisitor : public ObjectConstVisitor
{
  public:
    HasCrateForShipConstVisitor(const Ship *ship)
        : m_ship(ship), m_crate(NULL) {}

    ~HasCrateForShipConstVisitor()
    {
        m_ship = NULL;
        m_crate = NULL;
    }

    /**
     * @return A Crate, the ship landed on, or NULL.
     */
    inline const Crate *getCrate() const { return m_crate; }

  private:
    //------------------------------------------------------------------------
    void do_visit(const Crate *c);

    bool continueForLoopInVisitObjectList() const
    {
        return m_crate == NULL;
    }

    //------------------------------------------------------------------------
    const Ship *m_ship;
    const Crate *m_crate;
};


//----------------------------------------------------------------------------
class CreateExplosionParticleConstVisitor : public ObjectConstVisitor
{
  public:
    //------------------------------------------------------------------------
    CreateExplosionParticleConstVisitor() {}
    ~CreateExplosionParticleConstVisitor() {}

  private:
    //------------------------------------------------------------------------
    void do_visit(const Grenade *g);
    void do_visit(const Missile *m);
    void do_visit(const MortarBase *m);
    void do_visit(const ProjectileBase *p);
    void do_visit(const SAMBatteryBase *s);
    void do_visit(const Ship *s);
    void do_visit(const Tank *t);
    void do_visit(const TurretBase *t);

    //------------------------------------------------------------------------
    void do_do_visit(const ControllableObjectBase *o);

    void do_createExplosionParticles(const ObjectBase *o,
                                     unsigned numParticles);
};


//----------------------------------------------------------------------------
/**
 * This const visitor is used to initialize a given ExplosionParticle
 * for a single object, that is about to explode.
 * Depending on the type of object,
 * the particle will be initialized in a different way.
 */
class InitializeExplosionParticleConstVisitor : public ObjectConstVisitor
{
  public:
    //------------------------------------------------------------------------
    InitializeExplosionParticleConstVisitor(ExplosionParticle *particle)
        : m_particle(particle) {}
    ~InitializeExplosionParticleConstVisitor() { m_particle = NULL; }

  private:
    //------------------------------------------------------------------------
    void do_visit(const Grenade *g);
    void do_visit(const Missile *m);
    void do_visit(const MortarBase *m);
    void do_visit(const ProjectileBase *p);
    void do_visit(const SAMBatteryBase *s);
    void do_visit(const Ship *s);
    void do_visit(const Tank *t);
    void do_visit(const TurretBase *t);

    //------------------------------------------------------------------------
    void do_do_visit(const ControllableObjectBase *o);

    //------------------------------------------------------------------------
    ExplosionParticle *m_particle;
};

#endif //OBJECTVISITORS_H
