#ifndef TILES_H
#define TILES_H

#include <map>

#include "SDLCalls.h"
#include "Exception.h"


//----------------------------------------------------------------------------
class TileException : public Exception
{
  public:
    TileException(Uint8 categoryId, Uint8 tileId);
    ~TileException();

    std::string toString() const;

  protected:
    Uint8 m_categoryId;
    Uint8 m_tileId;
};



//----------------------------------------------------------------------------
/**
 * This class is a container for all tiles that can be used in maps.
 * The internal storage is organized in a double-indexed map.
 * The first key is built out of the category and tile id.
 * The second key 
 */
class Tiles
{
    /// A key type containing the alpha and brightness value.
    typedef std::pair<Uint8, Uint8> ABKey;

    struct ABLessThan
    {
        bool operator()(const ABKey &k1, const ABKey &k2) const
        {
            if (k1.first < k2.first)  return true;
            if (k1.first == k2.first && k1.second < k2.second)  return true;
            return false;
        }
    };

    typedef std::map<ABKey, SDL_Surface*, ABLessThan> TileMap;

    /// A key type containing the category and tile id.
    typedef std::pair<Uint8, Uint8> CTKey;

    struct CTLessThan
    {
        bool operator()(const CTKey &k1, const CTKey &k2) const
        {
            if (k1.first < k2.first)  return true;
            if (k1.first == k2.first && k1.second < k2.second)  return true;
            return false;
        }
    };

    typedef std::map<CTKey, TileMap, CTLessThan> CategoryMap;

  public:
    //------------------------------------------------------------------------
    ~Tiles();

    //------------------------------------------------------------------------
    /**
     * @param category The category id.
     * @param tile The tile id.
     * @param alpha The desired alpha value.
     * @param brightness The desired brightness.
     *
     * @return A const SDL_Surface pointer to the desired tile.
     */
    const SDL_Surface *getTile(Uint8 category, Uint8 tile,
                               Uint8 alpha, Uint8 brightness) const;


    //------------------------------------------------------------------------
    static void init();
    static void destroy();

    //------------------------------------------------------------------------
    static inline const Tiles *getInstance()
    {
        return sm_instance;
    }

  protected:
    //------------------------------------------------------------------------
    Tiles();

  private:
    //------------------------------------------------------------------------
    /**
     * Helper-method for init() to read and store the tiles
     * of the given category.
     */
    void initCategory(Uint8 category, const char *dir);

    //------------------------------------------------------------------------
    /**
     * Helper-method for getTile() to create and store tiles
     * with an alpha value != SDL_ALPHA_OPAQUE or brightness != 255
     * on demand.
     */
    const SDL_Surface *createTile(Uint8 alpha, Uint8 brightness,
                                  CategoryMap::iterator &iter) const;


    //------------------------------------------------------------------------
    mutable CategoryMap m_map;

    //------------------------------------------------------------------------
    static Tiles *sm_instance;
};

#endif //TILES_H
