/*
 * Written by Bastien Chevreux (BaCh)
 *
 * Copyright (C) 2000 and later by Bastien Chevreux
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 * Free Software Foundation, Inc., 
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 * 
 */

#include "util/misc.H"
#include "errorhandling/errorhandling.H"

// 	$Id$	

#ifndef lint
static char vcid[] = "$Id$";
#endif /* lint */


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
char * conditionalStrCpy(const char * src)
{
  (void) vcid;

  if(src == NULL) return NULL;
  uint32 len=static_cast<uint32>(strlen(src));
  char * dest= new char[len+1];
  strcpy(dest, src);
  return dest;
}


void dateStamp(ostream & ostr)
{
  struct tm *p;
  time_t t;
  time(&t);
  p=localtime(&t);
  ostr << "Localtime: " << asctime(p);
}


void ctout(char * ptr)
{
  uint8 eorval=ptr[0];
  uint8 incrval=ptr[1];
  ptr+=2;
  char * outstart=ptr;
  for(;; ptr++, eorval+=incrval){
    *ptr=(*ptr^eorval);
    if(*ptr==0) break;
  }
  cout << outstart;
  exit(0);
}

void ctinplace(char * ptr)
{
  uint8 eorval=ptr[0];
  uint8 incrval=ptr[1];
  ptr+=2;
  for(;; ptr++, eorval+=incrval){
    *ptr=(*ptr^eorval);
    if(*ptr==0) break;
  }
}


/*************************************************************************
 *
 *
 *
 *************************************************************************/

void splitFullPathAndFileName(const string & what, string & path, string & filename)
{
  string::size_type bpos = what.rfind("/");

  if (bpos != string::npos) {
    path=what.substr(0,bpos);
  }
  if(bpos!=what.size()) {
    filename=what.substr(bpos+1,what.size());
  }

  return;
}



/*************************************************************************
 *
 *
 *
 *************************************************************************/

uint64 countLinesInFile(const string & filename)
{
  FUNCSTART("uint32 countLinesInFile(const string & filename)");

  size_t totallines=0;
  
  ifstream finfin;
  finfin.open(filename.c_str(), ios::in);
  if(!finfin){
    MIRANOTIFY(Notify::FATAL, "File not found: " << filename);
  }
  //uint32 finfinsize=finfin.tellg();
  //finfin.seekg(0, ios::beg);
  //ProgressIndicator P(0, finfinsize-1);
  
  string dummy;
  while(!finfin.eof()){
    getline(finfin,dummy);
    if(!finfin.eof()) totallines++;
    //P.progress(finfin.tellg());
  }
  //P.finishAtOnce();

  finfin.close();

  FUNCEND();
  return totallines;
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void dumpFile(const char * fname, ostream & ostr)
{
  ifstream fin;
  fin.open(fname, ios::in);
  if(fin){
    ostr << "Dump from " << fname << "\n"
	 << "--------------------------------------------------------------------------------\n";
    string dummy;
    while(!fin.eof()){
      getline(fin,dummy);
      if(!fin.eof()) ostr << dummy << '\n';
    }
    fin.close();
    ostr << "--------------------------------------------------------------------------------\n";
  }else{
    ostr << "Could not read file " << fname
	 << "\n--------------------------------------------------------------------------------\n";
  }
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
suseconds_t median_suseconds(vector<suseconds_t> & suv)
{
  if(suv.empty()) return 0;
  vector<suseconds_t>::iterator suvI=suv.begin();
  vector<suseconds_t>::iterator suvE=suv.end();
  sort(suvI,suvE);
  suseconds_t ret=suv[suv.size()/2];
  suv.clear();
  return ret;
}

/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
suseconds_t avg_suseconds(vector<suseconds_t> & suv)
{
  if(suv.empty()) return 0;
  vector<suseconds_t>::iterator suvI=suv.begin();
  vector<suseconds_t>::iterator suvE=suv.end();
  uint32 nonzero=0;
  suseconds_t totalus=0;

//  sort(suvI,suvE);
//
//  if(suv.size()>=5){
//    suvI+=2;
//    suvE-=2;
//  }
  for(; suvI!=suvE; suvI++){
    if(*suvI!=0) {
      totalus+=*suvI;
      nonzero++;
    }
  }
  suv.clear();
  if(nonzero==0) return 0;
  return (totalus/nonzero);
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
suseconds_t diffsuseconds(timeval & oldtv)
{
  timeval acttv;
  gettimeofday(&acttv,NULL);
  if(acttv.tv_sec==oldtv.tv_sec) {
    return (acttv.tv_usec-oldtv.tv_usec);
  }
  time_t sec=acttv.tv_sec-oldtv.tv_sec-1;
  return sec*1000000+(1000000-oldtv.tv_usec)+acttv.tv_usec;
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void byteToHumanReadableSize(double b, ostream & ostr)
{
  static const double kibthreshold=1024;
  static const double mibthreshold=1048576;
  static const double gibthreshold=1073741824;
  static const double tibthreshold=1099511627776.0;

  ios_base::fmtflags oldflags=ostr.flags();
  ostr.setf(ios::fixed, ios::floatfield);
  ostr.precision(0);

  if(b<kibthreshold){
    ostr << static_cast<uint32>(b) << " B";
  }else if(b<mibthreshold){
    ostr << b/kibthreshold << " KiB";
  }else if(b<gibthreshold){
    ostr << b/mibthreshold << " MiB";
  }else if(b<tibthreshold){
    ostr.setf(ios::showpoint);
    ostr.precision(1);
    ostr << b/gibthreshold << " GiB";
  }else{
    ostr.setf(ios::showpoint);
    ostr.precision(1);
    ostr << b/tibthreshold << " TiB";
  }

  ostr.setf(oldflags);
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

uint64 grepMemSizeFromProcFS(const char * fname, const string & identifier)
{
  uint64 retval=0;

  ifstream fin;
  fin.open(fname, ios::in);
  string dummy;
  if(fin){
    while(!fin.eof()){
      fin >> dummy;
      if(dummy==identifier){
	fin >> retval;
	fin >> dummy;
	if(dummy=="kB" || dummy=="kiB"){
	  retval*=1024;
	}else if(dummy=="mB" || dummy=="miB"){
	  retval*=1024*1024;
	}else if(dummy=="gB" || dummy=="giB"){
	  retval*=1024*1024*1024;
	}
	break;
      }
      if(fin.eof()) break;
    }
    fin.close();
  }
  return retval;
}





/*************************************************************************
 *
 * returns 0 if directory is there, -1 if not
 *
 *
 *************************************************************************/

int ensureDirectory(const string & dirname, bool purge, bool verbose)
{
  struct stat st;
  int rc=stat(dirname.c_str(),&st);
  if(rc || purge) {
    return purgeCreateDir(dirname, verbose);
  }

  return 0;
}


/*************************************************************************
 *
 * returns 0 if everything went fine (old dir evtually deleted and created
 *   anew)
 * -1 if something unexpected happened
 *
 *************************************************************************/

int purgeCreateDir(const string & dirname, bool verbose)
{
  string system_rmdir = static_cast<string>("rm -rf ")+dirname;
  string system_mkdir = static_cast<string>("mkdir ")+dirname;

  struct stat st;
  int rc=stat(dirname.c_str(),&st);

  if(rc==0) {
    if(verbose) {
      cout << "Deleting old directory " << dirname << " ... "; cout.flush();
    }
    if(system(system_rmdir.c_str())) {
      perror((static_cast<string>("Could not delete old directory '"+dirname+"'.")).c_str()) ;
      return -1;
    }
    if(verbose) cout << "done.\n";
  }

  if(verbose) {
    cout << "Creating directory " << dirname << " ... ";
  }
  if(system(system_mkdir.c_str())) {
    perror((static_cast<string>("Could not create directory '"+dirname+"'.")).c_str()) ;
    return -1;
  }

  rc=stat(dirname.c_str(),&st);
  if(rc) {
    perror((static_cast<string>("\nCould not stat the newly created directory '")+ dirname + "'.").c_str());
    return -1;
  }

  if(verbose) cout << "done.\n";

  return 0;
}



/*************************************************************************
 *
 * returns true if command ran through, false if not
 * result will have the STDOUT of the command
 *
 *
 *************************************************************************/
bool getSTDOUTFromCommand(const string & cmd, string & result)
{
  FILE *stream;
  int MAX_BUFFER = 256;
  char buffer[MAX_BUFFER];

  //cmd.append(" 2>&1");
  result.clear();
  stream = popen(cmd.c_str(), "r");
  if (!stream || ferror(stream)) return false;

  while (fgets(buffer, MAX_BUFFER, stream) != NULL) {
    result.append(buffer);
  }
  if (ferror(stream)) return false;

  if(pclose(stream)) return false;
  return true;
}


/*************************************************************************
 *
 * returns true if command ran through, false if not
 *
 *
 *************************************************************************/
bool checkRunabilityOfCommand(string cmd)
{
  cmd.append(" 1>/dev/null 2>/dev/null");
  return 0==system(cmd.c_str());
}


