/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */
 
#include <libsyncml/syncml.h>
#include <libsyncml/syncml_internals.h>

#ifdef ENABLE_HTTP
#include <libsyncml/sml_transport_internals.h>

#include "http_client.h"
#include "http_client_internals.h"

static void smlTransportHttpClientSend(void *userdata, void *link, SmlTransportData *data, SmlError *error);
static SmlBool smlTransportHttpClientFinalize(void *data, SmlError **error);

/**
 * @defgroup GroupIDPrivate Group Description Internals
 * @ingroup ParentGroupID
 * @brief The private part
 * 
 */
/*@{*/

static void _msgReceived(SoupMessage *msg, gpointer userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, msg, userdata);
	SmlTransportHttpClientEnv *env = userdata;
	SmlError *error = NULL;

	/* this mutex lock is necessary to avoid conflicts with session shutdown */
	g_mutex_lock(env->cleanupMutex);

	/* handle potential session shutdown */
	if (env->shutdown)
	{
		/* a call to finalize failed because of libsoup
		 * so try to finalize again and ignore the message
		 */
		g_mutex_unlock(env->cleanupMutex);
		smlTrace(TRACE_INTERNAL, "%s - failed finalize detected", __func__);
		smlTransportHttpClientFinalize(userdata, NULL);
		smlTrace(TRACE_EXIT, "%s", __func__);
		return;
	}

	/* handle a fresh connection */
	if (!env->connectDone)
	{
		/* This is the first answer or any other event which is
		 * detected after the first message was sent.
		 * We have to check and signal if the connection is ok or not.
		 */
		if (!SOUP_STATUS_IS_SUCCESSFUL (msg->status_code))
		{
			smlErrorSet(&error,
				 SML_ERROR_GENERIC,
				 "Connection failed (%d) - %s", msg->status_code, msg->reason_phrase);
			goto error;
		}
		else
		{
			smlTransportReceiveEvent(env->tsp, NULL, SML_TRANSPORT_EVENT_CONNECT_DONE, NULL, NULL);
		}
		env->connectDone = TRUE;
	}
	smlTrace(TRACE_INTERNAL, "Result: %d %s", msg->status_code, msg->reason_phrase);

	/* free originally send data */
	if (env->data != NULL)
	{
		smlTransportDataDeref(env->data);
		env->data = NULL;
	} else {
		smlTrace(TRACE_INTERNAL, "%s: WARNING: We sent an empty message.", __func__);
	}

	/* start http handling with some checks */

	/* check the library status for errors */
	if (!SOUP_STATUS_IS_SUCCESSFUL (msg->status_code)) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Call not successfull: %d %s",
			msg->status_code, msg->reason_phrase);
		goto error;
	}

	/* check the header of the received message */
	const char *header = soup_message_get_header(msg->response_headers, "Content-Type");
	smlTrace(TRACE_INTERNAL, "content type: %s", header);
	
	SmlMimeType mimetype = SML_MIMETYPE_UNKNOWN;
	if (header && !g_strncasecmp(header, SML_ELEMENT_XML, strlen(SML_ELEMENT_XML)))
		mimetype = SML_MIMETYPE_XML;
	else if(header && !g_strncasecmp(header, SML_ELEMENT_WBXML, strlen(SML_ELEMENT_WBXML)))
		mimetype = SML_MIMETYPE_WBXML;
	else if(header && !g_strncasecmp(header, SML_ELEMENT_SAN, strlen(SML_ELEMENT_SAN)))
		mimetype = SML_MIMETYPE_SAN;
	else if (header) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unknown mimetype");
		goto error;
	} else {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Faulty mimetype");
		goto error;
		
	}

	/* prepare the received message */
	char *data = smlTryMalloc0(msg->response.length, &error);
	if (!data)
		goto error;
	memcpy(data, msg->response.body, msg->response.length);
	
	SmlTransportData *tspdata = smlTransportDataNew(data, msg->response.length, mimetype, TRUE, &error);
	if (!tspdata)
		goto error_free_data;

	/* signal the received message */
	smlTransportReceiveEvent(env->tsp, NULL, SML_TRANSPORT_EVENT_DATA, tspdata, NULL);

	/* cleanup */
	smlTransportDataDeref(tspdata);
	g_mutex_unlock(env->cleanupMutex);

	smlTrace(TRACE_EXIT, "%s", __func__);
	return;
	
error_free_data:
	g_free(data);
error:
	smlTransportReceiveEvent(env->tsp, NULL, SML_TRANSPORT_EVENT_ERROR, NULL, error);
	g_mutex_unlock(env->cleanupMutex);
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&error));
	smlErrorDeref(&error);
}

/* This is the authentication callback for libsoup.
 * If a SyncML server uses HTTP then the specification allows
 * standard HTTP authentication. This callback provides the
 * http library with the user and passphrase for the SyncML
 * account. Usually this is not required because the most
 * SyncML server manage the authentication via SyncHdr.
 */
static void
_authenticate (SoupSession *session, SoupMessage *msg,
	      const char *auth_type, const char *auth_realm,
	      char **username, char **password, gpointer data)
{
	SmlTransportHttpClientEnv *env = data;
	smlTrace(TRACE_INTERNAL, "%s: authentication via auth_type %s", __func__, auth_type);
	*username = g_strdup(env->username);
	*password = g_strdup(env->password);
}

/*@}*/

/**
 * @defgroup GroupID Group Description
 * @ingroup ParentGroupID
 * @brief What does this group do?
 * 
 */
/*@{*/

static void *smlTransportHttpClientInit(SmlTransport *tsp, const void *data, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, tsp, data, error);
	smlAssert(tsp);
	smlAssert(data);
	
	g_type_init();

	smlTransportRunAsync(tsp, error);

	SmlTransportHttpClientEnv *env = smlTryMalloc0(sizeof(SmlTransportHttpClientEnv), error);
	if (!env)
		goto error;
	
	const SmlTransportHttpClientConfig *config = data;
	
	//FIXME do url checking. use whitelist
	env->uri = soup_uri_new(config->url);
	if (env->uri == NULL) {
		smlErrorSet(error, SML_ERROR_MISCONFIGURATION, "The specified url %s is wrong.", env->url);
		goto error_free_env;
	}
	env->url = g_strdup(config->url);
	env->proxy = g_strdup(config->proxy);
	env->connectDone = FALSE;
	env->cleanupMutex = g_mutex_new();
	env->shutdown = FALSE;
	
	env->tsp = tsp;
	if (config->proxy != NULL && strlen(config->proxy) > 0 ||
	    config->cafile != NULL && strlen(config->cafile) > 0)
	{
		smlTrace(TRACE_INTERNAL, "proxy is %s", config->proxy);
		smlTrace(TRACE_INTERNAL, "cafile is %s", config->cafile);
		env->session = soup_session_async_new_with_options(
					SOUP_SESSION_PROXY_URI, config->proxy,
					SOUP_SESSION_SSL_CA_FILE, config->cafile,
					NULL);
	} else {
		smlTrace(TRACE_INTERNAL, "no proxy");
		env->session = soup_session_async_new();
	}
	if (!env->session) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Unable to create new session");
		goto error_free_env;
	}

	/* add credentials to the environment and
	 * enable authentication callback for http authentication
	 */
	if (config->username || config->password)
	{
		env->username = g_strdup(config->username);
		env->password = g_strdup(config->password);
		g_signal_connect (env->session, "authenticate", G_CALLBACK (_authenticate), env);
	} else {
		env->username = NULL;
		env->password = NULL;
	}

	smlTrace(TRACE_EXIT, "%s: %p", __func__, env);
	return env;

error_free_env:
	g_free(env->uri);
	g_free(env);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

static SmlBool smlTransportHttpClientFinalize(void *data, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, data, error);
	smlAssert(data);
	SmlTransportHttpClientEnv *env = data;
	g_mutex_lock(env->cleanupMutex);
	
	smlAssert(env->tsp);

	env->shutdown = TRUE;

	// try to stop the session
	// this is a really nasty thing with lipsoup
	if (!soup_session_try_prune_connection(env->session))
	{
		// do not touch the env because otherwise _msgReceived can produce crashs
		// do not signal this as an error because this memory leak is not fixable
		// if you know what to do then feel free to correct me :)
		smlTrace(TRACE_EXIT_ERROR, "%s - connection was not pruned", __func__);
		g_mutex_unlock(env->cleanupMutex);
		return TRUE;
	}
	g_object_unref(env->session);
	env->session = NULL;

	// if we are here then there are definitely no actions on the road
	// from libsoup session. so it is safe to give up control and cleanup
	g_mutex_unlock(env->cleanupMutex);
	g_mutex_free(env->cleanupMutex);

	if (env->data != NULL)
	{
		smlTransportDataDeref(env->data);
		env->data = NULL;
	}
	
	g_free(env->uri);
	g_free(env->url);
	g_free(env->proxy);
	g_free(env);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
}

static void smlTransportHttpClientSend(void *userdata, void *link, SmlTransportData *data, SmlError *error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %p)", __func__, userdata, link, data, error);
	smlAssert(data);
	smlAssert(userdata);
	SmlTransportHttpClientEnv *env = userdata;
	smlAssert(env);
	smlAssert(env->uri);
	
	if (error)
		goto error;
		
	smlTrace(TRACE_INTERNAL, "tsp->context: %p", env->tsp->context);
	SoupMessage *msg = soup_message_new_from_uri(SOUP_METHOD_POST, env->uri);
	if (!msg) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "unknown libsoup error during message_new");
		goto error;
	}
	
	switch (data->type) {
		case SML_MIMETYPE_XML:
			soup_message_add_header(msg->request_headers, "Content-Type", SML_ELEMENT_XML);
			soup_message_add_header(msg->request_headers, "Accept", SML_ELEMENT_XML);
			break;
		case SML_MIMETYPE_WBXML:
			soup_message_add_header(msg->request_headers, "Content-Type", SML_ELEMENT_WBXML);
			soup_message_add_header(msg->request_headers, "Accept",	SML_ELEMENT_WBXML);
			break;
		case SML_MIMETYPE_SAN:
			soup_message_add_header(msg->request_headers, "Content-Type", SML_ELEMENT_SAN);
			soup_message_add_header(msg->request_headers, "Accept",	SML_ELEMENT_SAN);
		default:
			smlErrorSet(&error, SML_ERROR_GENERIC, "Unknown Mimetype %d", data->type);
			goto error_free_message;
	}

	msg->request.body = data->data;
	msg->request.length = data->size;
	msg->request.owner = SOUP_BUFFER_USER_OWNED;
	env->data = data;
	smlTransportDataRef(data);
	smlTrace(TRACE_INTERNAL, "%s: data: %s", __func__, g_strdup(msg->request.body));
	
	soup_session_queue_message(env->session, msg, _msgReceived, userdata);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return;
	
error_free_message:
	g_object_unref(msg);
error:
	smlTransportReceiveEvent(env->tsp, NULL, SML_TRANSPORT_EVENT_ERROR, NULL, error);
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&error));
	smlErrorDeref(&error);
}

/** @brief Function description
 * 
 * @param parameter This parameter does great things
 * @returns What you always wanted to know
 * 
 */
SmlBool smlTransportHttpClientNew(SmlTransport *tsp, SmlError **error)
{
	tsp->functions.initialize = smlTransportHttpClientInit;
	tsp->functions.finalize = smlTransportHttpClientFinalize;
	tsp->functions.send = smlTransportHttpClientSend;
	return TRUE;
}

#endif //ENABLE_HTTP

/*@}*/
