/*
    Copyright (C) 2001, 2002 Alcve
    Copyright (C) 2001, 2002, 2005 Guillaume Morin

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    $Id: ent.c,v 1.58 2005/03/19 02:50:39 gmorin Exp $
    $Source: /cvsroot/nss-mysql/nss-mysql/src/ent.c,v $
    $Date: 2005/03/19 02:50:39 $
    $Author: gmorin $
*/


#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "nss-mysql.h"

#include "lib.h"
#include "parser.h"
#include "passwd.h"
#include "group.h"
#include "nss-shadow.h"
#include "ent.h"

/* EOP */

static void prepare_forkhandler(void);
static void parent_forkhandler(void);
static void child_forkhandler(void);
static int forkhandler_isset = 0;
static pthread_mutex_t forkhandler_mutex = NSS_MYSQL_PTHREAD_MUTEX_INITIALIZER;

/* This structure is used to share information between
 * all ***..;ent functions. This is a cache of all IDs */

typedef struct {
        MYSQL_RES * res;
}ENT_S;

static ENT_S * ent[3] = {NULL,NULL,NULL};

/*EOP*/

/* mutexes to make functions reentrant !! */
static pthread_mutex_t mutex[3] = 
        {NSS_MYSQL_PTHREAD_MUTEX_INITIALIZER,
         NSS_MYSQL_PTHREAD_MUTEX_INITIALIZER,
         NSS_MYSQL_PTHREAD_MUTEX_INITIALIZER};


/* frees parsing results */
#if USE_SHADOW
#if USE_GROUP
#define free_parse() \
_nss_mysql_free_users(&options); \
_nss_mysql_free_groups(&goptions); \
_nss_mysql_free_shadow(&soptions); 
#else 
#define free_parse() \
_nss_mysql_free_users(&options); \
_nss_mysql_free_shadow(&soptions); 
#endif
#else
#if USE_GROUP
#define free_parse() \
_nss_mysql_free_users(&options); \
_nss_mysql_free_groups(&goptions);
#else
#define free_parse() \
_nss_mysql_free_users(&options);
#endif
#endif

#if USE_GROUP
#if USE_SHADOW
#define check_type(t,f) \
switch(t) { \
        case PASSWD_ENT_TYPE: if (DEBUG) _nss_mysql_log(LOG_ERR,f " called for passwd(0)"); break; \
        case GROUP_ENT_TYPE: if (DEBUG) _nss_mysql_log(LOG_ERR,f " called for group(1)"); break; \
        case SHADOW_ENT_TYPE: if (DEBUG) _nss_mysql_log(LOG_ERR,f " called for shadow(2)"); break; \
        default: _nss_mysql_log(LOG_ERR,f " called for unknow type %d",t); \
                         return NSS_STATUS_UNAVAIL; \
};
#else
#define check_type(t,f) \
switch(t) { \
        case PASSWD_ENT_TYPE: if (DEBUG) _nss_mysql_log(LOG_ERR,f " called for passwd(0)"); break; \
        case GROUP_ENT_TYPE: if (DEBUG) _nss_mysql_log(LOG_ERR,f " called for group(1)"); break; \
        default: _nss_mysql_log(LOG_ERR,f " called for unknow type %d",t); \
                         return NSS_STATUS_UNAVAIL; \
};
#endif
#else
#if USE_SHADOW
#define check_type(t,f) \
switch(t) { \
        case PASSWD_ENT_TYPE: if (DEBUG) _nss_mysql_log(LOG_ERR,f " called for passwd(0)"); break; \
        case SHADOW_ENT_TYPE: if (DEBUG) _nss_mysql_log(LOG_ERR,f " called for shadow(2)"); break; \
        default: _nss_mysql_log(LOG_ERR,f " called for unknow type %d",t); \
                         return NSS_STATUS_UNAVAIL; \
};
#else
#define check_type(t,f) \
switch(t) { \
        case PASSWD_ENT_TYPE: if (DEBUG) _nss_mysql_log(LOG_ERR,f " called for passwd(0)"); break; \
        default: _nss_mysql_log(LOG_ERR,f " called for unknow type %d",t); \
                         return NSS_STATUS_UNAVAIL; \
};
#endif
#endif

/* setent
 * Initializes data for gr**ent functions
 * NOTE this function does _NOT_ use errno
 */
NSS_STATUS _nss_mysql_setent (int type, char * (*get_sql_query)(struct query *,
                        struct parse_result *, struct mysql_auth *)) {
        pthread_mutex_t * m;
        ENT_S **s;
        NSS_STATUS status;
        int fake_errno,ret;
        struct mysql_auth mysql_auth={NULL,0,NULL};
        struct passwdoptions options;
        struct groupoptions goptions;
        struct shadowoptions soptions;
        struct parse_result pr = {NULL,NULL,NULL};
        struct query q = {NULL,0,0,1};
        char * sql_query = NULL;
        
        check_type(type,"setent");
        m = &mutex[type];
       
        _nss_mysql_set_fork_handler(&forkhandler_isset,&forkhandler_mutex,
                        prepare_forkhandler,
                        parent_forkhandler,
                        child_forkhandler,
                        NULL);
        
        pthread_mutex_lock(m);
        s = &ent[type];

        memset(&options,0,sizeof(struct passwdoptions));
        memset(&goptions,0,sizeof(struct groupoptions));
        memset(&soptions,0,sizeof(struct shadowoptions));
        
        if ((*s) != NULL) {
                /* ent[type] must be completely reinitialized */
                if (DEBUG) _nss_mysql_log(LOG_ERR,"Warning: setent(%d): ent"
                                " was != NULL, it has been deleted", type);
                if ((*s)->res)
                       mysql_free_result((*s)->res); 

                free(*s);
                *s = NULL;
        }
        /* new ent */
        (*s) = malloc(sizeof(ENT_S));
        if ((*s) == NULL)
                goto out_nomem;
        memset((*s),0,sizeof(ENT_S));

        /* we parse conf files */
        switch(type) {
                case PASSWD_ENT_TYPE: 
                        pr.options_p = &options;
                        ret = _nss_mysql_read_conf_file("users",&pr);
                        break;
#if USE_GROUP
                case GROUP_ENT_TYPE: 
                        pr.options_p = &options;
                        pr.goptions_p = &goptions;
                        ret = _nss_mysql_read_conf_file("groups",&pr);
                        break;
#endif
#if USE_SHADOW
                case SHADOW_ENT_TYPE: 
                        pr.soptions_p = &soptions;
                        ret = _nss_mysql_read_conf_file("shadow",&pr);
                        break;
#endif
                default:
                        /* should not happen */
                        ret = 0;
        }
        
        if (! ret) {
                _nss_mysql_log(LOG_ERR,"setent(%d): conf file parsing failed",
                                type);
                goto out_unavail;
        }

        /* DB connection */
        switch(type) {
                case PASSWD_ENT_TYPE:
#if USE_GROUP
                case GROUP_ENT_TYPE: 
#endif
                        ret = _nss_mysql_db_connect(&mysql_auth,
                                        &options.con); 
                        break;
#if USE_SHADOW
                case SHADOW_ENT_TYPE: 
                        ret = _nss_mysql_db_connect(&mysql_auth,
                                        &soptions.con); 
                        break;
#endif
        }
        if (! ret) {
                _nss_mysql_log(LOG_ERR,"setent(%d): connection to the database"
                                " failed.",type);
                /* trust no one */
                mysql_auth.mysql = NULL;
                goto out_unavail;
        }


        sql_query = get_sql_query(&q,&pr,&mysql_auth);
        if (! sql_query) 
                goto out_nomem;
        status = _nss_mysql_send_query(&mysql_auth,sql_query,&(*s)->res,
                        &fake_errno);
	free(sql_query);

        if (status != NSS_STATUS_SUCCESS && (*s)->res) {
                mysql_free_result((*s)->res);
        }

        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"setent(%d): calling db_close()",type);
        _nss_mysql_db_close(&(mysql_auth.mysql)); 
        
        free_parse();
        pthread_mutex_unlock(m);
        if (DEBUG) 
                _nss_mysql_log(LOG_ERR,"setent(%d): finished. status: %d",
                                type,status);
        return status;

out_nomem:
        if (*s != NULL) {
                if ((*s)->res)
                        mysql_free_result((*s)->res);
                free(*s);
                *s = NULL; 
        }
        free_parse();
        if (mysql_auth.mysql)
                _nss_mysql_db_close(&(mysql_auth.mysql)); 
        
        pthread_mutex_unlock(m); 
        if (DEBUG) 
                _nss_mysql_log(LOG_ERR,"setent(%d): finished. status: %d",
                                type,NSS_STATUS_TRYAGAIN);
        return NSS_STATUS_TRYAGAIN;

out_unavail:
        if (*s != NULL) {
                if ((*s)->res)
                        mysql_free_result((*s)->res);
                free(*s);
                *s = NULL; 
        }
        free_parse();
        if (mysql_auth.mysql)
                _nss_mysql_db_close(&(mysql_auth.mysql)); 
         
        pthread_mutex_unlock(m); 
        if (DEBUG) 
                _nss_mysql_log(LOG_ERR,"setent(%d): finished. status: %d",
                                type,NSS_STATUS_UNAVAIL);
        return NSS_STATUS_UNAVAIL;
}

/* endent
 * Kills all data for [gs]et..ent functions
 * NOTE this function does _NOT_ use errno
 */
NSS_STATUS _nss_mysql_endent (int type) {
        ENT_S ** s;
        pthread_mutex_t * m;
        
        check_type(type,"endent");

        m = &mutex[type];
        pthread_mutex_lock(m);

        s = &ent[type];

        if ((*s) != NULL) {
                if ((*s)->res)
                        mysql_free_result((*s)->res);

                free((*s));
                (*s) = NULL;
        } else {
                if (DEBUG) _nss_mysql_log(LOG_ERR,"endent(%d): ent was NULL",
                                type);
        }
        
        if (DEBUG) _nss_mysql_log(LOG_ERR,"endent(%d) finished",type);
        pthread_mutex_unlock(m);
        return NSS_STATUS_SUCCESS;
}


/* getgrent
 * Gets info for every group
 * Arguments:
 * gr: array to fill
 * buffer: buffer, unused
 * buflen: size of buffer
 * errnop: ptr to the application errno
 */

NSS_STATUS _nss_mysql_getent_r (int type, void * to_fill,
                char * buffer, size_t buflen,int * errnop) {
        NSS_STATUS status = NSS_STATUS_NOTFOUND; /* to silence a warning */
        ENT_S ** s;
        pthread_mutex_t * m;
        
        check_type(type,"getent");

        m = &mutex[type];
        pthread_mutex_lock(m);

        s = &ent[type];
        *errnop = ENOENT;
       
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"getent called");

        /* sanity checks */
        if ((*s) == NULL) {
                if (DEBUG) _nss_mysql_log(LOG_ERR,"getent(%d): called with ent "
                                "== NULL",type);
                pthread_mutex_unlock(m);
                return NSS_STATUS_UNAVAIL;
        }

        if (! (*s)->res) {
                pthread_mutex_unlock(m);
                return NSS_STATUS_UNAVAIL;
        }
                

        switch(type) {
                case 0:
                        status = _nss_mysql_passwd_result_to_struct(
                                        to_fill,
                                        (*s)->res,
                                        errnop,
                                        buffer,
                                        buflen,
                                        0);

                        break;
#if USE_GROUP
                case 1:
                        status = _nss_mysql_group_result_to_struct(
                                        to_fill,
                                        (*s)->res,
                                        errnop,
                                        buffer,
                                        buflen,
                                        0);
                        break;
#endif
#if USE_SHADOW
                case 2:
                        status = _nss_mysql_shadow_result_to_struct(
                                        to_fill,
                                        (*s)->res,
                                        errnop,
                                        buffer,
                                        buflen,
                                        0);
                                break;
#endif
                }
        if (DEBUG) _nss_mysql_log(LOG_ERR,"getent(%d) exited with status %d",
                        type,status);
        pthread_mutex_unlock(m);
        return status;
}

void prepare_forkhandler(void) {
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"ent.c: calling prepare_forkhandler");
        pthread_mutex_lock(&mutex[0]);
        pthread_mutex_lock(&mutex[1]);
        pthread_mutex_lock(&mutex[2]);
        pthread_mutex_lock(&forkhandler_mutex);
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"ent.c: end of prepare_forkhandler");
}

void parent_forkhandler(void) {
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"ent.c: calling parent_forkhandler");
        pthread_mutex_unlock(&mutex[0]);
        pthread_mutex_unlock(&mutex[1]);
        pthread_mutex_unlock(&mutex[2]);
        pthread_mutex_unlock(&forkhandler_mutex);
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"ent.c: end of parent_forkhandler");
}

void child_forkhandler(void) {
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"ent.c: calling child_forkhandler");
        pthread_mutex_unlock(&mutex[0]);
        pthread_mutex_unlock(&mutex[1]);
        pthread_mutex_unlock(&mutex[2]);
        pthread_mutex_unlock(&forkhandler_mutex);
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"ent.c: end of child_forkhandler");
}

/* EOF */
