#ifndef BUFFY_CONFIG_STORAGE_H
#define BUFFY_CONFIG_STORAGE_H

/*
 * Copyright (C) 2004--2008  Enrico Zini <enrico@enricozini.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <string>
#include <vector>
#include <iosfwd>

namespace xmlpp {
class Element;
class Document;
class DomParser;
}

namespace buffy {
namespace config {

class Storage
{
protected:
	xmlpp::DomParser* m_xml_parser;
	
	// Cached elements
	xmlpp::Document* doc_conf;
	bool doc_conf_needs_freeing;
	xmlpp::Document* doc_defaults;

	xmlpp::Element* m_el_root;
	xmlpp::Element* m_def_root;

public:
	Storage();
	~Storage();
	
	void initEmpty(const std::string& rootName);
	void load(const std::string& fname);
	void save(const std::string& file);
	void dump(std::ostream& out);

	//
	// Low-level accessors
	//

	xmlpp::Element* getElement(xmlpp::Element* father, const std::string& path) const;
	xmlpp::Element* obtainElement(xmlpp::Element* father, const std::string& path);

	xmlpp::Element* getDefault(const std::string& key) const;

	/* Return the node pointed by this ConfigItem, or 0 if it does not exist */
	xmlpp::Element* nodeIfExists(const std::string& path) const;

	/* Return the node pointed by this ConfigItem, or the corresponding default
	 * node, or 0 if none of them exist */
	xmlpp::Element* nodeOrDefault(const std::string& path) const;

	/* Return the node pointed by this ConfigItem, creating it if it does not exist */
	xmlpp::Element* node(const std::string& path);

	//
	// General accessors
	//

	// Return true if this item is present in the configuration file
	bool isSet(const std::string& path) const;

	// Get the string value of this item
	std::string get(const std::string& path) const;

	// Get the value of this item, converted to bool
	bool getBool(const std::string& path) const;

	// Get the value of this item, converted to an int
	int getInt(const std::string& path) const;

	// Get the values of the child nodes of this item
	std::vector<std::string> getVector(const std::string& path) const;

	// Get the names of the child nodes of this item
	std::vector<std::string> children(const std::string& path) const;

	// Get the name="" attributes of the child nodes of this item
	std::vector<std::string> childNames(const std::string& path) const;

	// Return the item name (or name[attribute]) of the child item which
	// contains a child named 'selected' which has value 'true'
	// If none are found, returns the empty string.
	std::string selected(const std::string& path) const;

	void addDefault(const std::string& key, const std::string& val);
	void addDefault(const std::string& key, const std::vector<std::string>& val);

	// Delete the given key (and all subkeys) from the configuration.  After
	// this call, querying the given key will return the default value.
	void unset(const std::string& path);

	void set(const std::string& path, const std::string& val);
	void setBool(const std::string& path, bool val);
	void setInt(const std::string& path, int val);
	void setVector(const std::string& path, const std::vector<std::string>& val);

	static std::string escape(const std::string& val);
	static std::string unescape(const std::string& val);
};

}
}

// vim:set ts=4 sw=4:
#endif
