/***************************************************************************
 *   Copyright (C) 2005-2006 by Raul Fernandes                             *
 *   rgfbr@yahoo.com.br                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "stardict.h"

#include <qfile.h>
#include <qcstring.h>

#include <kdebug.h>

#include <zlib.h>

#define CHUNK 0xffffL

StarDict::StarDict( const QString &arq )
{
  // Ifo file
  if( !QFile::exists( arq ) )
  {
    m_isOk = false;
    return;
  }
  ifoFile = new QFile( arq );

  // Read the ifo file
  QString ifoline;
  ifoFile->open( IO_ReadOnly );
  while( !ifoFile->atEnd() )
  {
    ifoFile->readLine( ifoline, 1024 );
    if( ifoline.find( "=" ) == -1 ) continue;
    ifoline.remove( "\n" );
    if( ifoline.section( "=", 0, 0 ) == "version" ) m_version = ifoline.section( "=", 1 );
    if( ifoline.section( "=", 0, 0 ) == "bookname" ) m_bookname = ifoline.section( "=", 1 );
    if( ifoline.section( "=", 0, 0 ) == "sametypesequence" ) m_sametypesequence = ifoline.section( "=", 1 );
    if( ifoline.section( "=", 0, 0 ) == "idxfilesize" ) m_idxfilesize = ifoline.section( "=", 1 ).toULong();
    if( ifoline.section( "=", 0, 0 ) == "wordcount" ) m_wordcount = ifoline.section( "=", 1 ).toUInt();
    if( ifoline.section( "=", 0, 0 ) == "author" ) m_author = ifoline.section( "=", 1 );
    if( ifoline.section( "=", 0, 0 ) == "email" ) m_email = ifoline.section( "=", 1 );
    if( ifoline.section( "=", 0, 0 ) == "website" ) m_website = ifoline.section( "=", 1 );
    if( ifoline.section( "=", 0, 0 ) == "description" ) m_description = ifoline.section( "=", 1 );
    if( ifoline.section( "=", 0, 0 ) == "date" ) m_date = ifoline.section( "=", 1 );
  }
  ifoFile->close();
  if( m_sametypesequence != "m" )
  {
    // This class only support sametypesequence == "m" for now
    m_isOk = false;
    #ifndef NDEBUG

      kdDebug() << QString( "Dictionary not loaded. Stardict plugin supports only sametypesequence == \"m\"" ) << endl;

    #endif
    return;
  }

  // Index file
  // it should be on the same directory that ifo file
  QString defarq = arq;
  defarq.replace( ".ifo", ".idx" );
  if( !QFile::exists( defarq ) )
  {
    m_isOk = false;
    return;
  }
  idxFile = new QFile( defarq );

  // Definition file (.dict)
  // it should be on the same directory that index file
  defarq.replace( ".idx", ".dict" );
  // Checks if the definition file is compressed or not
  if( QFile::exists( defarq ) )
  {
    file = new QFile( defarq );
    isCompressed = false;
  }else{
    if( !QFile::exists( defarq + ".dz" ) )
    {
      m_isOk = false;
      return;
    }
    // The definition file is compressed (.dict.dz)
    // Read the header
    file = new QFile( defarq + ".dz" );
    isCompressed = true;
    file->open( IO_ReadOnly );
    if( file->getch() != 31 ) //ID1 = 31 (0x1f, \037)
    {
      m_isOk = false;
      return;
    }
    if( file->getch() != 139 ) //ID2 = 139 (0x8b, \213)
    {
      m_isOk = false;
      return;
    }
    file->getch(); // COMPRESSION
    FLAGS = file->getch();
    FTEXT = FLAGS & 1;
    FHCRC = FLAGS & 2;
    m_extraField = FLAGS & 4;
    m_hasName = FLAGS & 8;
    FCOMMENT = FLAGS & 16;
    m_mtime = (unsigned char)file->getch() + (unsigned char)file->getch() * 256 + (unsigned char)file->getch() * 256 * 256 + (unsigned char)file->getch() * 256 * 256 * 256;
    file->getch(); // Compression type
    file->getch(); // Operating System
    // If has extra field, read it
    if( m_extraField )
    {
      XLEN = (unsigned char)file->getch() + (unsigned char)file->getch() * 256;
      readExtraField();
    }
    // If has name of decompressed file, read it
    if( m_hasName )
    {
      readFileName();
    }
    // If has a comment field, read it
    if( FCOMMENT )
    {
      readFileName();
    }
    // If has a CRC field, read it
    if( FHCRC )
    {
      *crc16[0] = file->getch();
      *crc16[1] = file->getch();
    }
    // Get the current position
    // This is start position of the chunks of compressed data
    offset = file->at();
    file->close();
  }

#ifndef NOPTIMIZE
  QCString line(256);
  QString headword;
  struct entry entry;
  uint a;
  dic.clear();
  idxFile->open( IO_ReadOnly );
  idxFile->at( 4 );
  for(uint word=0;word<m_wordcount;word++)
  {
    a = 0;
    do
    {
      line[a] = idxFile->getch();
      a++;
    }while( line[a-1] != '\0' );

    headword = QString::fromUtf8( line.data() );

    for(a=0;a<8;a++) line[a] = idxFile->getch();
    // Catch the position of definition in definition file
    entry.position = (unsigned char)line[3] + 256 * (unsigned char)line[2] + 256 * 256 * (unsigned char)line[1] + 256 * 256 * 256 *(unsigned char)line[0];

    // Catch the size of definition in definition file
    entry.size = (unsigned char)line[7] + 256 * (unsigned char)line[6] + 256 * 256 * (unsigned char)line[5] + 256 * 256 * 256 *(unsigned char)line[4];

    dic.insert( headword, entry );
  }
  idxFile->close();
#endif
  m_isOk = true;
}


StarDict::~StarDict()
{
  delete file;
  delete idxFile;
}




/*!
    \fn StarDict::readExtraField()
 */
void StarDict::readExtraField()
{
  //kdDebug() << "readExtraField()" << endl;
  offsets.clear();
  SI1 = file->getch();
  SI2 = file->getch();
  // length of the subfield data
  LEN = (unsigned char)file->getch() + (unsigned char)file->getch() * 256;
  int size = (int)LEN - 6;
  // Version
  VER = (unsigned char)file->getch() + (unsigned char)file->getch() * 256;
  // length of a "chunk" of data
  CHLEN = (unsigned char)file->getch() + (unsigned char)file->getch() * 256;
  // how many chunks are preset
  CHCNT = (unsigned char)file->getch() + (unsigned char)file->getch() * 256;
  unsigned long data;
  for(int a = 0; a < size; a++)
  {
    // how long each chunk is after compression
    data = (unsigned char)file->getch() + (unsigned char)file->getch() * 256;
    a++;
    offsets.append( data );
  }
}


/*!
    \fn StarDict::readFileName()
 */
void StarDict::readFileName()
{
  //kdDebug() << "readFileName()" << endl;
  QString filename;
  char byte;
  byte = file->getch();
  while( byte != '\0' )
  {
    filename += byte;
    byte = file->getch();
  }
  m_filename = filename;
}


/*!
    \fn StarDict::readComment()
 */
void StarDict::readComment()
{
  kdDebug() << "readComment()" << endl;
  QString comment;
  char byte;
  byte = file->getch();
  while( byte != '\0' )
  {
    comment += byte;
    byte = file->getch();
  }
  m_comment = comment;
}


/*!
    \fn StarDict::search( const QString &word )
 */
QString StarDict::search( const QString &word )
{
  //kdDebug() << "StarDict::search()" << endl;
  struct entry entry;

#ifndef NOPTIMIZE
  // Find the headword in index file
  Dictionary::ConstIterator it = dic.find( word );
  if( it == dic.constEnd() ) return QString::null;
  entry = it.data();
#else
  QCString line(256);
  QString headword;
  bool found = false;
  idxFile->open( IO_ReadOnly );

  // Find the headword in index file
  uint a;
  idxFile->at( 4 );
  while( !idxFile->atEnd() )
  {
    a = 0;
    do
    {
      line[a] = idxFile->getch();
      a++;
    }while( line[a-1] != '\0' );

    headword = QString::fromUtf8( line.data() );

    for(a=0;a<8;a++) line[a] = idxFile->getch();

    if( QString::localeAwareCompare( headword, word ) != 0 ) continue;

    found = true;

    // Catch the position of definition in definition file
    entry.position = (unsigned char)line[3] + 256 * (unsigned char)line[2] + 256 * 256 * (unsigned char)line[1] + 256 * 256 * 256 *(unsigned char)line[0];

    // Catch the size of definition in definition file
    entry.size = (unsigned char)line[7] + 256 * (unsigned char)line[6] + 256 * 256 * (unsigned char)line[5] + 256 * 256 * 256 *(unsigned char)line[4];

    // Word found. Break the loop.
    break;
  }
  idxFile->close();

  // If not found, return a null string
  if( !found ) return QString::null;
#endif

  // Check if the definition file is compressed
  if( isCompressed )
  {
    ulong a = 0;

    // Calculate how many chunks we have to skip
    uint chunk = entry.position / CHLEN ;
    // Calculate the position of definition in chunk
    uint pos = entry.position % CHLEN;

    // Size of the chunk we are looking for
    unsigned long chunkLen = offsets[chunk];
    // If the word is in the end of chunk, we have to decompress two chunks
    if( (pos + entry.size) > CHLEN ) chunkLen += offsets[chunk + 1];

    // How many bytes we have to skip
    unsigned long skip = 0;
    for(uint a=0;a<chunk;a++) skip += offsets[a];

    // Stores the compressed data
    QByteArray data( chunkLen + 1 );
    data[chunkLen] = '\0';

    // Stores the decompressed data
    QCString result;

    // Definition file
    file->open( IO_ReadOnly );

    // Jump to chunk we are looking for
    file->at( offset + skip );

    // Get the compressed data
    for(a = 0; a < chunkLen; a++) data[a] = file->getch();
    data[a] = '\0';
    file->close();

    // Decompress the data
    result = Inflate( data );
    // Returns only the definition
    return QString::fromUtf8( result.mid( pos, entry.size ) );
  }else{
    // The file is not compressed
    file->open( IO_ReadOnly );
    // Jump to position of definition
    file->at( entry.position );
    // Get the definition
    QCString result( entry.size + 1 );
    for( uint a = 0; a < entry.size; a++ ) result[a] = file->getch();
    result[entry.size] = '\0';
    file->close();
    // Return the result
    return QString::fromUtf8( result.data() );
  }
}


QCString StarDict::Inflate( const QByteArray &data )
{
    //kdDebug()<< "Inflate()" << endl;
    int ret;
    z_stream strm;
    char out[CHUNK];
    QCString result( 65536 );

    // Inicialization of zlib
    strm.zalloc = Z_NULL;
    strm.zfree = Z_NULL;
    strm.opaque = Z_NULL;
    strm.avail_in = 0;
    strm.next_in = Z_NULL;
    ret = inflateInit2( &strm, -MAX_WBITS );
    if (ret != Z_OK)
      return "";

      // Compressed data
      strm.avail_in = data.size();
      strm.next_in = (Bytef*)data.data();

      /* run inflate() on input until output buffer not full */
      do {
        strm.avail_out = CHUNK;
        strm.next_out = (Bytef*)out;
        ret = inflate(&strm, Z_SYNC_FLUSH);
        switch (ret) {
          case Z_NEED_DICT:
            ret = Z_DATA_ERROR;     /* and fall through */
          case Z_DATA_ERROR:
          case Z_MEM_ERROR:
            (void)inflateEnd(&strm);
            return ""; // Error
        }
        result += out;
      } while (strm.avail_out == 0);

    /* clean up and return */
    ret = inflateEnd(&strm);
    return result;
}


int StarDict::size()
{
#ifndef NOPTIMIZE
  return dic.size();
#else
  return wordcount;
#endif
}
