/***************************************************************************
 *   Copyright (C) 2004-2008 by Giovanni Venturi                           *
 *   giovanni@ksniffer.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02110-1301, USA.          *
 ***************************************************************************/

#include <qapplication.h>
#include <qlayout.h>
#include <qsplitter.h>
#include <qheader.h>
#include <qtooltip.h>
#include <qwhatsthis.h>
#include <qstring.h>
#include <qstringlist.h>

#include <qgroupbox.h>
#include <qlabel.h>
#include <qtoolbutton.h>
#include <qtimer.h>
#include <qpoint.h>
#include <qtextedit.h>

#include <kurl.h>
#include <kiconloader.h>
#include <kicontheme.h>
#include <klistview.h>
#include <klistviewsearchline.h>

#include <klocale.h>
#include <kdebug.h>

#include <kuniqueapplication.h>
#include <kpopupmenu.h>

#include "ksniffermainwidget.h"
#include "ksnifferviewitem.h"
#include "packetmanager.h"
#include "frames/frameheadermanager.h"
#include "frames/frameprotocolmanager.h"
#include "protocols/protocolmanager.h"
#include "options/captureoptions.h"
#include "options/ksnifferconfig.h"

#include "kinfoip.h"

#include "ksniffermainwidget.moc"

KSnifferMainWidget::KSnifferMainWidget( QWidget* parent, const char* name )
    : QWidget( parent, name )
{
  if ( !name )
    setName( "MainWidget" );

  m_mainViewLayout = new QVBoxLayout( this, 11, 6, "MainWidgetLayout");
  m_mainViewLayout->setAlignment( Qt::AlignTop );

  // splitter widget
  QSplitter *vertSplitter = new QSplitter( QSplitter::Vertical, this, "vertSplitter" );

  // packets listview widget
  m_list = new KListView( vertSplitter, "m_list" );
  m_list->addColumn( i18n( "frame number", "Frame No." ) );
  m_list->addColumn( i18n( "Time" ) );
  m_list->addColumn( i18n( "Source" ) );
  m_list->addColumn( i18n( "Destination" ) );
  m_list->addColumn( i18n( "Protocol" ) );
  m_list->addColumn( i18n( "Information" ) );
  m_list->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)5, (QSizePolicy::SizeType)7, 0, 0, m_list->sizePolicy().hasHeightForWidth() ) );
  m_list->setAllColumnsShowFocus( true );

  // let sort column
  m_list->setShowSortIndicator( true );

  // disable clicking on "Information"
  //m_list->header()->setClickEnabled( false, 5 );

  // splitter widget
  QSplitter *horizSplitter = new QSplitter( QSplitter::Horizontal, vertSplitter, "horizSplitter" );

  // packets details listview widget
  m_detail = new QListView( horizSplitter, "m_detail" );
  KIconLoader kil;
  m_detail->addColumn( kil.loadIconSet("ksniffer", KIcon::Small), i18n( "Packet Details" ) );

  // disable "packet details"
  m_detail->header()->setClickEnabled( false, 0 );
  m_detail->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)5, (QSizePolicy::SizeType)3, 0, 0,
    m_detail->sizePolicy().hasHeightForWidth() ) );

  m_rawPacketAreaRight = new QTextEdit( horizSplitter, "rawbyte" );
  QToolTip::add( m_rawPacketAreaRight, i18n( "Show raw packet content" ) );
  QWhatsThis::add( m_rawPacketAreaRight, i18n( "In this area you will see the bytes contained by a network "
      "packet when you click on it from the packet list." ) );

  m_rawPacketAreaRight->hide();  // show this widget when you click onto a packet from the list
  QFont mono( "Monospace", 8 );
  m_rawPacketAreaRight->setFont( mono );
  QFontMetrics fm( mono );
  m_rawPacketAreaRight->setMinimumWidth( fm.width("Z")*77 );

  // the raw packet under the frame
  m_rawPacketAreaUnder = new QTextEdit( vertSplitter, "rawbytevert" );
  QToolTip::add( m_rawPacketAreaUnder, i18n( "Show raw packet content" ) );
  QWhatsThis::add( m_rawPacketAreaUnder, i18n( "In this area you will see the bytes contained by a network "
      "packet when you click on it from the packets list." ) );

  m_rawPacketAreaUnder->hide();  // show this widget when you click onto a packet from the list
  QFont monofont( "Monospace", 10 );
  m_rawPacketAreaUnder->setFont( monofont );

  // search line
  QToolButton* toolbtn = new QToolButton( this );
  toolbtn->setIconSet( SmallIconSet( QApplication::reverseLayout() ? "clear_left" : "locationbar_erase" ) );
  QToolTip::add( toolbtn, i18n( "Clear the \"Packet Filter\" search line" ) );
  QWhatsThis::add( toolbtn, i18n( "This clears the \"Packet Filter\" search line." ) );

  m_listSearch = new KListViewSearchLine( this, m_list );
  QToolTip::add( m_listSearch,
    i18n( "Type text here just after you stopped capture otherwise "\
          "you could lost packets" ) );

  QWhatsThis::add( m_listSearch,
    i18n( "<p>This bar allows you to quickly find packets in the list below. "\
          "Just type what you want to look for.</p><p>If you use it you "\
          "could lost packets during capture, so use it just when "\
          "you stopped capture or when the network traffic you are "\
          "monitoring is low.</p>" ) );

  QHBoxLayout *hLayout = new QHBoxLayout( 0L, -1, 0 );
  hLayout->addWidget( toolbtn );
  hLayout->addItem( new QSpacerItem( 10, 10, QSizePolicy::Minimum, QSizePolicy::Minimum ) );
  hLayout->addWidget( new QLabel( i18n("Packet Filter:"), this ) );
  hLayout->addItem( new QSpacerItem( 5, 5, QSizePolicy::Minimum, QSizePolicy::Minimum ) );
  hLayout->addWidget( m_listSearch );

  // add to layout
  m_mainViewLayout->addLayout( hLayout );
  m_mainViewLayout->addWidget( vertSplitter );

  // we don't need sorting detail
  m_detail->setSortColumn(-1);

  QWhatsThis::add( m_list, i18n("<p>In this area you get all the captured packets.</p>"\
    "<p>If you click with the right mouse button on a source or destination IP address you can get a "\
    "popup menu and choose the program that can give you some IP information (<i>whois</i>, "\
    "<i>traceroute</i>, ...).</p>") );
  QToolTip::add( m_list, i18n("The list of all captured packets") );

  QWhatsThis::add( m_detail, i18n("In this area you can see the details about a selected captured packet.") );
  QToolTip::add( m_detail, i18n("More details about the captured packets") );

  m_firstInsert = true;

  m_list->setColumnAlignment(0, Qt::AlignRight);
  m_list->setColumnAlignment(1, Qt::AlignRight);
  m_list->setColumnAlignment(2, Qt::AlignCenter);
  m_list->setColumnAlignment(3, Qt::AlignCenter);
  m_list->setColumnAlignment(4, Qt::AlignCenter);
  m_list->setColumnWidth(1, WIDTH_COL_1);
  m_list->setColumnWidth(2, WIDTH_COL_2);
  m_list->setColumnWidth(3, WIDTH_COL_3);
  m_list->setColumnWidth(4, WIDTH_COL_4);
  m_list->setColumnWidth(5, WIDTH_COL_5);

  m_detail->setColumnWidth(0, 550);
  m_detail->setRootIsDecorated(true);

  connect( m_list, SIGNAL( selectionChanged(QListViewItem*) ), this, SLOT( showDetails(QListViewItem*) ) );
  connect( m_list, SIGNAL( rightButtonPressed(QListViewItem*, const QPoint &, int) ),
    this, SLOT( clickedMouseButton(QListViewItem*, const QPoint &, int) ) );

  connect( m_detail, SIGNAL( expanded ( QListViewItem * ) ), this, SLOT( setExpand( QListViewItem * ) ) );
  connect( m_detail, SIGNAL( collapsed ( QListViewItem * ) ), this, SLOT( setCollapse( QListViewItem * ) ) );

  connect( toolbtn, SIGNAL( clicked() ), m_listSearch, SLOT( clear() ) );

  // need to disable updateSearch() when not useful if I need don't slow sniffing
  connect( m_listSearch, SIGNAL( textChanged ( const QString & ) ), this, SLOT( applyNewSearchFilter( const QString & ) ) );

  // no need to search when startup
  m_needToUpdateSearch = false;

  resize( QSize(414, 265) );
  m_options = new CaptureOptions;

  // at start up no loading from file
  m_loadFromFile = false;
  m_displayingPacket = true;

  m_info = new KInfoIP;

  // setting up the popup menu when click RMB on the packet list
  m_popupMenu = m_info->createPopupMenu();

  m_timer = new QTimer();
  connect(m_timer, SIGNAL(timeout()), this, SLOT(updateView()));
}


KSnifferMainWidget::~KSnifferMainWidget()
{
  // no need to delete child widgets, Qt does it all for us
  delete m_info;
  delete m_options;
}


void KSnifferMainWidget::print(QPainter */*p*/, int /*height*/, int /*width*/)
{
  // do the actual printing, here
  // p->drawText(etc..)
}


QString KSnifferMainWidget::currentURL()
{
  return QString::null;
}


void KSnifferMainWidget::openURL(const QString& url)
{
  openURL(KURL(url));
}


void KSnifferMainWidget::openURL(const KURL& url)
{
  Q_UNUSED( url );
}


void KSnifferMainWidget::slotOnURL(const QString& url)
{
  emit signalChangeStatusbar(url);
}


void KSnifferMainWidget::slotSetTitle(const QString& title)
{
  emit signalChangeCaption(title);
}


void KSnifferMainWidget::setExpand( QListViewItem* i )
{
  if ( i->parent() )
    m_bExpanding1 = true;
  else
    m_bExpanding0 = true;
}


void KSnifferMainWidget::setCollapse( QListViewItem* i )
{
  if ( i->parent() )
    m_bExpanding1 = false;
  else
    m_bExpanding0 = false;
}


void KSnifferMainWidget::applyNewSearchFilter( const QString & text )
{
  m_needToUpdateSearch = ( !text.isEmpty() );

  // we need to enable the timer if view need to be upadated, has to show data and the time is not yet enabled
  if (m_needToUpdateSearch)
  {
    if ((!m_options->showAfter()) && (!m_timer->isActive()))
      m_timer->start(TIME_WAITING_UPDATE);
  }
  else
  {
    // no more need to update the view if the filter was resetted
    m_timer->stop();
    m_mutex.lock();
    m_listSearch->updateSearch();
    m_mutex.unlock();
  }
}


void KSnifferMainWidget::clearView()
{
  m_list->clear();
  m_detail->clear();
  m_rawPacketArea->hide();
}


void KSnifferMainWidget::setSortList(bool value)
{
  if (value)
  {
    /*
    // this will "freeze" the GUI
    setUpdatesEnabled(false);
    m_list->setSortColumn(0);
    setUpdatesEnabled(true);
    repaint();
    */
  }
  else
    // to avoid crash you should not sort data when sniffing
    m_list->setSortColumn(-1);
}


void KSnifferMainWidget::setOptions( CaptureOptions *options )
{
  // just these three options are needed
  m_options->setShowAfter( options->showAfter() );
  m_options->setRawFramePosition( options->rawFramePosition() );
  m_options->setShowProtocolName( options->showProtocolName() );

  // assign the m_rawPacketArea
  if (m_options->rawFramePosition() == "under")
    m_rawPacketArea = m_rawPacketAreaUnder;
  else
    m_rawPacketArea = m_rawPacketAreaRight;
}


void KSnifferMainWidget::setPortNumberNameList( PortNumberNameList *pnnl )
{
  m_portNumberNameList = pnnl;
}


void KSnifferMainWidget::loadFromFile( bool load )
{
  m_loadFromFile = load;
}


void KSnifferMainWidget::displayPacket( bool toView )
{
  m_displayingPacket = toView;
}


bool KSnifferMainWidget::displayingPacket()
{
  return m_displayingPacket;
}


void KSnifferMainWidget::updateFilter()
{
  if (m_needToUpdateSearch)
    // update the view by the filter if really needed
    m_listSearch->updateSearch();
}


void KSnifferMainWidget::startSniffing()
{
  if (!m_options->showAfter())
    // just if the show packet is in real time you need to enable the view for the update
    if (m_needToUpdateSearch)
      m_timer->start(TIME_WAITING_UPDATE);
}


void KSnifferMainWidget::stopSniffing()
{
  // to stop sniffing the GUI has to update the view and stop the timer
  updateFilter();
  m_timer->stop();
}


void KSnifferMainWidget::updateView()
{
  // when you load a file you have to appy this updateView at the end of the load
  if (m_needToUpdateSearch)
  {
    m_mutex.lock();
    m_listSearch->updateSearch();
    m_mutex.unlock();
  }
}


void KSnifferMainWidget::changeRawFormPosition( bool rightPosition )
{
  if (rightPosition)
    m_rawPacketArea = m_rawPacketAreaRight;
  else
    m_rawPacketArea = m_rawPacketAreaUnder;
}


void KSnifferMainWidget::displayPacket(long ord, PacketManager *pktMan)
{
  // if the packets have to be displayed when captured or the
  // displaying request come from a sniffing from file can be displayed

  //FIXME: mmm than m_displayingPacket variable is unuseful?
  //if ( !m_options->showAfter() || m_loadFromFile )
  if ( m_displayingPacket || m_loadFromFile )
  {
    if (pktMan->getFrameNumber() != ord)
      pktMan->setFrameNumber( ord );

    if (m_firstInsert)
    {
      m_lastElem = new KSnifferViewItem( m_list, pktMan, m_portNumberNameList, m_options->showProtocolName() );
      m_firstInsert = false;
    }
    else
      m_lastElem = new KSnifferViewItem( m_list, m_lastElem, pktMan, m_portNumberNameList, m_options->showProtocolName() );
  }

  // we need to save PacketManager class reference first time
  if (ord == 1)
  {
    m_packetManager = pktMan;

    // if the the capture is not from file than start the thread
    if (!m_loadFromFile)
      startSniffing();
  }
}


void KSnifferMainWidget::showDetails( QListViewItem * item )
{
  if (item == NULL)
    return;

  // getting frame number
  int frameNumber = item->text(0).toInt();

  // clear the detail view
  m_detail->clear();

  m_packetManager->setFrameNumber( frameNumber );
  switch (m_packetManager->getFrameHType())
  {
    case DLT_EN10MB:
      break;
    case DLT_LINUX_SLL:
      break;
    default:
      kdDebug() << "Frame: decoding not yet implemented" << endl;
  }

  // "first" line in the detail view
  QListViewItem *itemFrame0 = new QListViewItem( m_detail, i18n("Frame number: %1").arg(frameNumber) );
  QListViewItem *subitemFrame0 = new QListViewItem( itemFrame0, i18n("general field for the frame header", "Capture time: %1")
      .arg(m_packetManager->strDetailTimeStamp()) );
  subitemFrame0 = new QListViewItem( itemFrame0, subitemFrame0,
                                     i18n("general field for the frame header", "Delay from the previous packet: %1 s")
                                         .arg(m_packetManager->strDiffTimePrevious()) );
  subitemFrame0 = new QListViewItem( itemFrame0, subitemFrame0,
                                     i18n("general field for the frame header", "Delay from the first packet: %1 s")
                                         .arg(m_packetManager->strDiffTimeFirst()) );
  subitemFrame0 = new QListViewItem( itemFrame0, subitemFrame0, i18n("general field for the frame header", "Packet length: %1 bytes")
      .arg(m_packetManager->frameLength()) );
  subitemFrame0 = new QListViewItem( itemFrame0, subitemFrame0,
                                     i18n("general field for the frame header", "Captured packet length: %1 bytes")
                                         .arg(m_packetManager->frameCapturedLength()) );

  FrameHeaderManager fhm(m_packetManager->getPacket());
  QStringList list = fhm.headerLines();
  if (!list.isEmpty())
  {
    // "second" line in the detail view
    QListViewItem *itemFrame1 = new QListViewItem( m_detail, itemFrame0, i18n("Frame header: %1").arg(fhm.strType()) );
    QListViewItem *subitemFrame1 = new QListViewItem ( itemFrame1, *list.begin() );

    for( QStringList::Iterator it = ++list.begin(); it != list.end(); ++it)
      subitemFrame1 = new QListViewItem( itemFrame1, subitemFrame1, *it );


    FrameProtocolManager fpm(m_packetManager->getPacket(), fhm.protocol());
    list = fpm.headerLines();
    if (!list.isEmpty())
    {
      // "third" line in the detail view
      QListViewItem *itemFrame2 = new QListViewItem( m_detail, itemFrame1, i18n("Protocol information: %1").arg( fpm.getProtocol()) );
      QListViewItem *subitemFrame2 = new QListViewItem ( itemFrame2, *list.begin() );

      for( QStringList::Iterator it = ++list.begin(); it != list.end(); ++it )
      {
        if (*it == "*open")
        {
          subitemFrame2 = new QListViewItem( itemFrame2, subitemFrame2, *(++it) );
          QListViewItem *subSubitemFrame2 = new QListViewItem( subitemFrame2, *(++it) );
          while( *(++it) != "*close" )
            subSubitemFrame2 = new QListViewItem( subitemFrame2, subSubitemFrame2, *it );
        }
        else
          subitemFrame2 = new QListViewItem( itemFrame2, subitemFrame2, *it );
      }

      ProtocolManager pm(m_packetManager->getPacket(), fpm.getSubProtocol(), m_portNumberNameList);
      list = pm.headerLines();
      if (!list.isEmpty())
      {
        // "forth" line in the detail view
        QListViewItem *itemFrame3 = new QListViewItem( m_detail, itemFrame2, i18n("specific protocol information", "%1 information:")
            .arg( fpm.strSubProtocol()) );
        QListViewItem *subitemFrame3 = new QListViewItem ( itemFrame3, *list.begin() );

        for( QStringList::Iterator it = ++list.begin(); it != list.end(); ++it)
        {
            viewSubtree(it, &itemFrame3, &subitemFrame3);
          //subitemFrame3 = new QListViewItem( itemFrame3, subitemFrame3, *it );
        }

        if (m_bExpanding1)
        {
          itemFrame1->setOpen(true);
          itemFrame2->setOpen(true);
          itemFrame3->setOpen(true);
        }
      }
    }
  }
  showRawPacketDetails();

  // expand item lines if needed
  if (m_bExpanding0)
    itemFrame0->setOpen(true);
}


void KSnifferMainWidget::viewSubtree(QStringList::Iterator &it, QListViewItem **itemFrame, QListViewItem **subitemFrame)
{
  if (*it == "*open")
  {
    *subitemFrame = new QListViewItem( *itemFrame, *subitemFrame, *(++it) );
    QListViewItem *subSubitemFrame = new QListViewItem( *subitemFrame, *(++it) );
    while( *(++it) != "*close" )
    {
      if (*it == "*open")
      {
        subSubitemFrame = new QListViewItem( *subitemFrame, subSubitemFrame, *(++it) );
        QListViewItem *subSubSubitemFrame = new QListViewItem( subSubitemFrame, *(++it) );
        ++it;
        while (*it != "*close")
          subSubSubitemFrame = new QListViewItem( subSubitemFrame, subSubSubitemFrame, *(it++) );
      }
      else
        subSubitemFrame = new QListViewItem( *subitemFrame, subSubitemFrame, *it );
    }
  }
  else if (*it != "*close")
    *subitemFrame = new QListViewItem( *itemFrame, *subitemFrame, *it );
}


void KSnifferMainWidget::showRawPacketDetails()
{
  QString stuff, packetOut, packetRaw;

  // show the correct raw packet area: can happen that it's been selected
  // a different area area during the view
  if (m_rawPacketArea == m_rawPacketAreaRight)
  {
    m_rawPacketAreaUnder->hide();
    m_rawPacketAreaUnder->clear();
  }
  else
  {
    m_rawPacketAreaRight->hide();
    m_rawPacketAreaRight->clear();
  }

  m_rawPacketArea->show();

  ptrPacketType pkt = m_packetManager->getPacket();
  int length = m_packetManager->packetSize();
  int i = 0;
  while (i < length)
  {
    if (i % 16 == 0)
    {
      QString hexAddress = QString("%1").arg(i, 0, 16);
      stuff.fill('0', 4 - hexAddress.length());

      // 'stuff' contains the starting address in hexadecimal
      packetOut += stuff + hexAddress + "  ";
    }
    if ((pkt[i] >= ' ') && (pkt[i] <= '~'))
      packetRaw += pkt[i];
    else
      packetRaw += '.';
    QString hex = QString("%1").arg(pkt[i], 0, 16);

    // preappend '0' if the hex number is just a digit
    if (hex.length() == 1)
      packetOut += "0" + hex + " ";
    else
      packetOut += hex + " ";

    i++;

    // got the 16 bytes per line?
    if (i % 16 == 0)
    {
      packetOut += "  " + packetRaw + "\n";
      packetRaw = "";
    }
    else if (i % 8 == 0)
    {
      packetRaw += "  ";
      packetOut += " ";
    }
  }

  // align the last packetRaw line
  if (i % 16 < 8)
    stuff.fill(' ', 51 - (i % 16) * 3);
  else
    stuff.fill(' ', 50 - (i % 16) * 3);
  packetOut += stuff + packetRaw;

  // display the RAW packet in the view
  m_rawPacketArea->setText( packetOut );
}


void KSnifferMainWidget::clickedMouseButton(QListViewItem* item, const QPoint& pos, int column)
{
  QString ip;

  switch (column)
  {
    case 2:
      // source address
      ip = item->text(column);
      break;
    case 3:
      // destination address
      ip = item->text(column);
      break;
    default:
      ip = "";
      break;
  }

  // show popup menu
  if (m_info->manageIP( ip ))
    m_popupMenu->exec( pos );
}
