/* $Id: worldgui.c 1025 2009-02-11 16:56:47Z ekalin $ */

/*
 * Copyright (C) 2004-2009 Eduardo M Kalinowski <eduardo@kalinowski.com.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <kcconfig.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <libintl.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <glade/glade.h>
#ifdef __WIN32__
#  include <windows.h>
#  include <shellapi.h>
#endif

#include "kcircularqueue.h"
#include "simocombobox.h"

#include "kildclient.h"
#include "ansi.h"
#include "perlscript.h"


/******************
 * File variables *
 ******************/
static gboolean   hovering_over_link = FALSE;
static GdkCursor *hand_cursor = NULL;
static GdkCursor *regular_cursor = NULL;
#define TOOLTIP_DELAY 300


/***********************
 * Function prototypes *
 ***********************/
static void      combo_changed_cb(SimoComboBox *combo, gpointer data);
static void      combo_poped_cb(SimoComboBox *combo, gpointer data);
static void      combo_closed_cb(SimoComboBox *combo, gpointer data);
static void      n_lines_changed_cb(SimoComboBox *combo, gpointer data);
static void      activate_cb(SimoComboBox *cmb, gpointer data);
static gboolean  keypress_cb(GtkWidget   *widget,
                             GdkEventKey *evt,
                             gpointer    data);
static void      text_changed_cb(SimoComboBox *cmb, gpointer data);
static void      cursor_moved_cb(GtkTextBuffer *buffer,
                                 GtkTextIter   *iter,
                                 GtkTextMark   *mark,
                                 gpointer       data);
static gboolean  completion_function(GtkEntryCompletion *completion,
                                     const gchar        *key,
                                     GtkTreeIter        *iter,
                                     gpointer            user_data);
static gboolean  completion_match_selected_cb(GtkEntryCompletion *widget,
                                              GtkTreeModel       *model,
                                              GtkTreeIter        *iter,
                                              gpointer            data);
static void      tab_close_cb(GtkWidget *widget, gpointer data);
static gboolean  textview_button_press_cb(GtkWidget      *text_view,
                                          GdkEventButton *event,
                                          gpointer        data);
static gboolean  act_if_link(WorldGUI       *gui,
                             GtkTextIter    *iter,
                             GdkEventButton *event);
static void      menu_popup_url_copy(GtkWidget *widget, gchar *url);
static gboolean  textview_motion_notify_event(GtkWidget      *text_view,
                                              GdkEventMotion *event,
                                              gpointer        data);
static gboolean  textview_visibility_notify_event(GtkWidget         *text_view,
                                                  GdkEventVisibility *event,
                                                  gpointer            data);
static gboolean  textview_leave_notify_event(GtkWidget        *text_view,
                                             GdkEventCrossing *event,
                                             gpointer          data);
static void      set_cursor_if_appropriate(WorldGUI *gui,
                                           gint      x,
                                           gint      y);
static gboolean  worldgui_tooltip_timeout(gpointer data);
static gboolean  worldgui_paint_tooltip_cb(GtkWidget      *widget,
                                          GdkEventExpose *event,
                                          gpointer        data);
static gchar    *worldgui_get_tooltip_text(WorldGUI *gui);
static void      textview_size_allocate_cb(GtkWidget     *widget,
                                           GtkAllocation *allocation,
                                           gpointer       data);
static void      find_incremental_cb(GtkEditable *widget, gpointer data);
static void      do_find(WorldGUI *gui);
static void      close_search_box_cb(GtkButton *widget, gpointer data);
static gboolean  search_box_keypress_cb(GtkWidget   *widget,
                                        GdkEventKey *evt,
                                        gpointer     data);



WorldGUI *
world_gui_new(gboolean is_for_mud)
{
  /* is_for_mud determines the kind of GUI we're creating.
     If TRUE, it is used for really displaying MUD output, and it
     will be included in the notebook.
     If FALSE, it is a KCWin gui, and it has less features.
  */
  WorldGUI    *gui;
  GtkWidget   *iconClearWidget;
  GtkWidget   *frame;
  GtkTooltips *tooltips;

  GtkTreeViewColumn *column;
  GtkCellRenderer   *renderer;

  GList *combochildren;

  PangoFontDescription *fontDesc;
  GtkTextIter           end_iter;
  GdkColor              black = { 0, 0, 0, 0 };


  /* First, create the cursors if this is the first run */
  if (!hand_cursor) {
    hand_cursor = gdk_cursor_new (GDK_HAND2);
    regular_cursor = gdk_cursor_new (GDK_XTERM);
  }

  gui = g_new0(WorldGUI, 1);

  gui->vbox = gtk_vbox_new(FALSE, 0);
  g_object_set_data(G_OBJECT(gui->vbox), "gui", gui);

  gui->scrolled_win = GTK_SCROLLED_WINDOW(gtk_scrolled_window_new(NULL, NULL));
  gtk_scrolled_window_set_policy(gui->scrolled_win,
                                 GTK_POLICY_AUTOMATIC,
                                 GTK_POLICY_ALWAYS);
  gui->txtView = GTK_TEXT_VIEW(gtk_text_view_new());
  gtk_text_view_set_editable(gui->txtView, FALSE);
  gui->txtBuffer = gtk_text_view_get_buffer(gui->txtView);
  /* Store a mark at the end, to allow scrolling to it */
  gtk_text_buffer_get_iter_at_offset(gui->txtBuffer, &end_iter, -1);
  gui->txtmark_end = gtk_text_buffer_create_mark(gui->txtBuffer,
                                                 "end",
                                                 &end_iter,
                                                 FALSE);

  gtk_container_add(GTK_CONTAINER(gui->scrolled_win),
                    GTK_WIDGET(gui->txtView));
  gtk_box_pack_start(GTK_BOX(gui->vbox), GTK_WIDGET(gui->scrolled_win),
                     TRUE, TRUE, 0);

  /* Search box */
  if (is_for_mud) {
    GtkWidget *iconCloseWidget;
    GtkWidget *btnClose;
    GtkWidget *label;

    gui->search_box = gtk_hbox_new(FALSE, 6);
    gtk_container_set_border_width(GTK_CONTAINER(gui->search_box), 2);
    g_object_set(G_OBJECT(gui->search_box), "no-show-all", TRUE, NULL);

    btnClose = gtk_button_new();
    gtk_button_set_relief(GTK_BUTTON(btnClose), GTK_RELIEF_NONE);
    iconCloseWidget = gtk_image_new_from_stock(GTK_STOCK_CLOSE,
                                               GTK_ICON_SIZE_MENU);
    gtk_container_add(GTK_CONTAINER(btnClose), iconCloseWidget);
    g_signal_connect(G_OBJECT(btnClose), "clicked",
                     G_CALLBACK(close_search_box_cb), gui);
    gtk_box_pack_start(GTK_BOX(gui->search_box), btnClose, FALSE, FALSE, 0);

    label = gtk_label_new_with_mnemonic(_("_Find:"));
    gtk_box_pack_start(GTK_BOX(gui->search_box), label, FALSE, FALSE, 0);

    gui->txtSearchTerm = gtk_entry_new();
    gtk_label_set_mnemonic_widget(GTK_LABEL(label), gui->txtSearchTerm);
    g_signal_connect(G_OBJECT(gui->txtSearchTerm), "changed",
                     G_CALLBACK(find_incremental_cb), gui);
    g_signal_connect(G_OBJECT(gui->txtSearchTerm), "activate",
                     G_CALLBACK(menu_findnext_activate_cb), NULL);
    g_signal_connect(G_OBJECT(gui->txtSearchTerm), "key-press-event",
                     G_CALLBACK(search_box_keypress_cb), gui);
    gtk_box_pack_start(GTK_BOX(gui->search_box), gui->txtSearchTerm,
                       FALSE, FALSE, 0);

    gui->btnFindNext = gtk_button_new_with_label(_("Find Next"));
    gtk_button_set_image(GTK_BUTTON(gui->btnFindNext),
                         gtk_image_new_from_stock(GTK_STOCK_GO_DOWN,
                                                  GTK_ICON_SIZE_MENU));
    gtk_button_set_relief(GTK_BUTTON(gui->btnFindNext), GTK_RELIEF_NONE);
    g_signal_connect(G_OBJECT(gui->btnFindNext), "clicked",
                     G_CALLBACK(find_next_cb), gui);
    gtk_box_pack_start(GTK_BOX(gui->search_box), gui->btnFindNext,
                       FALSE, FALSE, 0);

    gui->lblSearchInfo = GTK_LABEL(gtk_label_new(NULL));
    gtk_box_pack_start(GTK_BOX(gui->search_box),
                       GTK_WIDGET(gui->lblSearchInfo),
                       FALSE, FALSE, 0);

    gtk_box_pack_start(GTK_BOX(gui->vbox), gui->search_box, FALSE, FALSE, 0);
  }

  /* Command area */
  gui->commandArea = gtk_hbox_new(FALSE, 4);
  gtk_container_set_border_width(GTK_CONTAINER(gui->commandArea), 2);

  gui->btnClear = gtk_button_new();
  iconClearWidget = gtk_image_new_from_stock(GTK_STOCK_CLEAR,
                                             GTK_ICON_SIZE_MENU);
  g_signal_connect(G_OBJECT(gui->btnClear), "clicked",
                   G_CALLBACK(clear_button_cb), gui);
  gtk_container_add(GTK_CONTAINER(gui->btnClear), iconClearWidget);
  gtk_box_pack_start(GTK_BOX(gui->commandArea), gui->btnClear,
                     FALSE, FALSE, 0);

  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip(tooltips, gui->btnClear,
                       _("Click to clear the command input area."),
                       _("Click this button to erase the text that is in "
                         "the command input area (at the right of this "
                         "button) so that you can start typing again."));


  if (is_for_mud) {
    /* The full-featured SimoComboBox */
    gui->cmbEntry = SIMO_COMBO_BOX(simo_combo_box_new());
    g_signal_connect(G_OBJECT(gui->cmbEntry), "size-changed",
                     G_CALLBACK(n_lines_changed_cb), gui);
    g_signal_connect(G_OBJECT(gui->cmbEntry), "clicked",
                     G_CALLBACK(combo_changed_cb), gui);
    g_signal_connect(G_OBJECT(gui->cmbEntry), "popup-displayed",
                     G_CALLBACK(combo_poped_cb), gui);
    g_signal_connect(G_OBJECT(gui->cmbEntry), "popup-closed",
                     G_CALLBACK(combo_closed_cb), gui);
    combochildren = gtk_container_get_children(GTK_CONTAINER(gui->cmbEntry));
    g_signal_connect(G_OBJECT(gui->cmbEntry), "activate",
                     G_CALLBACK(activate_cb), gui);
    g_signal_connect(G_OBJECT(gui->cmbEntry->txtview), "key-press-event",
                     G_CALLBACK(keypress_cb), gui);
    g_signal_connect(G_OBJECT(gui->cmbEntry->entry), "key-press-event",
                     G_CALLBACK(keypress_cb), gui);
    g_signal_connect(G_OBJECT(gui->cmbEntry), "changed",
                     G_CALLBACK(text_changed_cb), gui);
    g_signal_connect(G_OBJECT(gui->cmbEntry->txtbuffer), "mark-set",
                     G_CALLBACK(cursor_moved_cb), gui);
    gtk_box_pack_start(GTK_BOX(gui->commandArea), GTK_WIDGET(gui->cmbEntry),
                       TRUE, TRUE, 0);
    /* Configure renderer for combo column */
    renderer = gtk_cell_renderer_text_new();
    column   = gtk_tree_view_column_new();
    gtk_tree_view_column_pack_start(column, renderer, TRUE);
    gtk_tree_view_column_set_cell_data_func(column,
                                            renderer,
                                            (GtkTreeCellDataFunc) completion_cell_data_function,
                                            NULL,
                                            NULL);
    simo_combo_box_set_combo_column(gui->cmbEntry, column);
  } else {
    /* Just a GtkEntry */
    gui->txtEntry = GTK_ENTRY(gtk_entry_new());
    gtk_box_pack_start(GTK_BOX(gui->commandArea), GTK_WIDGET(gui->txtEntry),
                       TRUE, TRUE, 0);
  }
  gtk_box_pack_start(GTK_BOX(gui->vbox), gui->commandArea, FALSE, FALSE, 0);

  /* Keep the focus always in the entry box */
  g_signal_connect_swapped(G_OBJECT(gui->txtView), "focus-in-event",
                           G_CALLBACK(gtk_widget_grab_focus), gui->cmbEntry);


  /* Set initial colors and state */
  gui->ta.state.fg_color = gui->ta.saved_state.fg_color
    = ANSI_DEFAULT_COLOR_IDX;
  gui->ta.state.bg_color = gui->ta.saved_state.bg_color
    = ANSI_DEFAULT_COLOR_IDX;
  gui->ta.state.underline = PANGO_UNDERLINE_NONE;

  /* Create tag for text with attributes */
  gui->ta.underline_tag = gtk_text_buffer_create_tag(gui->txtBuffer,
                                                     NULL,
                                                     "underline",
                                                     PANGO_UNDERLINE_SINGLE,
                                                     NULL);
  gui->ta.dblunderline_tag = gtk_text_buffer_create_tag(gui->txtBuffer,
                                                        NULL,
                                                        "underline",
                                                        PANGO_UNDERLINE_DOUBLE,
                                                        NULL);
  gui->ta.strike_tag = gtk_text_buffer_create_tag(gui->txtBuffer,
                                                  NULL,
                                                  "strikethrough",
                                                  TRUE,
                                                  NULL);
  gui->ta.italics_tag = gtk_text_buffer_create_tag(gui->txtBuffer,
                                                   NULL,
                                                   "style",
                                                   PANGO_STYLE_ITALIC,
                                                   NULL);

  /* Initialize structures in TextAppearance */
  gui->ta.rgb_fore_tags = g_hash_table_new(NULL, NULL);
  gui->ta.rgb_back_tags = g_hash_table_new(NULL, NULL);

  /* We don't know yet which characters are supported. */
  gui->ta.sup_geom_shapes   = -1;
  gui->ta.sup_block         = -1;
  gui->ta.sup_control       = -1;
  gui->ta.sup_l1_supplement = -1;
  gui->ta.sup_box_drawing   = -1;
  gui->ta.sup_misc_tech     = -1;
  gui->ta.sup_math          = -1;
  gui->ta.sup_greek         = -1;

  if (is_for_mud) {   /* Special features */
    GtkWidget *btnClose;
    GtkRcStyle *rcstyle;
    GtkWidget *iconCloseWidget;

    /* Notebook tab */
    gui->hboxTab = GTK_BOX(gtk_hbox_new(FALSE, 4));
    gui->lblNotebook = GTK_LABEL(gtk_label_new(_("No world")));
    gtk_box_pack_start(gui->hboxTab, GTK_WIDGET(gui->lblNotebook),
                       FALSE, FALSE, 0);

    btnClose = gtk_button_new();
    /* Make the button smaller. (Taken from gedit code) */
    gtk_button_set_focus_on_click(GTK_BUTTON(btnClose), FALSE);
    gtk_button_set_relief(GTK_BUTTON(btnClose), GTK_RELIEF_NONE);
    rcstyle  = gtk_rc_style_new();
    rcstyle->xthickness = rcstyle->ythickness = 0;
    gtk_widget_modify_style(btnClose, rcstyle);
    gtk_rc_style_unref(rcstyle);

    iconCloseWidget = gtk_image_new_from_stock(GTK_STOCK_CLOSE,
                                               GTK_ICON_SIZE_MENU);
    g_signal_connect(G_OBJECT(btnClose), "clicked",
                     G_CALLBACK(tab_close_cb), gui);
    gtk_container_add(GTK_CONTAINER(btnClose), iconCloseWidget);
    gtk_box_pack_start(GTK_BOX(gui->hboxTab), btnClose,
                       FALSE, FALSE, 0);

    /* The color and font will be changed later to the colors and font
       of the opened world, but we set the background to black so that is
       appears black while the world is being selected, and the font to a
       default font so that the size can be calculated correctly. */
    fontDesc = pango_font_description_from_string(DEFAULT_TERMINAL_FONT);
    gtk_widget_modify_font(GTK_WIDGET(gui->txtView), fontDesc);
    pango_font_description_free(fontDesc);
    gtk_widget_modify_base(GTK_WIDGET(gui->txtView),
                           GTK_STATE_NORMAL,
                           &black);

    gtk_widget_add_events(GTK_WIDGET(gui->txtView), GDK_LEAVE_NOTIFY_MASK);

    /* Detect resizes */
    g_signal_connect(G_OBJECT(gui->txtView), "size-allocate",
                     G_CALLBACK(textview_size_allocate_cb), gui);

    /* Create a mark to hold the start of the current line */
    gui->txtmark_linestart = gtk_text_buffer_create_mark(gui->txtBuffer,
                                                         "linestart",
                                                         &end_iter,
                                                         TRUE);
    /* Create tag for URLs */
    gui->txttag_url = gtk_text_buffer_create_tag(gui->txtBuffer,
                                                 NULL,
                                                 "underline",
                                                 PANGO_UNDERLINE_SINGLE,
                                                 NULL);
    /* URLs support */
    g_signal_connect(gui->txtView, "button-press-event",
                     G_CALLBACK(textview_button_press_cb), gui);
    g_signal_connect(gui->txtView, "motion-notify-event",
                     G_CALLBACK(textview_motion_notify_event), gui);
    g_signal_connect(gui->txtView, "visibility-notify-event",
                     G_CALLBACK(textview_visibility_notify_event), gui);

    /* Tool tips */
    /* We do not connect motion-notify because we already have a callback
       used in the URL support stuff. */
    g_signal_connect(gui->txtView, "leave-notify-event",
                     G_CALLBACK(textview_leave_notify_event), gui);

    /* Status bar */
    gui->statusbar_box = GTK_BOX(gtk_hbox_new(FALSE, 4));

    gui->lblStatus = GTK_LABEL(gtk_label_new(_("KildClient ready")));
    gtk_misc_set_alignment(GTK_MISC(gui->lblStatus), 0, 0.5);
    gtk_label_set_single_line_mode(gui->lblStatus, TRUE);
    gtk_label_set_ellipsize(gui->lblStatus, PANGO_ELLIPSIZE_END);
    gtk_box_pack_start(gui->statusbar_box, GTK_WIDGET(gui->lblStatus),
                       TRUE, TRUE, 0);

    gui->lblLines = GTK_LABEL(gtk_label_new(_("0 lines")));
    gtk_misc_set_alignment(GTK_MISC(gui->lblLines), 0, 0.5);
    gtk_label_set_single_line_mode(gui->lblLines, TRUE);
    frame = gtk_frame_new(NULL);
    gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
    gtk_container_add(GTK_CONTAINER(frame), GTK_WIDGET(gui->lblLines));
    gtk_box_pack_start(gui->statusbar_box, frame,
                       FALSE, FALSE, 0);

    gui->lblTime = GTK_LABEL(gtk_label_new(NULL));
    gtk_misc_set_alignment(GTK_MISC(gui->lblTime), 0, 0.5);
    gtk_label_set_single_line_mode(gui->lblTime, TRUE);
    frame = gtk_frame_new(NULL);
    gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
    gtk_container_add(GTK_CONTAINER(frame), GTK_WIDGET(gui->lblTime));
    gtk_box_pack_start(gui->statusbar_box, frame,
                       FALSE, FALSE, 0);

    gtk_box_pack_start(GTK_BOX(gui->vbox), GTK_WIDGET(gui->statusbar_box),
                       FALSE, FALSE, 0);

    /* Clear lines at the beginning of the buffer to keep it to a
       reasonable size. */
    gui->prune_timeout_id = g_timeout_add(1000,
                                          ansitextview_prune_extra_lines,
                                          gui);
  }

  return gui;
}


void
world_gui_size_textview(WorldGUI *gui)
{
  gint char_height;
  gint char_width;

  ansitextview_get_char_size(gui, &char_height, &char_width);
  gtk_widget_set_size_request(GTK_WIDGET(gui->txtView),
                              80 * char_width,
                              24 * char_height);
}


void
worldgui_determine_supported_chars(WorldGUI *gui)
{
  PangoContext         *context;
  PangoFontDescription *fontdesc;
  PangoFont            *font;
  PangoCoverage        *coverage;

  if (gui->ta.sup_geom_shapes != -1) {
    return;     /* There are cached values */
  }

  context = gtk_widget_get_pango_context(GTK_WIDGET(gui->txtView));
  fontdesc = pango_font_description_from_string(gui->world->terminalfont);
  font = pango_context_load_font(context, fontdesc);
  coverage = pango_font_get_coverage(font,
                                     pango_context_get_language(context));

  /* U+25C6 BLACK DIAMOND */
  gui->ta.sup_geom_shapes   = pango_coverage_get(coverage, 0x25C6);
  /* U+2592 MEDIUM SHADE */
  gui->ta.sup_block         = pango_coverage_get(coverage, 0x2592);
  /* U+2409 SYMBOL FOR HORIZONTAL TABULATION */
  gui->ta.sup_control       = pango_coverage_get(coverage, 0x2409);
  /* U+00B0 DEGREE SIGN */
  gui->ta.sup_l1_supplement = pango_coverage_get(coverage, 0x00B0);
  /* U+2518 BOX DRAWINGS LIGHT UP AND LEFT */
  gui->ta.sup_box_drawing   = pango_coverage_get(coverage, 0x2518);
  /* U+23BA HORIZONTAL SCAN LINE-1 */
  gui->ta.sup_misc_tech     = pango_coverage_get(coverage, 0x23BA);
  /* U+2264 LESS-THAN OR EQUAL TO */
  gui->ta.sup_math          = pango_coverage_get(coverage, 0x2264);
  /* U+03C0 GREEK SMALL LETTER PI */
  gui->ta.sup_greek         = pango_coverage_get(coverage, 0x03C0);

  pango_font_description_free(fontdesc);
  pango_coverage_unref(coverage);
}


void
free_world_gui(WorldGUI *gui)
{
  if (gui->prune_timeout_id) {
    g_source_remove(gui->prune_timeout_id);
  }

  /* The tags themselves are supposed to be destroyed when the TextBuffer
     is destroyed, which happens when the widget is destroyed. */
  g_free(gui->ta.ansi_fore_tags);
  g_free(gui->ta.ansi_back_tags);
  g_hash_table_destroy(gui->ta.rgb_fore_tags);
  g_hash_table_destroy(gui->ta.rgb_back_tags);

  if (gui->line_times) {
    k_circular_queue_free(gui->line_times);
  }
  worldgui_destroy_tooltip_window(gui);
  if (gui->tooltip_timeout_id) {
    g_source_remove(gui->tooltip_timeout_id);
  }

  g_free(gui);
}


static
void
combo_changed_cb(SimoComboBox *combo, gpointer data)
{
  GtkTreeIter  current;
  World       *world;
  gchar       *command;

  world = ((WorldGUI *) data)->world;

  if (simo_combo_box_get_active_iter(combo, &current)) {
    world->current_cmd = current;
    command = cmdhistory_get_command(world->cmd_list, &current);
    simo_combo_box_set_text(combo, command);
    world->cmd_just_selected_from_combo = TRUE;
    g_free(command);
  }
}


static
void
combo_poped_cb(SimoComboBox *combo, gpointer data)
{
  World *world = ((WorldGUI *) data)->world;

  gtk_list_store_set(GTK_LIST_STORE(world->cmd_list), &world->current_cmd,
                     CMDLIST_COL_TENTATIVE,
                     simo_combo_box_get_text(combo),
                     -1);
  world->combo_is_poped = TRUE;
}


static
void
combo_closed_cb(SimoComboBox *combo, gpointer data)
{
  World *world = ((WorldGUI *) data)->world;

  world->combo_is_poped = FALSE;
}


static
void
n_lines_changed_cb(SimoComboBox *combo, gpointer data)
{
  World *world = ((WorldGUI *) data)->world;

  world->input_n_lines = simo_combo_box_get_n_lines(combo);
  set_input_line_controls(world, NULL, NULL, NULL);
}


static
void
activate_cb(SimoComboBox *cmb, gpointer data)
{
  World *world = ((WorldGUI *) data)->world;

  if (world) {
    const gchar *str;

    str = simo_combo_box_get_text(cmb);
    parse_commands(world, str, strlen(str));

    /* Ignore empty commands and passwords */
    if (strcmp(str, "") != 0 && !world->noecho) {
      add_recent_command(world, str);
    }
  }
  if (world->noecho || !world->repeat_commands)
    simo_combo_box_set_text(cmb, "");
  else
    simo_combo_box_select_region(cmb, 0, -1);

  if (world->itime_reset_activate) {
    world->last_command_time = time(NULL);
  }
  world->cmd_position_changed = FALSE;
}


static
gboolean
keypress_cb(GtkWidget   *widget,
            GdkEventKey *evt,
            gpointer     data)
{
  WorldGUI      *gui = (WorldGUI *) data;
  World         *world = gui->world;
  GtkAdjustment *adjustment;
  int            direction = 1;
  gdouble        newval;

  /* By personal experience, tooltips are only useful when idle. When the
     user is typing, they probably do not want them. */
  worldgui_destroy_tooltip_window(gui);
  if (gui->tooltip_timeout_id) {
    g_source_remove(gui->tooltip_timeout_id);
  }

  if (world->print_next_keycode) {
    if (gtk_accelerator_valid(evt->keyval, evt->state)) {
      world->keycode_to_print = gtk_accelerator_name(evt->keyval, evt->state);
      ansitextview_append_stringf(world->gui,
                                  "Key code: %s\n",
                                  world->keycode_to_print);


      world->print_next_keycode = FALSE;
      gtk_main_quit();
    }

    return TRUE;
  }

  switch (evt->keyval) {
  case GDK_Left:
  case GDK_KP_Left:
    if (evt->state & GDK_MOD1_MASK) { /* Alt+left */
      menu_perl_run_cb(NULL, "$world->prev");
      return TRUE;
    }
    break;

  case GDK_Right:
  case GDK_KP_Right:
    if (evt->state & GDK_MOD1_MASK) { /* Alt+right */
      menu_perl_run_cb(NULL, "$world->next");
      return TRUE;
    }
    break;

  case GDK_Tab:
  case GDK_ISO_Left_Tab:
    if (evt->state & GDK_CONTROL_MASK) {
      if (evt->state & GDK_SHIFT_MASK) { /* Ctrl+Shift+Tab */
        menu_perl_run_cb(NULL, "$world->prev");
      } else {                           /* Ctrl+Tab */
        menu_perl_run_cb(NULL, "$world->next");
      }
      return TRUE;
    }
    break;

  case '1': case '2': case '3': case '4': case '5':
  case '6': case '7': case '8': case '9':
    if (evt->state & GDK_MOD1_MASK) { /* Alt+digit */
      int newpage = evt->keyval - '1';

      if (newpage < g_list_length(open_worlds)) {
        set_focused_world(newpage);
      }
      return TRUE;
    }
    break;

  case GDK_Up:
  case GDK_KP_Up:
    direction = 0;

  case GDK_Down:
  case GDK_KP_Down:
    if (world->input_n_lines > 1 && world->cmd_position_changed) {
      return FALSE;
    }
    prev_or_next_command(world, direction);
    return TRUE;

  case GDK_Page_Up:
  case GDK_KP_Page_Up:
    direction = -1;
    /* Fall through */

  case GDK_Page_Down:
  case GDK_KP_Page_Down:
    adjustment = gtk_scrolled_window_get_vadjustment(gui->scrolled_win);
    newval = gtk_adjustment_get_value(adjustment)
      + direction*adjustment->page_increment/2;
    if (newval > (adjustment->upper - adjustment->page_size)) {
      newval = (adjustment->upper - adjustment->page_size);
    }
    gtk_adjustment_set_value(adjustment, newval);
    gtk_adjustment_changed(adjustment);
    return TRUE;

  case GDK_End:
  case GDK_KP_End:
    if (evt->state & GDK_CONTROL_MASK) {
      adjustment = gtk_scrolled_window_get_vadjustment(gui->scrolled_win);
      gtk_adjustment_set_value(adjustment,
                               adjustment->upper - adjustment->page_size);
      gtk_adjustment_changed(adjustment);
      return TRUE;
    }
    break;
  }

  return check_macros(world, evt->keyval, evt->state);
}


static
void
text_changed_cb(SimoComboBox *cmb, gpointer data)
{
  World *world = ((WorldGUI *) data)->world;
  const gchar *command;

  if (world->gui->execute_changed_signal && !world->combo_is_poped) {
    if (!world->cmd_just_selected_from_combo) {
      command = simo_combo_box_get_text(cmb);
      insert_recent_command(world, command);
      /* Set the pointer to point to the new command */
      gtk_tree_model_get_iter_first(world->cmd_list, &world->current_cmd);
    } else {
      insert_recent_command(world, "");
    }

    world->gui->execute_changed_signal = FALSE;
    world->cmd_has_been_edited = TRUE;
    world->cmd_just_selected_from_combo = FALSE;
    if (world->saved_command_find_search) {
      g_free(world->saved_command_find_search);
      world->saved_command_find_search = NULL;
    }
  }
}


static
void
cursor_moved_cb(GtkTextBuffer *buffer,
                GtkTextIter   *iter,
                GtkTextMark   *mark,
                gpointer       data)
{
  WorldGUI    *gui = (WorldGUI *) data;
  const gchar *name;

  name = gtk_text_mark_get_name(mark);
  if (!name || strcmp(name, "insert") != 0) {
    return;
  }

  if (gui->world) {
    gui->world->cmd_position_changed = TRUE;
  }
}


void
completion_cell_data_function(GtkCellLayout   *cell_layout,
                              GtkCellRenderer *renderer,
                              GtkTreeModel    *model,
                              GtkTreeIter     *iter,
                              gpointer         user_data)
{
  /* Used in the combo box and completion display. If there is a tentative
     command, displays it, otherwise displays the normal text. */
  gchar *command;

  command = cmdhistory_get_command(model, iter);
  g_object_set(renderer, "text", command, NULL);
  g_free(command);
}


static
gboolean
completion_function(GtkEntryCompletion *completion,
                    const gchar        *key,
                    GtkTreeIter        *iter,
                    gpointer            user_data)
{
  GtkTreeModel *model;
  gchar        *command;
  gchar        *normalized_command;
  gchar        *case_normalized_command;
  gboolean      ret = FALSE;

  model = (GtkTreeModel *) user_data;

  command = cmdhistory_get_command(model, iter);

  if (!command) {
    return FALSE;
  }

  normalized_command      = g_utf8_normalize(command, -1, G_NORMALIZE_ALL);
  case_normalized_command = g_utf8_casefold(normalized_command, -1);

  if (strncmp(key, case_normalized_command, strlen(key)) == 0) {
    ret = TRUE;
  }

  g_free(case_normalized_command);
  g_free(normalized_command);
  g_free(command);

  return ret;
}


static
gboolean
completion_match_selected_cb(GtkEntryCompletion *widget,
                             GtkTreeModel       *model,
                             GtkTreeIter        *iter,
                             gpointer            user_data)
{
  World *world = (World *) user_data;
  gchar *command;

  command = cmdhistory_get_command(model, iter);
  simo_combo_box_set_text(world->gui->cmbEntry, command);
  simo_combo_box_set_position(world->gui->cmbEntry, -1);
  g_free(command);

  return TRUE;
}


static
void
tab_close_cb(GtkWidget *widget, gpointer data)
{
  WorldGUI *gui   = (WorldGUI *) data;
  World    *world = gui->world;

  if (world->connected) {
    GtkWidget *msgdlg;
    GtkWidget *dlgbutton;
    GtkWidget *dlgbuttonimage;
    gint       response;

    msgdlg = gtk_message_dialog_new(GTK_WINDOW(wndMain),
                                    GTK_DIALOG_MODAL,
                                    GTK_MESSAGE_QUESTION,
                                    GTK_BUTTONS_NONE,
                                    _("Are you sure you want to close this world?"));
    gtk_window_set_title(GTK_WINDOW(msgdlg), _("Really close?"));
    /* Keep Open is RESPONSE_NO because of NO close. But we use
       STOCK_YES to pass a positive image. */
    dlgbutton = gtk_button_new_with_mnemonic(_("Keep _open"));
    dlgbuttonimage = gtk_image_new_from_stock(GTK_STOCK_YES,
                                              GTK_ICON_SIZE_BUTTON);
    gtk_button_set_image(GTK_BUTTON(dlgbutton), dlgbuttonimage);
    gtk_dialog_add_action_widget(GTK_DIALOG(msgdlg),
                                 dlgbutton, GTK_RESPONSE_NO);

    /*    dlgbutton = gtk_button_new_with_label(_("Close"));
    dlgbuttonimage = gtk_image_new_from_stock(GTK_STOCK_CLOSE,
                                              GTK_ICON_SIZE_BUTTON);
    gtk_button_set_image(GTK_BUTTON(dlgbutton), dlgbuttonimage);
    gtk_widget_show_all(dlgbutton);*/
    dlgbutton = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
    gtk_dialog_add_action_widget(GTK_DIALOG(msgdlg),
                                 dlgbutton, GTK_RESPONSE_YES);

    gtk_widget_show_all(msgdlg);
    response = gtk_dialog_run(GTK_DIALOG(msgdlg));
    gtk_widget_destroy(msgdlg);

    if (response != GTK_RESPONSE_YES) {
      return;
    }
  }

  remove_world(world, TRUE);
}



void
clear_button_cb(GtkButton *button, gpointer data)
{
  gchar    *last_path;
  WorldGUI *gui = (WorldGUI *) data;
  World    *world = gui->world;

  simo_combo_box_clear_text(gui->cmbEntry);
  gtk_widget_grab_focus(GTK_WIDGET(gui->cmbEntry));

  if (world) {
    /* So that up key recalls the previous command */
    last_path = g_strdup_printf("%d", world->cmd_list_size - 1);
    gtk_tree_model_get_iter_from_string(world->cmd_list,
                                        &world->current_cmd,
                                        last_path);
  }
}


void
configure_gui(WorldGUI *gui, World *world)
{
  PangoFontDescription *fontDesc;
  GtkEntryCompletion   *completion;
  GtkCellRenderer      *renderer;

  fontDesc = pango_font_description_from_string(world->terminalfont);
  gtk_widget_modify_font(GTK_WIDGET(gui->txtView), fontDesc);
  pango_font_description_free(fontDesc);
  fontDesc = pango_font_description_from_string(world->entryfont);
  if (gui->world) {
    simo_combo_box_set_entry_font(gui->cmbEntry, fontDesc);
  } else {
    gtk_widget_modify_font(GTK_WIDGET(gui->txtEntry), fontDesc);
  }
  pango_font_description_free(fontDesc);
  if (gui->lblStatus) {
    fontDesc = pango_font_description_from_string(world->statusfont);
    gtk_widget_modify_font(GTK_WIDGET(gui->lblStatus), fontDesc);
    pango_font_description_free(fontDesc);
  }

  gtk_text_view_set_wrap_mode(gui->txtView,
                              world->wrap
                                ? GTK_WRAP_WORD_CHAR
                                : GTK_WRAP_CHAR);


  /* If this GUI is for a MUD, and not for a KCWin */
  if (gui->world) {
    simo_combo_box_set_model(gui->cmbEntry, world->cmd_list);
    completion = simo_combo_box_get_completion(gui->cmbEntry);
    if (world->autocompletion) {
      gtk_entry_completion_set_minimum_key_length(completion,
                                                  world->autocompl_minprefix);
    } else {
      gtk_entry_completion_set_minimum_key_length(completion,
                                                  9999);
    }
    /* Configure rendererer for completion */
    renderer = gtk_cell_renderer_text_new();
    simo_combo_box_set_completion_renderer(gui->cmbEntry, renderer);
    simo_combo_box_set_completion_cell_func(gui->cmbEntry,
                                            renderer,
                                            completion_cell_data_function,
                                            NULL,
                                            NULL);
    gtk_entry_completion_set_match_func(completion,
                                        completion_function,
                                        world->cmd_list,
                                        NULL);
    /* Correct behaviour when a completion is selected */
    g_signal_connect(G_OBJECT(completion), "match-selected",
                     G_CALLBACK(completion_match_selected_cb), world);

    /* Set up structure to hold time of each line's arrival */
    gui->line_times = k_circular_queue_new(sizeof(time_t),
                                           world->buffer_lines);

    /* Size of input box */
    simo_combo_box_set_n_lines(gui->cmbEntry, world->input_n_lines);

    /* Spell checking. */
    worldgui_configure_spell(gui);
  }
}


void
worldgui_configure_spell(WorldGUI *gui)
{
  GError *error = NULL;

#ifdef HAVE_GTKSPELL
  simo_combo_box_set_spell(gui->cmbEntry,
                           gui->world->spell, gui->world->spell_language,
                           &error);
  if (error) {
    ansitextview_append_stringf(gui,_("Error setting spell checker: %s\n"), error->message);
    g_error_free(error);
  }
#else
  ansitextview_append_string_nl(gui, _("Spell checking support not included in this build."));
#endif
}


static
gboolean
textview_button_press_cb(GtkWidget      *text_view,
                         GdkEventButton *event,
                         gpointer        data)
{
  WorldGUI       *gui;
  GtkTextIter     iter;
  gint            x, y;

  gui = (WorldGUI *) data;

  if (event->button != 1 && event->button != 3) {
    return FALSE;
  }

  gtk_text_view_window_to_buffer_coords(GTK_TEXT_VIEW(text_view),
                                        GTK_TEXT_WINDOW_WIDGET,
                                        event->x, event->y, &x, &y);

  gtk_text_view_get_iter_at_location(GTK_TEXT_VIEW(text_view), &iter, x, y);

  return act_if_link(gui, &iter, event);
}


static
gboolean
act_if_link(WorldGUI *gui, GtkTextIter *iter, GdkEventButton *event)
{
  GtkTextIter  start;
  GtkTextIter  end;
  gchar       *matched_url;
  GladeXML    *gladexml;
  GtkWidget   *mnuPopupURL = NULL;
  GtkWidget   *mnuURLOpen;
  GtkWidget   *mnuURLCopy;

  if (!gtk_text_iter_has_tag(iter, gui->txttag_url)) {
    return FALSE;
  }

  start = end = *iter;
  if (!gtk_text_iter_backward_to_tag_toggle(&start, gui->txttag_url)) {
    /* Not found must mean we are at the tag boundary */
    start = *iter;
  }
  if (!gtk_text_iter_forward_to_tag_toggle(&end, gui->txttag_url)) {
    /* Not found must mean we are at the tag boundary */
    end = *iter;
  }
  matched_url = gtk_text_buffer_get_text(gui->txtBuffer, &start, &end, FALSE);

  if (event->button == 1) {
    menu_url_open(NULL, matched_url);
  } else {
    gladexml = glade_xml_new(get_kildclient_installed_file("kildclient.glade"),
                             "mnuPopupURL", NULL);
    mnuPopupURL = glade_xml_get_widget(gladexml, "mnuPopupURL");
    mnuURLOpen  = glade_xml_get_widget(gladexml, "mnuURLOpen");
    mnuURLCopy  = glade_xml_get_widget(gladexml, "mnuURLCopy");

    g_signal_connect(G_OBJECT(mnuURLOpen), "activate",
                     G_CALLBACK(menu_url_open), matched_url);
    g_signal_connect(G_OBJECT(mnuURLCopy), "activate",
                     G_CALLBACK(menu_popup_url_copy), matched_url);

    g_object_unref(gladexml);
    gtk_menu_popup(GTK_MENU(mnuPopupURL), NULL, NULL, NULL, NULL,
                   event->button, event->time);
  }

  return TRUE;
}


#ifndef __WIN32__
void
menu_url_open(GtkWidget *widget, char *url)
{
  char *to_run;

  to_run = g_strdup_printf(globalPrefs.browser_command, url);

  system(to_run);

  g_free(to_run);
  g_free(url);
}
#else /* defined __WIN32__ */
void
menu_url_open(GtkWidget *widget, char *url)
{
  ShellExecute(NULL, "open", url, NULL, NULL, SW_SHOWNORMAL);

  g_free(url);
}
#endif


static
void
menu_popup_url_copy(GtkWidget *widget, char *url)
{
  GdkDisplay   *display;
  GtkClipboard *clipboard;

  display = gtk_widget_get_display(GTK_WIDGET(widget));

  clipboard = gtk_clipboard_get_for_display(display, GDK_SELECTION_PRIMARY);
  gtk_clipboard_set_text(clipboard, url, -1);

  clipboard = gtk_clipboard_get_for_display(display, GDK_SELECTION_CLIPBOARD);
  gtk_clipboard_set_text(clipboard, url, -1);

  g_free(url);
}


static
gboolean
textview_motion_notify_event(GtkWidget      *text_view,
                             GdkEventMotion *event,
                             gpointer        data)
{
  WorldGUI    *gui = (WorldGUI *) data;
  gint         x, y;
  GtkTextIter  iter;

  /*
   * Update the cursor image if the pointer moved.
   */
  gtk_text_view_window_to_buffer_coords(gui->txtView,
                                        GTK_TEXT_WINDOW_WIDGET,
                                        event->x, event->y, &x, &y);

  set_cursor_if_appropriate(gui, x, y);

  /* I don't know why this call is here, but I copied this from the
     gtk-demo, and it is necessary. */
  gdk_window_get_pointer(text_view->window, NULL, NULL, NULL);


  /*
   * Tooltip
   */
  gui->tooltip_x = x;
  gui->tooltip_y = y;
  if (gui->tooltip_timeout_id) {
    if ((y > gui->tooltip_line_y)
        && ((y - gui->tooltip_line_height) < gui->tooltip_line_y)) {
      return FALSE;
    }
    /* We have left the line. Remove timeout, a new one will be created
       later. */
    worldgui_destroy_tooltip_window(gui);
    g_source_remove(gui->tooltip_timeout_id);
  }

  gtk_text_view_get_line_at_y(gui->txtView, &iter, y, &gui->tooltip_line_y);
  gtk_text_view_get_line_yrange(gui->txtView, &iter,
                                &gui->tooltip_line_y,
                                &gui->tooltip_line_height);
  gui->tooltip_line_number = gtk_text_iter_get_line(&iter);

  gui->tooltip_timeout_id = g_timeout_add(TOOLTIP_DELAY,
                                          worldgui_tooltip_timeout, gui);

  return FALSE;
}


/* Also update the cursor image if the window becomes visible
 * (e.g. when a window covering it got iconified).
 */
static
gboolean
textview_visibility_notify_event(GtkWidget          *text_view,
                                 GdkEventVisibility *event,
                                 gpointer            data)
{
  /* Also update the cursor image if the window becomes visible
   * (e.g. when a window covering it got iconified). */
  WorldGUI *gui = (WorldGUI *) data;
  gint      wx, wy, bx, by;

  gdk_window_get_pointer(GTK_WIDGET(gui->txtView)->window, &wx, &wy, NULL);

  gtk_text_view_window_to_buffer_coords(gui->txtView,
                                        GTK_TEXT_WINDOW_WIDGET,
                                        wx, wy, &bx, &by);

  set_cursor_if_appropriate(gui, bx, by);

  return FALSE;
}


static
void
set_cursor_if_appropriate(WorldGUI *gui,
                          gint      x,
                          gint      y)
{
  GtkTextIter    iter;
  gboolean       hovering = FALSE;

  gtk_text_view_get_iter_at_location(gui->txtView, &iter, x, y);
  hovering = gtk_text_iter_has_tag(&iter, gui->txttag_url);

  if (hovering != hovering_over_link) {
    hovering_over_link = hovering;

    if (hovering_over_link) {
      gdk_window_set_cursor(gtk_text_view_get_window(gui->txtView,
                                                     GTK_TEXT_WINDOW_TEXT),
                            hand_cursor);
    } else {
      gdk_window_set_cursor(gtk_text_view_get_window(gui->txtView,
                                                     GTK_TEXT_WINDOW_TEXT),
                            regular_cursor);
    }
  }
}


static
gboolean
textview_leave_notify_event(GtkWidget        *text_view,
                            GdkEventCrossing *event,
                            gpointer          data)
{
  WorldGUI *gui = (WorldGUI *) data;

  if (gui->tooltip_timeout_id) {
    g_source_remove(gui->tooltip_timeout_id);
    gui->tooltip_timeout_id = 0;
  }

  worldgui_destroy_tooltip_window(gui);

  return FALSE;
}


static
gboolean
worldgui_tooltip_timeout(gpointer data)
{
  WorldGUI    *gui = (WorldGUI *) data;
  gchar       *linetime;
  PangoLayout *layout;
  gint         scr_w, scr_h, w, h, x, y;
  int          mon_num;
  GdkScreen   *screen = NULL;
  GdkRectangle mon_size;
  gboolean     tooltip_top = FALSE;

  linetime = worldgui_get_tooltip_text(gui);

  if (gui->tooltip_window) {
    gtk_widget_destroy(gui->tooltip_window);
  }

  gui->tooltip_window = gtk_window_new(GTK_WINDOW_POPUP);
  gtk_widget_set_app_paintable(gui->tooltip_window, TRUE);
  gtk_window_set_resizable(GTK_WINDOW(gui->tooltip_window), FALSE);
  gtk_widget_set_name(gui->tooltip_window, "gtk-tooltips");
  g_signal_connect(G_OBJECT(gui->tooltip_window), "expose_event",
                   G_CALLBACK(worldgui_paint_tooltip_cb), gui);
  gtk_widget_ensure_style(gui->tooltip_window);

  layout = gtk_widget_create_pango_layout(gui->tooltip_window, NULL);
  pango_layout_set_wrap(layout, PANGO_WRAP_WORD);
  pango_layout_set_width(layout, 300000);
  pango_layout_set_text(layout, linetime, -1);
  pango_layout_get_size(layout, &w, &h);

  /* Code to determine size and copied from Gaim (just like the rest
     of this tooltip code, by the way, but this is more mystical and
     wasn't touched). */
  gdk_display_get_pointer(gdk_display_get_default(), &screen, &x, &y, NULL);
  mon_num = gdk_screen_get_monitor_at_point(screen, x, y);
  gdk_screen_get_monitor_geometry(screen, mon_num, &mon_size);

  scr_w = mon_size.width + mon_size.x;
  scr_h = mon_size.height + mon_size.y;

  w = PANGO_PIXELS(w) + 8;
  h = PANGO_PIXELS(h) + 8;

  if (w > mon_size.width)
    w = mon_size.width - 10;

  if (h > mon_size.height)
    h = mon_size.height - 10;

  //if (GTK_WIDGET_NO_WINDOW(gtkblist->window))
  //  y += gtkblist->window->allocation.y;

  x -= ((w >> 1) + 4);

  if ((y + h + 4) > scr_h || tooltip_top)
    y = y - h - 5;
  else
    y = y + 6;

  if (y < mon_size.y)
    y = mon_size.y;

  if (y != mon_size.y) {
    if ((x + w) > scr_w)
      x -= (x + w + 5) - scr_w;
    else if (x < mon_size.x)
      x = mon_size.x;
  } else {
    x -= (w / 2 + 10);
    if (x < mon_size.x)
      x = mon_size.x;
  }

  g_free(linetime);
  g_object_unref(layout);
  gtk_widget_set_size_request(gui->tooltip_window, w, h);
  gtk_window_move(GTK_WINDOW(gui->tooltip_window), x, y);
  gtk_widget_show(gui->tooltip_window);

  return FALSE;
}


static
gboolean
worldgui_paint_tooltip_cb(GtkWidget      *widget,
                          GdkEventExpose *event,
                          gpointer        data)
{
  WorldGUI    *gui = (WorldGUI *) data;
  gchar       *linetime;
  PangoLayout *layout;
  GtkStyle    *style;

  linetime = worldgui_get_tooltip_text(gui);

  layout = gtk_widget_create_pango_layout(gui->tooltip_window, NULL);
  pango_layout_set_wrap(layout, PANGO_WRAP_WORD);
  pango_layout_set_width(layout, 300000);
  pango_layout_set_text(layout, linetime, -1);
  style = gui->tooltip_window->style;

  gtk_paint_flat_box(style,
                     gui->tooltip_window->window,
                     GTK_STATE_NORMAL, GTK_SHADOW_OUT,
                     NULL, gui->tooltip_window, "tooltip",
                     0, 0, -1, -1);

  gtk_paint_layout(style,
                   gui->tooltip_window->window,
                   GTK_STATE_NORMAL, FALSE,
                   NULL, gui->tooltip_window, "tooltip",
                   4, 4, layout);

  g_free(linetime);
  g_object_unref(layout);

  return TRUE;
}


static
gchar *
worldgui_get_tooltip_text(WorldGUI *gui)
{
  GtkTextIter  iter;
  gchar       *tip;

  gtk_text_view_get_iter_at_location(GTK_TEXT_VIEW(gui->txtView), &iter,
                                     gui->tooltip_x, gui->tooltip_y);

  if (gtk_text_iter_has_tag(&iter, gui->txttag_url)) {
    tip = g_strdup(_("Click to open link; right-click for more options."));
  } else {
    size_t    len;
#ifdef HAVE_LOCALTIME_R
    struct tm time_tm;
#endif
    struct tm *time_tmp;

    tip = g_malloc(MAX_BUFFER * sizeof(gchar));

    g_snprintf(tip, MAX_BUFFER,
               "Line %ld, ",
               gui->tooltip_line_number + gui->world->deleted_lines + 1);
    len = strlen(tip);

#ifdef HAVE_LOCALTIME_R
    localtime_r(&k_circular_queue_nth(gui->line_times,
                                      time_t,
                                      gui->tooltip_line_number),
                &time_tm);
    time_tmp = &time_tm;
#else
    time_tmp = localtime(&k_circular_queue_nth(gui->line_times,
                                               time_t,
                                               gui->tooltip_line_number));
#endif

    strftime(tip + len, MAX_BUFFER - len, "%c", time_tmp);
  }

  return tip;
}


void
worldgui_destroy_tooltip_window(WorldGUI *gui)
{
  if (gui->tooltip_window == NULL) {
    return;
  }

  gtk_widget_destroy(gui->tooltip_window);
  gui->tooltip_window = NULL;
}


static
void
textview_size_allocate_cb(GtkWidget     *widget,
                          GtkAllocation *allocation,
                          gpointer       data)
{
  WorldGUI *gui   = (WorldGUI *) data;
  World    *world = gui->world;

  /* Some checks to prevent send_naws_size() from being called when
     world is invalid. */
  if (world && world->gui && world->gui == gui
      && world->connected && world->use_naws == TRUE) {
    GdkRectangle rect;

    /* See if the size has changed. */
    gtk_text_view_get_visible_rect(gui->txtView, &rect);
    if (rect.height != world->last_naws_size.height
        || rect.width != world->last_naws_size.width) {
      send_naws_size(world);
      world->last_naws_size = rect;
    }
  }
}


static
void
find_incremental_cb(GtkEditable *widget, gpointer data)
{
  WorldGUI *gui = (WorldGUI *) data;

  do_find(gui);
}


void
find_next_cb(GtkButton *button, gpointer data)
{
  WorldGUI    *gui = (WorldGUI *) data;
  GtkTextIter  search_start;

  gtk_text_buffer_get_iter_at_mark(gui->txtBuffer,
                                   &search_start,
                                   gui->txtmark_next_search_start);
  gtk_text_buffer_move_mark(gui->txtBuffer,
                            gui->txtmark_search_start,
                            &search_start);
  do_find(gui);
}


static
void
do_find(WorldGUI *gui)
{
  const gchar *text;
  GtkTextIter  search_start;
  GtkTextIter  match_start;
  GtkTextIter  match_end;

  gtk_label_set_text(gui->lblSearchInfo, NULL);

  text = gtk_entry_get_text(GTK_ENTRY(gui->txtSearchTerm));
  if (strcmp(text, "") == 0) {
    /* Reset search when the text is blanked. Either if the user clears
       everything, or when the Find menu is used. */
    gtk_text_buffer_get_start_iter(gui->txtBuffer, &search_start);
    gtk_text_buffer_move_mark(gui->txtBuffer,
                              gui->txtmark_search_start,
                              &search_start);
    gtk_text_buffer_select_range(gui->txtBuffer, &search_start, &search_start);
    return;
  }

  gtk_text_buffer_get_iter_at_mark(gui->txtBuffer,
                                   &search_start,
                                   gui->txtmark_search_start);

  if (gtk_text_iter_forward_search(&search_start,
                                   text,
                                   0,
                                   &match_start,
                                   &match_end,
                                   NULL)) {
    gtk_text_buffer_select_range(gui->txtBuffer, &match_start, &match_end);
    gtk_text_view_scroll_to_iter(gui->txtView, &match_start,
                                 0, FALSE, 0, 0);

    /* Mark point at which Find Next will start */
    gtk_text_iter_forward_char(&match_start);
    gtk_text_buffer_move_mark(gui->txtBuffer,
                              gui->txtmark_next_search_start,
                              &match_start);
  } else {
    gtk_label_set_text(gui->lblSearchInfo, _("Not found"));
  }
}


static
void
close_search_box_cb(GtkButton *widget, gpointer data)
{
  WorldGUI *gui = (WorldGUI *) data;

  g_object_set(G_OBJECT(gui->search_box), "no-show-all", TRUE, NULL);
  gtk_widget_hide(GTK_WIDGET(gui->search_box));
  gtk_widget_grab_focus(GTK_WIDGET(gui->cmbEntry));
}


static
gboolean
search_box_keypress_cb(GtkWidget   *widget,
                       GdkEventKey *evt,
                       gpointer     data)
{
  if (evt->keyval == GDK_Escape) {
    close_search_box_cb(NULL, data);
    return TRUE;
  }

  return FALSE;
}
