<?php

/**
* Smarty Internal Plugin Data
* 
* This file contains the basic classes and methodes for template and variable creation
* 
* @package Smarty
* @subpackage Templates
* @author Uwe Tews 
*/

/**
* Base class with template and variable methodes
*/
class Smarty_Internal_Data {
    // class used for templates
    public $template_class = 'Smarty_Internal_Template';

    /**
    * assigns a Smarty variable
    * 
    * @param array $ |string $tpl_var the template variable name(s)
    * @param mixed $value the value to assign
    * @param boolean $nocache if true any output of this variable will be not cached
    * @param boolean $scope the scope the variable will have  (local,parent or root)
    */
    public function assign($tpl_var, $value = null, $nocache = false, $scope = SMARTY_LOCAL_SCOPE)
    {
        if (is_array($tpl_var)) {
            foreach ($tpl_var as $_key => $_val) {
                if ($_key != '') {
                    $this->tpl_vars[$_key] = new Smarty_variable($_val, $nocache, $scope);
                } 
            } 
        } else {
            if ($tpl_var != '') {
                $this->tpl_vars[$tpl_var] = new Smarty_variable($value, $nocache, $scope);
            } 
        } 
    } 
    /**
    * assigns a global Smarty variable
    * 
    * @param string $varname the global variable name
    * @param mixed $value the value to assign
    * @param boolean $nocache if true any output of this variable will be not cached
    */
    public function assign_global($varname, $value = null, $nocache = false)
    {
        if ($varname != '') {
            $this->smarty->global_tpl_vars[$varname] = new Smarty_variable($value, $nocache);
        } 
    } 
    /**
    * assigns values to template variables by reference
    * 
    * @param string $tpl_var the template variable name
    * @param mixed $ &$value the referenced value to assign
    * @param boolean $nocache if true any output of this variable will be not cached
    * @param boolean $scope the scope the variable will have  (local,parent or root)
    */
    public function assign_by_ref($tpl_var, &$value, $nocache = false, $scope = SMARTY_LOCAL_SCOPE)
    {
        if ($tpl_var != '') {
            $this->tpl_vars[$tpl_var] = new Smarty_variable(null, $nocache, $scope);
            $this->tpl_vars[$tpl_var]->value = &$value;
        } 
    } 
    /**
    * appends values to template variables
    * 
    * @param array $ |string $tpl_var the template variable name(s)
    * @param mixed $value the value to append
    * @param boolean $merge flag if array elements shall be merged
    * @param boolean $nocache if true any output of this variable will be not cached
    * @param boolean $scope the scope the variable will have  (local,parent or root)
    */
    public function append($tpl_var, $value = null, $merge = false, $nocache = false, $scope = SMARTY_LOCAL_SCOPE)
    {
        if (is_array($tpl_var)) {
            // $tpl_var is an array, ignore $value
            foreach ($tpl_var as $_key => $_val) {
                if ($_key != '') {
                    if (!isset($this->tpl_vars[$_key])) {
                        $tpl_var_inst = $this->getVariable($_key, null, true, false);
                        if ($tpl_var_inst instanceof Undefined_Smarty_Variable) {
                            $this->tpl_vars[$_key] = new Smarty_variable(null, $nocache, $scope);
                        } else {
                            $this->tpl_vars[$_key] = clone $tpl_var_inst;
                            if ($scope != SMARTY_LOCAL_SCOPE) {
                                $this->tpl_vars[$_key]->scope = $scope;
                            } 
                        } 
                    } 
                    if (!(is_array($this->tpl_vars[$_key]->value) || $this->tpl_vars[$_key]->value instanceof ArrayAccess)) {
                        settype($this->tpl_vars[$_key]->value, 'array');
                    } 
                    if ($merge && is_array($_val)) {
                        foreach($_val as $_mkey => $_mval) {
                            $this->tpl_vars[$_key]->value[$_mkey] = $_mval;
                        } 
                    } else {
                        $this->tpl_vars[$_key]->value[] = $_val;
                    } 
                } 
            } 
        } else {
            if ($tpl_var != '' && isset($value)) {
                if (!isset($this->tpl_vars[$tpl_var])) {
                    $tpl_var_inst = $this->getVariable($tpl_var, null, true, false);
                    if ($tpl_var_inst instanceof Undefined_Smarty_Variable) {
                        $this->tpl_vars[$tpl_var] = new Smarty_variable(null, $nocache, $scope);
                    } else {
                        $this->tpl_vars[$tpl_var] = clone $tpl_var_inst;
                        if ($scope != SMARTY_LOCAL_SCOPE) {
                            $this->tpl_vars[$tpl_var]->scope = $scope;
                        } 
                    } 
                } 
                if (!(is_array($this->tpl_vars[$tpl_var]->value) || $this->tpl_vars[$tpl_var]->value instanceof ArrayAccess)) {
                    settype($this->tpl_vars[$tpl_var]->value, 'array');
                } 
                if ($merge && is_array($value)) {
                    foreach($value as $_mkey => $_mval) {
                        $this->tpl_vars[$tpl_var]->value[$_mkey] = $_mval;
                    } 
                } else {
                    $this->tpl_vars[$tpl_var]->value[] = $value;
                } 
            } 
        } 
    } 

    /**
    * appends values to template variables by reference
    * 
    * @param string $tpl_var the template variable name
    * @param mixed $ &$value the referenced value to append
    * @param boolean $merge flag if array elements shall be merged
    */
    public function append_by_ref($tpl_var, &$value, $merge = false)
    {
        if ($tpl_var != '' && isset($value)) {
            if (!isset($this->tpl_vars[$tpl_var])) {
                $this->tpl_vars[$tpl_var] = new Smarty_variable();
            } 
            if (!@is_array($this->tpl_vars[$tpl_var]->value)) {
                settype($this->tpl_vars[$tpl_var]->value, 'array');
            } 
            if ($merge && is_array($value)) {
                foreach($value as $_key => $_val) {
                    $this->tpl_vars[$tpl_var]->value[$_key] = &$value[$_key];
                } 
            } else {
                $this->tpl_vars[$tpl_var]->value[] = &$value;
            } 
        } 
    } 


    /**
    * clear the given assigned template variable.
    * 
    * @param string $ |array $tpl_var the template variable(s) to clear
    */
    public function clear_assign($tpl_var)
    {
        if (is_array($tpl_var)) {
            foreach ($tpl_var as $curr_var) {
                unset($this->tpl_vars[$curr_var]);
            } 
        } else {
            unset($this->tpl_vars[$tpl_var]);
        } 
    } 

    /**
    * clear all the assigned template variables.
    */
    public function clear_all_assign()
    {
        $this->tpl_vars = array();
    } 

    /**
    * load a config file, optionally load just selected sections
    * 
    * @param string $config_file filename
    * @param mixed $sections array of section names, single section or null
    */
    public function config_load($config_file, $sections = null)
    { 
        // load Config class
        $config = new Smarty_Internal_Config($config_file, $this->smarty);
        $config->loadConfigVars($sections, $this);
    } 

    /**
    * gets the object of a Smarty variable
    * 
    * @param string $variable the name of the Smarty variable
    * @param object $_ptr optional pointer to data object
    * @param boolean $search_parents search also in parent data
    * @return object the object of the variable
    */
    public function getVariable($variable, $_ptr = null, $search_parents = true, $error_enable = true)
    {
        if ($_ptr === null) {
            $_ptr = $this;
        } while ($_ptr !== null) {
            if (isset($_ptr->tpl_vars[$variable])) {
                // found it, return it
                return $_ptr->tpl_vars[$variable];
            } 
            // not found, try at parent
            if ($search_parents) {
                $_ptr = $_ptr->parent;
            } else {
                $_ptr = null;
            } 
        } 
        if (isset($this->smarty->global_tpl_vars[$variable])) {
            // found it, return it
            return $this->smarty->global_tpl_vars[$variable];
        } 
        if ($this->smarty->error_unassigned && $error_enable) {
            throw new Exception('Undefined Smarty variable "' . $variable . '"');
        } else {
            return new Undefined_Smarty_Variable;
        } 
    } 
    /**
    * gets  a config variable
    * 
    * @param string $variable the name of the config variable
    * @return mixed the value of the config variable
    */
    public function getConfigVariable($variable)
    {
        $_ptr = $this;
        while ($_ptr !== null) {
            if (isset($_ptr->config_vars[$variable])) {
                // found it, return it
                return $_ptr->config_vars[$variable];
            } 
            // not found, try at parent
            $_ptr = $_ptr->parent;
        } 
        if ($this->smarty->error_unassigned) {
            throw new Exception('Undefined config variable "' . $variable . '"');
        } else {
            return '';
        } 
    } 
    /**
    * gets  a stream variable
    * 
    * @param string $variable the stream of the variable
    * @return mixed the value of the stream variable
    */
    public function getStreamVariable($variable)
    {
        $_result = '';
        if ($fp = fopen($variable, 'r+')) {
            while (!feof($fp)) {
                $_result .= fgets($fp);
            } 
            fclose($fp);
            return $_result;
        } 

        if ($this->smarty->$error_unassigned) {
            throw new Exception('Undefined stream variable "' . $variable . '"');
        } else {
            return '';
        } 
    } 

    /**
    * creates a template object
    * 
    * @param string $template the resource handle of the template file
    * @param object $parent next higher level of Smarty variables
    * @param mixed $cache_id cache id to be used with this template
    * @param mixed $compile_id compile id to be used with this template
    * @returns object template object
    */
    public function createTemplate($template, $cache_id = null, $compile_id = null, $parent = null)
    {
        if (is_object($cache_id) || is_array($cache_id)) {
            $parent = $cache_id;
            $cache_id = null;
        } 
        if (is_array($parent)) {
            $data = $parent;
            $parent = null;
        } else {
            $data = null;
        } 
        if (!is_object($template)) {
            // we got a template resource
            // already in template cache?
            $_templateId = crc32($template . $cache_id . $compile_id);
            if (isset($this->smarty->template_objects[$_templateId]) && $this->smarty->caching) {
                // return cached template object
                $tpl = $this->smarty->template_objects[$_templateId];
            } else {
                // create new template object
                $tpl = new $this->template_class($template, $this->smarty, $parent, $cache_id, $compile_id);
            } 
        } else {
            // just return a copy of template class
            $tpl = $template;
        } 
        // fill data if present
        if (is_array($data)) {
            // set up variable values
            foreach ($data as $_key => $_val) {
                $tpl->tpl_vars[$_key] = new Smarty_variable($_val);
            } 
        } 
        return $tpl;
    } 

    /**
    * return current time
    * 
    * @returns double current time
    */
    function _get_time()
    {
        $_mtime = microtime();
        $_mtime = explode(" ", $_mtime);
        return (double)($_mtime[1]) + (double)($_mtime[0]);
    } 
} 

/**
* class for the Smarty data object
* 
* The Smarty data object will hold Smarty variables in the current scope
* 
* @param object $parent tpl_vars next higher level of Smarty variables
*/
class Smarty_Data extends Smarty_Internal_Data {
    // array of variable objects
    public $tpl_vars = array(); 
    // back pointer to parent object
    public $parent = null; 
    // config vars
    public $config_vars = array();
    /**
    * create Smarty data object
    */
    public function __construct ($_parent = null)
    {
        if (is_object($_parent)) {
            // when object set up back pointer
            $this->parent = $_parent;
        } elseif (is_array($_parent)) {
            // set up variable values
            foreach ($_parent as $_key => $_val) {
                $this->tpl_vars[$_key] = new Smarty_variable($_val);
            } 
        } else {
            throw new Exception("Wrong type for template variables");
        } 
    } 
} 
/**
* class for the Smarty variable object
* 
* This class defines the Smarty variable object
*/
class Smarty_Variable {
    // template variable
    public $value;
    public $nocache;
    public $scope;
    /**
    * create Smarty variable object
    * 
    * @param mixed $value the value to assign
    * @param boolean $nocache if true any output of this variable will be not cached
    * @param boolean $scope the scope the variable will have  (local,parent or root)
    */
    public function __construct ($value = null, $nocache = false, $scope = SMARTY_LOCAL_SCOPE)
    {
        $this->value = $value;
        $this->nocache = $nocache;
        $this->scope = $scope;
    } 
} 

/**
* class for undefined variable object
* 
* This class defines an object for undefined variable handling
*/
class Undefined_Smarty_Variable {
    // return always false
    public function __get ($name)
    {
        if ($name == 'nocache') {
            return false;
        } else {
            return null;
        } 
    } 
} 

?>
