/*
 *  
 *  $Id: geometria3d.h 3521 2011-03-16 14:54:22Z carlos $
 *  Ginkgo CADx Project
 *
 *  Copyright 2008-10 MetaEmotion S.L. All rights reserved.
 *  http://ginkgo-cadx.com
 *
 *  This file is licensed under LGPL v3 license.
 *  See License.txt for details
 *
 */
#pragma once
#include <list>
#include <limits>

#include <api/globals.h>
#include <api/math/geometria_defs.h>


namespace GNC {
	namespace GCS {

		template <class TComp, class TReal> class GMatriz3x3;

		template <class TComp=PRECISION_COMPONENTE_POR_DEFECTO, class TReal=PRECISION_REALES_POR_DEFECTO> class GVector3D {
		public:
			typedef TComp                 TComponente;
			typedef TReal                 TReales;
			typedef GVector3D<TComp, TReal> TVector;

			TComp v[3];         // Coordenadas

			TComp& x;           // Coordenada X
			TComp& y;           // Coordenada Y
			TComp& z;           // Coordenada Z

		public:

			static const GVector3D<TComp, TReal> NaN() {
				return GVector3D<TComp, TReal>(NaNDeComponentes(), NaNDeComponentes(), NaNDeComponentes());
			}

			static const GVector3D<TComp, TReal> Inf() {
				return GVector3D<TComp, TReal>(InfinitoDeComponentes(), InfinitoDeComponentes(), InfinitoDeComponentes());
			}

			static const GVector3D<TComp, TReal> Zero() {
				return GVector3D<TComp, TReal>();
			}

			static const GVector3D<TComp, TReal> Unidad() {
				return GVector3D<TComp, TReal>(1.0f, 1.0f, 1.0f);
			}

			//----------------------------------------------
			//-- Constructores y destructores
			//----------------------------------------------

			inline GVector3D() : x(v[0]), y(v[1]), z(v[2]) {
				x = (TComp) 0.0f;
				y = (TComp) 0.0f;
				z = (TComp) 0.0f;
			}

			/* Constructor copia transparente */
			template <class TTipoComp, class TTipoReal> inline GVector3D(const GVector3D<TTipoComp, TTipoReal>& b) : x(v[0]), y(v[1]), z(v[2]) {
				x = (TComp) b.x;
				y = (TComp) b.y;
				z = (TComp) b.z;
			}

			template <class TipoComp1, class TipoComp2, class TipoComp3> inline GVector3D(const TipoComp1& valX, const TipoComp2& valY, const TipoComp3& valZ) : x(v[0]), y(v[1]), z(v[2]) {
				x = (TComp) valX;
				y = (TComp) valY;
				z = (TComp) valZ;
			}

			template <class TipoComp> inline GVector3D(const TipoComp* const valor) : x(v[0]), y(v[1]), z(v[2]) {
				x = (TComp) valor[0];
				y = (TComp) valor[1];
				z = (TComp) valor[2];
			}

			~GVector3D() {

			}

			//----------------------------------------------
			//-- Asignacion explicita
			//----------------------------------------------

			/* Asigna las coordenadas al vector */
			template <class TipoComp1, class TipoComp2, class TipoComp3> inline TVector& Asignar(const TipoComp1& valX, const TipoComp2& valY, const TipoComp3& valZ) {
				x = (TComp) valX;
				y = (TComp) valY;
				z = (TComp) valZ;
				return (*this);
			}

			/* Asigna el valor a todas las coordenadas del vector */
			template <class TipoComp> inline TVector& Asignar(const TipoComp& valor) {
				x = (TComp) valor;
				y = (TComp) valor;
				z = (TComp) valor;
				return *this;
			}

			//----------------------------------------------
			//-- Normas Vectoriales, Distancias y Producto Interior
			//----------------------------------------------

			inline TComp Norma1() const {
				return std::abs(x) + std::abs(y) + std::abs(z);
			}

			/* Devuelve la norma 2 (euclidea) al cuadrado */
			inline TComp Norma2Cuadrado() const {
				return (TReal)x * (TReal)x + (TReal)y * (TReal)y + (TReal)z * (TReal)z;
			}

			/* Devuelve la norma 2 (euclidea) */
			inline TReal Norma2() const {
				return std::sqrt((TReal) (*this).Norma2Cuadrado());
			}

			inline TReal NormaEnesima(unsigned int n) const {
				return std::pow( std::pow((TReal)x, (TReal)n) + std::pow((TReal)y, (TReal)n) + std::pow((TReal)z, (TReal)n), (TReal)1.0f / (TReal)n);
			}

			inline TComp NormaInfinito() const {
				return std::max(x, std::max(y, z));
			}

			/* Devuelve la distancia euclidea a "b" */
			inline TReal DistanciaEuclidea(const TVector& b) const {
				const TVector& a = *this;
				return std::sqrt( (double) (b - a).Norma2Cuadrado() );
			}

			/* Devuelve el cuadrado de la distancia euclidea */
			inline TReal DistanciaEuclideaCuadrado(const TVector& b) const {
				const TVector& a = *this;
				return (b - a).Norma2Cuadrado();
			}

			/* Normaliza un vector teniendo en cuenta singularidades */
			inline TVector& Normalizar() {

				TVector& a = (*this);

				const TReal& norma = a.Norma2();

				if ( std::abs(norma) < EpsilonDeReales()) {
					a.Asignar( (TComp)0.0f, (TComp)0.0f);
				}
				else {
					a /= norma;
				}
				return a;
			}

			/* Devuelve el vector normalizado teniendo en cuenta singularidades */
			inline TVector Normalizado() const {
				return TVector(*this).Normalizar();
			}

			/* Devuelve el producto escalar por b */
			inline TReal ProductoEscalar(const GVector3D<TComp>& b) const {
				const TVector& a = *this;
				return ((TReal)a.x * (TReal)b.x) + ((TReal)a.y * (TReal)b.y) + ((TReal)a.z * (TReal)b.z);
			}

			/* Devuelve el producto vectorial */
			inline TVector ProductoVectorial(const GVector3D<TComp>& b) const {
				const TVector& a = *this;
				return TVector(
					(TReal)a.y * (TReal)b.z) - ((TReal)b.y * (TReal)a.z,
					(TReal)a.x * (TReal)b.z) - ((TReal)b.x * (TReal)a.z,
					(TReal)a.x * (TReal)b.y) - ((TReal)b.x * (TReal)a.y);
			}

			//----------------------------------------------
			//-- Angulos
			//----------------------------------------------

			/* Devuelve el angulo { entre [0, -PI) } que forma con "b" */
			inline TReal Angulo(const TVector& b) const {
				const TVector& a = (*this);
				const TReal& angulo = std::acos( ((TReal) a.ProductoEscalar(b)) / (a.Norma2() * b.Norma2()) );
				return angulo;
			}


			//----------------------------------------------
			//-- Ajuste e interpolacion
			//----------------------------------------------



			//----------------------------------------------
			//-- Componentes
			//----------------------------------------------

			/* Devuelve la mayor de las componentes */
			inline const TComp& Mayor() const {
				return x > y ? (x > z ? x : z) : (y > z ? y : z);
			}

			/* Devuelve la menor de las componentes */
			inline const TComp& Menor() const {
				return x > y ? (y > z ? z : y) : (x > z ? z : x );
			}

			//----------------------------------------------
			//-- Constantes
			//----------------------------------------------

			static inline const TComp InfinitoDeComponentes() {
				return std::numeric_limits<TComp>::infinity();
			}

			static inline const TReal InfinitoDeReales() {
				return std::numeric_limits<TReal>::infinity();
			}

			static inline const TComp NaNDeComponentes() {
				return std::numeric_limits<TComp>::quiet_NaN();
			}

			static inline const TReal NaNDeReales() {
				return std::numeric_limits<TReal>::quiet_NaN();
			}

			static inline const TComp EpsilonDeComponentes() {
				return std::numeric_limits<TComp>::epsilon();
			}

			static inline const TReal EpsilonDeReales() {
				return std::numeric_limits<TReal>::epsilon();
			}

			//----------------------------------------------
			//-- Comprobaciones
			//----------------------------------------------

			inline bool EsNulo() const {
				return x == 0.0 && y == 0.0 && z == 0.0;
			}

			inline bool EsInvalido() const {
				return (x != x) || (y != y) || (z != z);
			}

			inline bool EsNaN() const {
				return (x != x) || (y != y) || (z != z);
			}

			//----------------------------------------------
			//-- Geometria
			//----------------------------------------------

			/** Devuelve el punto de interseccion entre una recta y un plano
			La recta
			**/

			static TVector InterseccionEntreRectaYPlano(const TVector& punto_recta, const TVector& vdir_recta, const TVector& centro_plano, const TVector& vnorm_plano)
			{
				TReal numerador = (centro_plano - punto_recta).ProductoEscalar(vnorm_plano);
				TReal denominador = vdir_recta.ProductoEscalar(vnorm_plano);
				TReal distancia = 0.0;
				if (std::abs(denominador) < EpsilonDeReales()) {
					if (std::abs(numerador) < EpsilonDeReales()) { // Colineales
						return punto_recta;
					}
					else {
						return Inf(); // Paralelos. Se cortan en el infinito
					}
				}
				else {
					distancia = numerador / denominador;
					return punto_recta + vdir_recta * distancia;
				}
			}

			/* Devuelve el punto medio entre dos vectores */
			inline TVector PuntoMedio(const TVector& b) const {
				const TVector& a = (*this);
				return (a + b) / 2.0f;
			}

			/* Devuelve el punto de la proyeccion ortogonal del punto sobre la recta definida por p0 y p1 */
			inline TVector ProyeccionOrtogonalSobreRecta(const TVector& p0, const TVector& p1) const {
				//return InterseccionEntreRectas(*this, *this + (p1-p0).VectorOrtogonal(), p0, p1);
				return TVector();
			}

			/* Devuelve la distancia del punto a la recta definida por p0 y p1 */
			inline TReal DistanciaARecta(const TVector& p0, const TVector& p1) const {
				//return (ProyeccionOrtogonalSobreRecta(p0, p1) - (*this)).Norma2();
				//
				return 0.0;
			}

			/* Devuelve el cuadrado de la distancia del punto a la recta definida por p0 y p1 */
			inline TComp DistanciaARectaCuadrado(const GVector3D& p0, const GVector3D& p1) const {
				return (ProyeccionOrtogonalSobreRecta(p0, p1) - (*this)).Norma2Cuadrado();
			}

			//----------------------------------------------
			//-- Acotaciones
			//----------------------------------------------

			inline TVector ValorAbsoluto() {
				return TVector(std::abs(x), std::abs(y), std::abs(z));
			}

			inline TVector TruncadoAMaximo(const TComp& val) const
			{
				return TVector(*this).TruncarAMaximo(val);
			}

			inline TVector TruncadoAMinimo(const TComp& val) const
			{
				return TVector(*this).TruncarAMinimo(val);
			}

			inline TVector& TruncarAMaximo(const TComp& val)
			{
				x = std::max(x, val);
				y = std::max(y, val);
				z = std::max(z, val);
				return *this;
			}

			inline TVector& TruncarAMinimo(const TComp& val)
			{
				x = std::min(x, val);
				y = std::min(y, val);
				z = std::min(z, val);
				return *this;
			}

			inline TVector TruncadoAMaximo(TComp valx, TComp valy) const
			{
				return TVector(*this).TruncarAMaximo(valx, valy);
			}

			inline TVector TruncadoAMinimo(TComp valx, TComp valy) const
			{
				return TVector(*this).TruncarAMinimo(valx, valy);
			}

			inline TVector& TruncarAMaximo(TComp valx, TComp valy, TComp valz)
			{
				x = std::max(x, valx);
				y = std::max(y, valy);
				z = std::max(z, valz);
				return *this;
			}

			inline TVector& TruncarAMinimo(TComp valx, TComp valy, TComp valz)
			{
				x = std::min(x, valx);
				y = std::min(y, valy);
				z = std::min(z, valz);
				return *this;
			}

			inline TComp ComponenteMaxima() const
			{
				return std::max(x,std::max(y, z));
			}

			inline TComp ComponenteMinima() const
			{
				return std::min(x,std::min(y, z));
			}

			static inline TVector ComponentesMaximas(const TVector& a, const TVector& b)
			{
				return TVector(std::max(a.x, b.x), std::max(a.y, b.y), std::max(a.z, b.z));
			}

			static inline TVector ComponentesMinimas(const TVector& a, const TVector& b)
			{
				return TVector(std::min(a.x, b.x), std::min(a.y, b.y), std::min(a.z, b.z));
			}

			inline TVector& AsignarMaximos(const TVector& o)
			{
				x = std::max(x, o.x);
				y = std::max(y, o.y);
				z = std::max(x, o.z);
				return *this;
			}

			inline TVector& AsignarMinimos(const TVector& o)
			{
				x = std::min(x, o.x);
				y = std::min(y, o.y);
				z = std::min(z, o.z);
				return *this;
			}

			inline const TVector& AsignarMaximos(const TVector& o) const
			{
				return TVector(std::max(x, o.x), std::max(y, o.y), std::max(z, o.z));
			}

			inline const TVector& AsignarMinimos(const TVector& o) const
			{
				return TVector(std::min(x, o.x), std::min(y, o.y), std::min(z, o.z));
			}

			inline bool TieneComponenteMayorQue(const TVector& o) const
			{
				return ( (x > o.x) || (y > o.y) || (z > o.z));
			}

			inline bool TieneComponenteMenorQue(const TVector& o) const
			{
				return ( (x < o.x) || (y < o.y)  || (z < o.z));
			}

			/* Devuelve verdadero si el vector es singular (distancia a (0,0) < epsilon =~ (0,0) ) */
			inline bool Singular() const {
				return std::abs(x) < EpsilonDeComponentes() || std::abs(y) < EpsilonDeComponentes() || std::abs(z) < EpsilonDeComponentes();
			}

			inline TVector RedondeoAlza() const {
				return TVector(*this).RedondearAlza();
			}

			inline TVector& RedondearAlza() {
				x = std::ceil(x);
				y = std::ceil(y);
				z = std::ceil(z);
				return *this;
			}

			inline TVector RedondeoBaja() const {
				return GVector3D(*this).RedondearBaja();
			}

			inline TVector& RedondearBaja() {
				x = std::floor(x);
				y = std::floor(y);
				z = std::floor(z);
				return *this;
			}

			template <class TipoComp> static TipoComp ValorRedondeado(const TipoComp& val) {
				const TipoComp& d = std::floor(val);
				const TipoComp& u = std::ceil(val);
				if ( val - d < u - val) {
					return d;
				}
				else {
					return u;
				}
			}

			inline TVector Redondeado() const {
				return TVector(*this).Redondear();
			}

			inline TVector& Redondear() {
				x = ValorRedondeado(x);
				y = ValorRedondeado(y);
				z = ValorRedondeado(z);
				return *this;
			}

			// Calcula y devuelve el centroide de una coleccion de puntos.
			template <class TipoVector>
			static TVector Centroide(const TipoVector v[], const int& numVertices)
			{
				TVector vSum;

				if (numVertices > 0) {
					for (int i = 0; i < numVertices; ++i) {
						vSum += v[i];
					}
					vSum /= (TComp)numVertices;
				}
				return vSum;
			}

			// Calcula y devuelve el centroide de una coleccion de puntos.
			template <class TipoVector>
			static TVector Centroide(const std::list<TipoVector>& lista)
			{
				typename std::list<TipoVector>::const_iterator i = lista.begin();
				TVector vSum;

				if (i != lista.end()) {
					for ( i = lista.begin(); i != lista.end(); i++) {
						vSum += (*i);
					}
					vSum /= (TComp)lista.size();
				}
				return vSum;
			}


			// Intercambia los valores de p0 a p1 y viceversa
			template <class TipoComp, class TipoReal> static void IntercambiarValores(GVector3D<TipoComp, TipoReal>& p0, GVector3D<TipoComp, TipoReal>& p1) {
				const GVector3D<TipoComp, TipoReal>& temp = p0;
				p0 = p1;
				p1 = temp;
			}

			inline GVector3D<TComp,TReal>& operator=(const GVector3D<TComp,TReal>& otro) {
				v[0] = (TComp) otro.v[0];
				v[1] = (TComp) otro.v[1];
				v[2] = (TComp) otro.v[2];
				return *this;
			}


			template <class TipoComp, class TipoReal> inline GVector3D<TComp,TReal>& operator=(const GVector3D<TipoComp, TipoReal>& otro) {

				v[0] = (TComp) otro.v[0];
				v[1] = (TComp) otro.v[1];
				v[2] = (TComp) otro.v[1];
				return *this;
			}


			template <class TipoComp> inline TVector& operator=(const TipoComp* const vec) {
				x = (TComp) vec[0];
				y = (TComp) vec[1];
				z = (TComp) vec[2];
				return *this;
			}

			inline bool operator==(const TVector& otro) const {
				return x == (TComp) otro.x && y == (TComp) otro.y && z == (TComp) otro.z;
			}

			inline bool operator!=(const TVector& otro) const {
				return !(*this == otro);
			}

			inline TVector operator*(const TVector& otro) const {
				return TVector(x * otro.x, y * otro.y, z * otro.z);
			}

			inline TVector& operator*=(const TVector& otro) {
				x *= otro.x;
				y *= otro.y;
				z *= otro.z;
				return *this;
			}

			template <class TipoComp> inline TVector operator*(const TipoComp& valor) const {
				return TVector(x * valor, y * valor, z * valor);
			}

			template <class TipoComp> inline TVector& operator*=(const TipoComp& valor) {
				x *= (TComp) valor;
				y *= (TComp) valor;
				z *= (TComp) valor;
				return *this;
			}

			inline TVector operator/(const TVector& otro) const {
				return TVector(x / otro.x, y / otro.y, z / otro.z);
			}

			inline TVector& operator/=(const TVector v) {
				x /= v.x;
				y /= v.y;
				z /= v.z;
				return *this;
			}

			template <class TipoComp> inline TVector operator/(const TipoComp& v) const {
				return TVector(x / v, y / v, z / v);
			}

			template <class TipoComp> inline TVector& operator/=(const TipoComp& v) {
				x /= v;
				y /= v;
				z /= v;
				return *this;
			}

			inline GVector3D operator-(const GVector3D& otro) const {
				return TVector(x - otro.x, y - otro.y, z - otro.z);
			}

			inline GVector3D& operator-=(const GVector3D& otro) {
				x -= otro.x;
				y -= otro.y;
				z -= otro.z;
				return *this;
			}

			inline GVector3D operator-(const TComp& valor) const {
				return TVector(x - valor, y - valor, z - valor);
			}

			inline GVector3D& operator-=(const TComp& valor) {
				x -= valor;
				y -= valor;
				z -= valor;
				return *this;
			}

			inline TVector operator+(const GVector3D& otro) const {
				return TVector(x + otro.x, y + otro.y, z + otro.z);
			}

			inline GVector3D& operator+=(const GVector3D& otro) {
				x += otro.x;
				y += otro.y;
				z += otro.z;
				return *this;
			}

			inline GVector3D operator+(const TComp& valor) const {
				TVector t(*this);
				return t += valor;
			}

			inline GVector3D& operator+=(const TComp& valor) {
				x += (TComp) valor;
				y += (TComp) valor;
				z += (TComp) valor;
				return *this;
			}

			//----------------------------------------------
			//-- Accesores
			//----------------------------------------------

			inline TComp& operator [] (int i) {
				return v[i];
			}

			inline friend std::ostream& operator<<(std::ostream& out, const GVector3D& c) {
				out << "[ x = " << c.x << ", y = " << c.y << ", z = " << c.z << " ]";
				return out;
			}

			inline friend std::ostream& operator<<(std::ostream& out, const GVector3D* const c) {
				if (c == NULL) {
					out << "[ NULL ]";
				}
				else {
					out << *c;
				}
				return out;
			}
		};

		template <class TComp=PRECISION_COMPONENTE_POR_DEFECTO, class TReal=PRECISION_REALES_POR_DEFECTO>
		class  GMatriz3x3 {

		public:
			typedef TComp TipoComponente;
			typedef TReal TipoReales;
			typedef GVector3D<TComp, TReal> TVector;
			typedef GMatriz3x3<TComp, TReal> TMatriz;

			TReal v[9];

			TReal& a00;
			TReal& a01;
			TReal& a02;
			TReal& a10;
			TReal& a11;
			TReal& a12;
			TReal& a20;
			TReal& a21;
			TReal& a22;

			inline  GMatriz3x3() : a00(v[0]), a01(v[1]), a02(v[2]), a10(v[3]), a11(v[4]), a12(v[5]), a20(v[6]), a21(v[7]), a22(v[8])
			{
				for (int i = 0; i < 9; i++) {
					v[i] = 0.0;
				}

			}

			inline  GMatriz3x3(const GMatriz3x3& otra) :  a00(v[0]), a01(v[1]), a02(v[2]), a10(v[3]), a11(v[4]), a12(v[5]), a20(v[6]), a21(v[7]), a22(v[8])
			{
				*this = otra;
			}

			inline TMatriz& operator = (const TMatriz& otra)
			{
				for (int i = 0; i < 9; i++) {
					v[i] = otra.v[i];
				}
			}

			/*
			template <class TipoComp> inline static TMatriz MatrizRotacion(const TipoComp& angulo) {
				return TMatriz(std::cos(angulo), -std::sin(angulo), std::sin(angulo), std::cos(angulo));
			}

			inline TMatriz ProductoMatricial(const TMatriz& otra) const {
				return TMatriz(
							  a00 * otra.a00 + a01 * otra.a10,
							  a00 * otra.a01 + a01 * otra.a11,
							  a10 * otra.a00 + a11 * otra.a10,
							  a10 * otra.a01 + a11 * otra.a11);
			}

			inline TVector ProductoMatricial(const TVector& otro) const {
				return TVector(
							   (typename TVector::TComponente)( a00 * (TReal) otro.x + a01 * (TReal) otro.y ),
							   (typename TVector::TComponente)( a10 * (TReal) otro.x + a11 * (TReal) otro.y ) );
			}
			*/

			inline friend std::ostream& operator<<(std::ostream& out, const  GMatriz3x3& m) {
				out << "[ a00 = " << m.a00 << ", a01 = " << m.a01 << ", a02 = " << m.a02 << ";  " << std::endl;
				out << "[ a10 = " << m.a10 << ", a11 = " << m.a11 << ", a12 = " << m.a12 << ";  " << std::endl;
				out << "[ a20 = " << m.a20 << ", a21 = " << m.a21 << ", a22 = " << m.a22 << "]";
				return out;
			}

			inline friend std::ostream& operator<<(std::ostream& out, const  GMatriz3x3* m) {
				if (m == NULL) {
					out << "[ NULL ]";
				}
				else {
					out << *m;
				}
				return out;
			}
		};


		/* Devuelve un nuevo vector rotado el vector sobre el eje de coordenadas Z */
		/*
		template <class TComp, class TReal>
		inline GVector3D<TComp,TReal> GVector3D<TComp,TReal>::RotacionSobreZ(const TReal& angulo) const
		{
			return  GMatriz3x3<TComp,TReal>::MatrizRotacion(angulo).ProductoMatricial(*this);
		}

		template <class TComp, class TReal>
		inline GVector3D<TComp,TReal> GVector3D<TComp,TReal>::RotacionSobreZ(const TVector& centro, const TReal& angulo) const
		{
			return centro +  GMatriz3x3<TComp,TReal>::MatrizRotacion(angulo).ProductoMatricial(*this - centro);
		}
		*/

		typedef GVector3D<> Vector3D;
		typedef GMatriz3x3<> Matriz3x3;

	}
}
