#pragma once

#include <map>
#include <string>
#include <iostream>


namespace GIL {
	namespace DICOM {

		class IDICOMImg2DCM;
		class IDICOMManager;
	}
}

class DcmElement;

namespace GIL {
	namespace DICOM {

		class IDICOMImg2DCM;
		class IDICOMManager;

		class EXTAPI TipoTagPrivado {
		public:
			TipoTagPrivado() {}
			virtual ~TipoTagPrivado() {}
			virtual std::string ToString() = 0;
			virtual DcmElement* ToElement(unsigned int group, unsigned int element) = 0;
		};

		class EXTAPI TagPrivadoString: public TipoTagPrivado {
		public:
			TagPrivadoString(const std::string& valor)
			{
				Valor = valor;
			}
			virtual ~TagPrivadoString() {}

			virtual std::string ToString() {
				return Valor;
			}

			virtual DcmElement* ToElement(unsigned int group, unsigned int element);
		protected:
			std::string Valor;
		};


		class EXTAPI TagPrivadoUndefined: public TipoTagPrivado {
		public:
			TagPrivadoUndefined();

			TagPrivadoUndefined(unsigned int size);

			TagPrivadoUndefined(unsigned char* valor, unsigned int size);

			virtual ~TagPrivadoUndefined();

			TagPrivadoUndefined(const TagPrivadoUndefined& otro);

			TagPrivadoUndefined& operator = (const TagPrivadoUndefined& otro) ;

			void Copiar(void* ptrCopia, int size) ;

			virtual std::string ToString() ;

			unsigned char* GetValor();

			unsigned int GetSize();

			virtual DcmElement* ToElement(unsigned int group, unsigned int element);
		protected:
			unsigned char* Valor;
			unsigned int Size;
		};

		typedef struct EXTAPI TipoPrivateTags {
		public:
			typedef std::map<unsigned char, TipoTagPrivado* > ListaTags;
			std::string UIDModulo;

			TipoPrivateTags(){

			}

			TipoPrivateTags(const std::string& uidModulo){
				this->UIDModulo = uidModulo;
			}

			~TipoPrivateTags();

			ListaTags& GetListaTags()
			{
				return Tags;
			}

			void Vaciar()
			{
				for(ListaTags::iterator it=Tags.begin(); it!=Tags.end(); it++){
					if((*it).second!=NULL){
						delete ((*it).second);
					}
				}
				Tags.clear();
			}

			void DeleteTag(unsigned char tag){
				if(Tags.find(tag)!=Tags.end()){
					//se borra
					if (Tags[tag] != NULL) {
						delete Tags[tag];
					}
					Tags.erase(tag);
				}
			}

			bool GetTag(unsigned char tag, std::string& valor) {
				if(Tags.find(tag)!=Tags.end()){
					valor = Tags[tag]->ToString();
					return true;
				} else {
					valor="";
					return false;
				}
			}

			TagPrivadoUndefined* GetTagUndefined(unsigned char tag)
			{
				if(Tags.find(tag)!=Tags.end()){
					TagPrivadoUndefined* pTag = dynamic_cast<TagPrivadoUndefined*> (Tags[tag]);
					return pTag;
				} else {
					return NULL;
				}				
			}

			void SetTag(unsigned char tag, const std::string& valor){
				if(Tags.find(tag) != Tags.end()) {
					delete Tags[tag];
				}
				Tags[tag] = new TagPrivadoString(valor);
			}

			void SetTag(unsigned char tag, unsigned char* valor, int longitud) {
				if(Tags.find(tag) != Tags.end()) {
					delete Tags[tag];
				}
				Tags[tag] = new TagPrivadoUndefined(valor, longitud);
			}

			TagPrivadoUndefined* NewTagUndefined(unsigned char tag, int longitud) {
				if(Tags.find(tag) != Tags.end()) {
					delete Tags[tag];
				}
				TagPrivadoUndefined* pTag = new TagPrivadoUndefined(longitud);
				Tags[tag] = pTag;
				return pTag;
			}
		protected:			
			ListaTags Tags;

		} TipoPrivateTags;
		
		typedef struct EXTAPI TipoMetaInfo {
			typedef std::map<std::string, std::string> ListaTags;
			
			ListaTags tags;
			
			~TipoMetaInfo()
			{
			}
			
			/* Obtiene un tag y lo asigna en el parámetro de salida valor.
			 Devuelve true si el tag existe y false si el tag no existe */
			bool getTag(const std::string& tag, std::string & valor)
			{
				ListaTags::iterator it = tags.find(tag);
				bool found = false;
				if (it != tags.end()) {
					valor = (*it).second;
					found = true;
				}
				return found;
			}
			
			std::string getTag(const std::string& tag)
			{
				ListaTags::iterator it = tags.find(tag);
				if (it != tags.end()) {
					return (*it).second;
				}
				return "";
			}
			
			template<class T> T getTagAs(const std::string& tag, const T& defaultValue)
			{
				T ret = defaultValue;
				std::stringstream is(getTag(tag));
				is >> ret;
				return ret;
			}
		} TipoMetaInfo;

		typedef std::map<std::string, std::string> ListaTags;


		typedef struct EXTAPI TipoJerarquia {
			typedef std::list<TipoJerarquia> ListaJerarquias;

			ListaJerarquias secuencias;
			ListaJerarquias items;
			ListaTags tags;

			std::string tagName;

			~TipoJerarquia()
			{
			}

			//añade referencias a las imagenes a las que hace referencia
			void AddReference(const std::string& sopClassUID, const std::string& sopInstanceUID) {
				//se busca la secuencia de referencias
				GIL::DICOM::TipoJerarquia referencia;
				referencia.tags[std::string("0008|1150")] =  sopClassUID;
				referencia.tags[std::string("0008|1155")] =  sopInstanceUID;
				TipoJerarquia* pSecuencia_referencias = buscar_secuencia("0008|1140");
				if (  pSecuencia_referencias== NULL) {
					TipoJerarquia secuencia_referencias;
					secuencia_referencias.items.push_back(referencia);
					secuencia_referencias.tagName = "0008|1140";
					secuencias.push_back(secuencia_referencias);
				} else {
					//se busca si ya esta metida la referencia
					bool encontrado = false;
					for(GIL::DICOM::TipoJerarquia::ListaJerarquias::iterator it = pSecuencia_referencias->items.begin(); it!= pSecuencia_referencias->items.end(); it++) {
						std::string uid;
						if((*it).getTag("0008|1155",uid)){
							if (uid == sopInstanceUID)
							{
								encontrado = true;
								break;
							}
						}
					}
					if(!encontrado) {
						pSecuencia_referencias->items.push_back(referencia);
					}
				}
			}

			//añade referencias a las imagenes originales
			void AddReferenceSource(const std::string& sopClassUID, const std::string& sopInstanceUID) {
				//se busca la secuencia de referencias
				GIL::DICOM::TipoJerarquia referencia;
				referencia.tags[std::string("0008|1150")] =  sopClassUID;
				referencia.tags[std::string("0008|1155")] =  sopInstanceUID;
				TipoJerarquia* pSecuencia_referencias = buscar_secuencia("0008|2112");
				if (  pSecuencia_referencias== NULL) {
					TipoJerarquia secuencia_referencias;
					secuencia_referencias.items.push_back(referencia);
					secuencia_referencias.tagName = "0008|2112";
					secuencias.push_back(secuencia_referencias);
				} else {
					//se busca si ya esta metida la referencia
					bool encontrado = false;
					for(GIL::DICOM::TipoJerarquia::ListaJerarquias::iterator it = pSecuencia_referencias->items.begin(); it!= pSecuencia_referencias->items.end(); it++) {
						std::string uid;
						if((*it).getTag("0008|1155",uid)){
							if (uid == sopInstanceUID)
							{
								encontrado = true;
								break;
							}
						}
					}
					if(!encontrado) {
						pSecuencia_referencias->items.push_back(referencia);
					}
				}
			}

			TipoJerarquia* buscar_secuencia(const char* tag)
			{
				std::string stag = tag;
				return buscar_secuencia(stag);
			}

			TipoJerarquia* buscar_secuencia(const std::string & tag)
			{
				for (ListaJerarquias::iterator it = secuencias.begin(); it != secuencias.end(); it++) {
					if ((*it).tagName == tag) {
						return &(*it);
					}
				}
				return NULL;
			}

			/* Obtiene un tag y lo asigna en el parámetro de salida valor.
						Devuelve true si el tag existe y false si el tag no existe */
			bool getTag(const std::string& tag, std::string & valor) const
			{
				ListaTags::const_iterator it = tags.find(tag);
				bool found = false;
				if (it != tags.end()) {
					valor = (*it).second;
					found = true;
				}
				return found;
			}

			std::string getTag(const std::string& tag) const
			{
				ListaTags::const_iterator it = tags.find(tag);
				if (it != tags.end()) {
					return (*it).second;
				}
				return "";
			}

			template<class T> T getTagAs(const std::string& tag, const T& defaultValue)
			{
				T ret = defaultValue;
				std::stringstream is(getTag(tag));
				is >> ret;
				return ret;
			}

			inline void dump(std::ostream& out, int indentLevel = 0) const
			{
				for (int i = 0; i < indentLevel; i++) {
					out << " ";
				}
				out << "[ " << std::endl;

				for (ListaJerarquias::const_iterator it = secuencias.begin(); it != secuencias.end(); it++) {
					for (int i = 0; i < indentLevel + 3; i++) {
						out << " ";
					}
					out << "SEQ " << (*it).tagName << std::endl;
					(*it).dump(out, indentLevel + 6);
				}
				for (ListaJerarquias::const_iterator it = items.begin(); it != items.end(); it++) {
					for (int i = 0; i < indentLevel + 3; i++) {
						out << " ";
					}
					out << "ITEM " << std::endl;

					(*it).dump(out, indentLevel + 6);
				}
				for (ListaTags::const_iterator it = tags.begin(); it != tags.end(); it++) {
					for (int i = 0; i < indentLevel + 3; i++) {
						out << " ";
					}
					out << "TAG " << (*it).first << " = " << (*it).second << std::endl;

				}
				for (int i = 0; i < indentLevel; i++) {
					out << " ";
				}
				out << "]" << std::endl;
			}

			inline friend std::ostream & operator <<(std::ostream& out, const TipoJerarquia & j)
			{
				j.dump(out);
				return out;
			}

			inline friend std::ostream & operator <<(std::ostream& out, const TipoJerarquia * j)
			{
				if (j == NULL) {
					out << "[ NULL ]" << std::endl;
				} else {
					out << *j;
				}
				return out;
			}

		} TipoJerarquia;

		class EXTAPI I2DException : public std::exception {
		public:
			I2DException(const std::string& msg, const std::string& componente="GIL/I2D", bool fatal = true) throw()
			{
				this->cause = msg;
				this->component = componente;
				this->fatal = fatal;
			}
			~I2DException() throw()
			{
			}

			const std::string& GetComponent() const
			{
				return component;
			}

			const std::string& GetCause() const {
				return cause;
			}

			std::string GetFullCause() const {
				return *this;
			}

			bool IsFatal()  const {
				return fatal;
			}

			operator std::string () const {
				return std::string("Excepcion en componente ") + GetComponent() + std::string(": ")  + GetCause();
			}

			friend std::ostream& operator<<(std::ostream& out, const I2DException& ex)
			{
				out << (const std::string&) ex;
				return out;
			}

			friend std::ostream& operator<<(std::ostream& out, const I2DException* ex)
			{
				if (ex != NULL) {
					out << *ex;
				}
				else {
					out << "(NULL)";
				}
				return out;
			}

		protected:
			std::string cause;
			std::string component;
			bool fatal;
		};
	}
}
