#!/usr/bin/env python
#
# HOMR v0.1 - A plugin for Freevo 1.5/6.x
#
# Is is basicly written for helping me to keep track of my Simpson
# collection.
# It downloads the seasons from wikipedia including screenshots and
# information and shows you which seasons you have and which episodes.
#
# Written by Jonas Aaberg <cja@gmx.net> during the winter of 2007.
#
# -----------------------------------------------------------------------
# Freevo - A Home Theater PC framework
# Copyright (C) 2003 Krister Lagerstrom, et al. 
# Please see the file freevo/Docs/CREDITS for a complete list of authors.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of MER-
# CHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#
# ----------------------------------------------------------------------- */

import re, time, shelve, os, glob, datetime, urllib2, zlib, gzip, StringIO, random

# Freevo 
import plugin
import util
import menu
import config

from item import Item
from gui.PopupBox import PopupBox
from video import videoitem
from event import *

HOMR_DIR = config.SHARE_DIR + os.sep +  'images' + os.sep + 'HOMR'

Simpsons_episode_tag = re.compile("<td height=\"10\">(\d+) (-|[^a-z]*) (\d+)</td>\n"
                         "<td><a href=\"\S+\" title=\".*\">.*</a></td>\n"
                         "<td>(.*)</td>\n"
                         "(<td>.*</td>\n)?"
                         "<td>\"?<a href=\".*\" title=\".*\">(.*)</a>(.*)\"?</td>\n"
                         "<td( rowspan=\"\d+\")?>"
                         "(<a href=\".*\" class=\"image\" title=\".*\"><img src=\"(.*)\" alt=\".*\" width=\"\d+\" height=\"\d+\" longdesc=\".*\" /></a>)?</td>\n"
                         "</tr>\n"
                         "<tr( style=\"background-color: #FFFFFF?\")?>\n"
                         "<td (bgcolor=\"#FFFFFF\" )?colspan=\"\d+\" valign=\"top\"( height=\"\d+\")?>(.*)\n"
                         )

Simpsons_season_desc_tag = re.compile("<p><b>(<i>)?(.*)<a href=\"/wiki/The_Simpsons\" title=\"The Simpsons\">(.*)</p>")

Simpsons_season_image_tag = re.compile("<div class=\"thumbinner\" style=\"width:\d+px;\"><a href=\"/wiki/Image:.*\" class=\"internal\" title=\".*\">"
                              "<img src=\"(.*)\" alt=\".*\" width=\"\d+\" height=\"\d+\" longdesc=\"/wiki/Image:.*\" class=\"thumbimage\" /></a>")


HOMR_Data = {"The Simpsons": {"url":"http://en.wikipedia.org/wiki/The_Simpsons_(season_%d)",
                              "episode":Simpsons_episode_tag,
                              "season":Simpsons_season_desc_tag,
                              "season_image":Simpsons_season_image_tag,
                              "key":{"number":0, "id":2, "show":3, "name":5, "image":9, "description":13}                              
                              },
             }

IMAGES = True



class FetchSeason:
    def __init__(self, show, season_divider, storage):
        self.season = 1
        self.show = show
        self.storage = storage
        self.season_divider = season_divider
        self.popupbox = None
        
    def popup(self, msg):
        if self.popupbox != None:
            self.popupbox.destroy()
        self.popupbox = PopupBox(text=msg)
        self.popupbox.show()
        
    def download(self, url):
        data = ""
        for i in range(4):
            try:
                req = urllib2.Request(url)
                req.add_header("User-agent", "Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.7.5) Gecko/20041221")
                req.add_header("Accept", "text/xml,application/xml,application/xhtml+xml,text/html;q=0.9,text/plain;q=0.8,image/png,*/*;q=0.5")
                req.add_header("Accept-Language", "en-us,en;q=0.5")
                req.add_header("Accept-Encoding", "gzip,deflate")
                req.add_header("Accept-Charset", "ISO-8859-1,utf-8;q=0.7,*;q=0.7")
                connection = urllib2.urlopen(req)
                rawdata = connection.read()
                
                # You can look at the header string in connection, but this is easier :-)
                try:
                    data = zlib.decompress(rawdata)
                except:
                    try:
                        fh = StringIO.StringIO(rawdata)
                        cfh = gzip.GzipFile(fileobj=fh)
                        data = cfh.read()
                        cfh.close()
                        fh.close()
                    except:
                        data = rawdata
                connection.close()
                break
            except IOError:
                self.popup(_("Retrying fetching: %s in 20s") % url)
                time.sleep(20)
                pass
        else:
            self.popup(_("Failed to download season data"))
            raise
        return data
    
    def removehtml(self, data):
        d = StringIO.StringIO()
        depth = 0
        for i in range(len(data)):
            if data[i] == '<':
                depth += 1
                continue
            if data[i] == '>':
                depth -= 1
                continue
            if depth == 0:
                d.write(data[i])
        d.seek(0)
        newdata = d.read()
        d.close()
        return newdata
    
    def saveimage(self, image, path, filename):
        if not os.path.isdir(path):
            os.makedirs(path)
        fh = open(path + os.sep + filename, "w")
        fh.write(image)
        fh.close()
    
        
    def fetchseason(self, season):
        self.season = season
        
        self.popup(_("Fetching information for season %d...") % self.season)

        data = self.download(HOMR_Data[self.show]["url"] % self.season)

        season_desc = HOMR_Data[self.show]["season"].findall(data)
        season_image = HOMR_Data[self.show]["season_image"].findall(data)
        
        episodes = HOMR_Data[self.show]["episode"].findall(data)

        # [0] - Number
        # [2] - Id
        # [3] - Show
        # [5] - Name
        # [9] - image
        # [13] - Description

        
        NUMBER = HOMR_Data[self.show]["key"]["number"]
        ID = HOMR_Data[self.show]["key"]["id"]
        SHOW = HOMR_Data[self.show]["key"]["show"]
        NAME = HOMR_Data[self.show]["key"]["name"]
        IMAGE = HOMR_Data[self.show]["key"]["image"]
        DESCRIPTION = HOMR_Data[self.show]["key"]["description"]

        
        ok = True
        for i in range(len(episodes)):
            num = int(episodes[i][ID])
            if num / 100 != self.season:
                self.popup(_("ERROR: Wanted season %d, but this show is for season %d") % (self.season, num/100))
                print episodes[i][NUMBER],episodes[i][ID],episodes[i][SHOW],episodes[i][NAME]
                ok = False
            if num % (self.season*100) != i+1:
                self.popup(_("ERROR: Expected episode %d but got episode %d") % (i+1, num % (self.season*100)))
                print episodes[i][NUMBER],episodes[i][ID],episodes[i][SHOW],episodes[i][NAME]
                ok = False
        if ok:
            self.popup(_("Found %d episodes") % len(episodes))
        else:
            self.popup(_("ERROR: parsing episodes, continues as good as possible"))

        seasonid = self.season_divider % self.season
        
        self.storage[seasonid] = {}
        self.storage[seasonid]["season"] = self.season

        if len(season_desc) == 1:
            self.storage[seasonid]["description"] = self.removehtml(season_desc[0][1] +season_desc[0][2])
        else:
            self.storage[seasonid]["description"] = ""
            
        if len(season_image) == 1:
            imagefilename = seasonid + "." + season_image[0].split(".")[-1]
            
            self.storage[seasonid]["image"] = imagefilename
            self.popup(_("Fetching season image..."))
            if IMAGES:
                image = self.download(season_image[0])
                self.saveimage(image, config.FREEVO_CACHEDIR + os.sep + self.show , imagefilename)
        else:
            self.storage[seasonid]["image"] = ""
            
        self.storage[seasonid]["name"] = _("Season %d") % self.season
        for episode in episodes:

            if len(episode[IMAGE]) > 0:
                imagefilename = episode[SHOW]+"."+episode[IMAGE].split(".")[-1]
            else:
                imagefilename = None
            
            self.storage[seasonid][episode[NUMBER]] = {"number":episode[NUMBER],
                                                       "show":episode[SHOW],
                                                       "id":episode[ID],
                                                       "name":episode[NAME],
                                                       "image":imagefilename,
                                                       "description":self.removehtml(episode[DESCRIPTION])}
            if imagefilename:
                self.popup(_("Fetching image for show number %s season %d, episode %s...") % (episode[NUMBER], self.season, episode[ID]))
                if IMAGES:
                    image = self.download(episode[IMAGE])
                    self.saveimage(image, config.FREEVO_CACHEDIR + os.sep + self.show + os.sep + seasonid, imagefilename)
        self.popup(_("Finished!"))
        self.storage.sync()        
        self.popupbox.destroy()

        
class EpisodeItem(videoitem.VideoItem):
    def __init__(self, parent, master, seasonid, episodeid, season):

        videoitem.VideoItem.__init__(self, "", parent)
        self.master = master
        self.seasonid = seasonid
        self.episodeid = episodeid
        
        if self.master.storage[seasonid][episodeid]["image"] != None:
            self.image = config.FREEVO_CACHEDIR  + os.sep + self.master.name + os.sep + "season%.2d" % season + os.sep + self.master.storage[seasonid][episodeid]["image"]

        files = glob.glob(self.master.videopath + os.sep + (self.master.season_divider % season) + os.sep + self.master.storage[seasonid][episodeid]["show"] + "*")
        
        if len(files) == 1:
            self.filepath = files[0]
            self.set_url(self.filepath)
            self.icon = HOMR_DIR + os.sep + "on.png"
        else:
            self.filepath = None
            self.icon = HOMR_DIR + os.sep + "off.png"
            
        self.name = self.master.storage[seasonid][episodeid]["show"] + " - " +self.master.storage[seasonid][episodeid]["name"]
        if self.master.storage[seasonid][episodeid].has_key("lastseen"):
            self.description = _("Last seen: %s\n%s") % (self.master.storage[seasonid][episodeid]["lastseen"].strftime("%y-%m-%d"), self.master.storage[seasonid][episodeid]["description"])
        else:
            self.description = self.master.storage[seasonid][episodeid]["description"]

    def play(self, arg=None, menuw=None, alternateplayer=False):
        self.starttime = datetime.datetime.today()
        videoitem.VideoItem.play(self, arg, menuw, alternateplayer)

    def eventhandler(self, event, menuw=None):
        ret_val = videoitem.VideoItem.eventhandler(self, event, menuw)

        if event == USER_END or event == PLAY_END:
            if datetime.datetime.today() - self.starttime > datetime.timedelta(seconds=25):
                self.master.storage[self.seasonid][self.episodeid]["lastseen"] =  datetime.date.today()
                self.master.storage.sync()
                self.description = _("Last seen: %s\n%s") % (self.master.storage[self.seasonid][self.episodeid]["lastseen"].strftime("%y-%m-%d"),
                                                                  self.master.storage[self.seasonid][self.episodeid]["description"])



class RandomEpisode(EpisodeItem):
    def __init__(self, parent, master):
        videoitem.VideoItem.__init__(self, "", parent)
        self.image = None
        self.master = master
        self.name = _("Random Episode")
        self.description = _("Plays a random episode, that has not been showed in three days.")

        image = util.getimage(HOMR_DIR + os.sep + self.master.name)
        if image:
            self.image = image
        
    def play(self, arg=None, menuw=None, alternateplayer=False):

        episode_files = []
        
        l = self.master.storage.keys()
        l.sort()
        
        for k in l:
            if os.path.isdir(self.master.videopath + os.sep + k):
                for m in self.master.storage[k].keys():
                    if type(self.master.storage[k][m]) == dict:
                        if self.master.storage[k][m].has_key("lastseen"):
                            if (datetime.date.today() - self.master.storage[k][m]["lastseen"]) < datetime.timedelta(days=3):
                                continue
                        files = glob.glob(self.master.videopath + os.sep + k + os.sep + self.master.storage[k][m]["show"] + "*")
                        if len(files) == 1:
                            episode_files.append((files[0], k, m))
        print len(episode_files)
        val = random.randint(0, len(episode_files))

        self.filepath = episode_files[val][0]
        self.seasonid =  episode_files[val][1]
        self.episodeid =  episode_files[val][2]

        self.set_url(self.filepath)
        
        self.starttime = datetime.datetime.today()
        videoitem.VideoItem.play(self, arg, menuw, alternateplayer)
         


class SeasonItem(Item):
    def __init__(self, parent, master, seasonid):
        Item.__init__(self, parent)
        self.parent = parent
        self.seasonid = seasonid
        self.master = master
        self.localstorage = self.master.storage[seasonid]
        self.season = self.localstorage["season"]
        self.image = None
        
        if self.localstorage.has_key("image"):
            image = config.FREEVO_CACHEDIR + os.sep + self.master.name + os.sep + self.localstorage["image"]
            if not os.path.isdir(image):
                self.image = image
                
        if self.image == None:
            image = util.getimage(HOMR_DIR + os.sep + self.master.name)
            if image:
                self.image = image
            
        self.name = self.localstorage["name"]
        self.description = self.localstorage["description"]
        
    def actions(self):

        return [ ( self.play, 'Play' ) ]
    def play(self, arg=None, menuw=None):
        episode_items = []

        k = self.localstorage.keys()
        k.sort()

        for episodekey in k:
            if episodekey != "image" and episodekey != "name" and episodekey != "description" and episodekey != "season":
                episode_items.append(EpisodeItem(self.parent, self.master, self.seasonid, episodekey, self.season))

        season_menu = menu.Menu(self.name, episode_items)
        menuw.pushmenu(season_menu)
        menuw.refresh()


class ShowItem(Item):
    def __init__(self, parent, master):
        Item.__init__(self, parent)
        self.parent = parent
        self.master = master
        self.name = self.master.name

        image = util.getimage(HOMR_DIR + os.sep + self.master.name)
        if image:
            self.image = image

    def actions(self):
        self.master.storage.sync()        
        return [ ( self.play, 'Play' ) ]
    
    def play(self, arg=None, menuw=None):
        season_items = [RandomEpisode(self.parent, self.master)]
        
        # The simpsons are running on season 18, and it is a very very long running show.
        for i in range(1,50):
            if os.path.isdir(self.master.videopath + os.sep + self.master.season_divider % i):
                if not self.master.storage.has_key(self.master.season_divider % i):
                    f = FetchSeason(self.master.name, self.master.season_divider, self.master.storage)
                    f.fetchseason(i)

        l = self.master.storage.keys()
        l.sort()
        for k in l:
            if os.path.isdir(self.master.videopath + os.sep + k):
                season_items.append(SeasonItem(self.parent, self.master, k))

        season_menu = menu.Menu(self.name, season_items)
        menuw.pushmenu(season_menu)
        menuw.refresh()



class PluginInterface(plugin.MainMenuPlugin):
    """
    HOMR - A Simpson episode guide plugin

    This plugin provides organization for your Simpson episode collection.

    Add this to your local_conf.py file:
    --
    plugin.activate('HOMR', args=("The Simpsons", "/TV/shows/Simpsons", "season%.2d") )
    where: "/TV/shows/Simpsons" is the directory where your simpsons seasons are stored
    and "season%d" are how to find each season.(season01 and so on) use ("season%d" for
    "season1" storage order)
    """
    def __init__(self, name, videopath, season_divider):
        plugin.MainMenuPlugin.__init__(self)
        self.translation("HOMR")
        self.name = name
        self.videopath = videopath
        self.season_divider = season_divider

        try:
            self.storage = shelve.open(config.FREEVO_CACHEDIR + os.sep + self.name + ".HOMR", writeback=True)
        except:
            print _("ERROR: Failed to load/create shelve database for HOMR at:\"%s\" ") % (config.FREEVO_CACHEDIR + os.sep + self.name + ".HOMR")

    def items(self, parent):
        return [ ShowItem(parent, self) ]
    
