/*
 * $Id: chip_ite_8661F.c,v 1.57 2009-06-03 11:34:04 vrsieh Exp $
 *
 * Copyright (C) 2003-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

#include "fixme.h"
#include "glue-main.h"

#include "chip_ite_8661F.h"

struct cpssp {
	/*
	 * Config
	 */

	/*
	 * Signals
	 */
	unsigned int state_power;
	struct sig_boolean_or *port_int3;
	struct sig_boolean_or *port_int4;
	struct sig_boolean_or *port_int5;
	struct sig_boolean_or *port_int6;
	struct sig_boolean_or *port_int7;
	struct sig_boolean_or *port_int10;
	struct sig_boolean_or *port_int11;
	struct sig_isa_bus_dma *port_dma0;
	struct sig_isa_bus_dma *port_dma1;
	struct sig_isa_bus_dma *port_dma2;
	struct sig_isa_bus_dma *port_dma3;
	struct sig_shugart_bus *port_shugart_bus;
	struct sig_boolean_or *sig_shugart_bus_trk0;
	struct sig_boolean_or *sig_shugart_bus_dskchg;
	struct sig_parallel *port_par;
	struct sig_serial *port_ser0;
	struct sig_serial *port_ser1;

	/*
	 * State
	 */
	/* Floppy Disk Controller */
#define STATE
#define NAME		fdc
#define NAME_(x)	fdc_ ## x
#include "arch_fdc.c"
#undef NAME_
#undef NAME
#undef STATE

	/* Parallel Device */
#define STATE
#define NAME		par
#define NAME_(x)	par_ ## x
#include "arch_par.c"
#undef NAME_
#undef NAME
#undef STATE

	/* Serial Devices */
#define STATE
#define NAME		ser0
#define NAME_(x)	ser0_ ## x
#include "arch_ser.c"
#undef NAME_
#undef NAME
#define NAME		ser1
#define NAME_(x)	ser1_ ## x
#include "arch_ser.c"
#undef NAME_
#undef NAME
#undef STATE

	/* PnP */
	unsigned int fdc_irq;
	uint16_t fdc_port;
	unsigned int fdc_dma;

	unsigned int par_irq;
	uint16_t par_port;

	unsigned int ser0_irq;
	uint16_t ser0_port;
	unsigned int ser1_irq;
	uint16_t ser1_port;
};

static void
fdc_irq_set(struct cpssp *cpssp, unsigned int val)
{
	switch (cpssp->fdc_irq) {
	case 0:
		/* Do nothing... */
		break;
	case 3:
		sig_boolean_or_set(cpssp->port_int3, cpssp, val);
		break;
	case 4:
		sig_boolean_or_set(cpssp->port_int4, cpssp, val);
		break;
	case 5:
		sig_boolean_or_set(cpssp->port_int5, cpssp, val);
		break;
	case 6:
		sig_boolean_or_set(cpssp->port_int6, cpssp, val);
		break;
	case 7:
		sig_boolean_or_set(cpssp->port_int7, cpssp, val);
		break;
	case 10:
		sig_boolean_or_set(cpssp->port_int10, cpssp, val);
		break;
	case 11:
		sig_boolean_or_set(cpssp->port_int11, cpssp, val);
		break;
	default:
		fixme();
	}
}

static void
fdc_hds_set(struct cpssp *cpssp, unsigned int val)
{
	sig_shugart_bus_hds_set(cpssp->port_shugart_bus, cpssp, val);
}

static void
fdc_step_in(struct cpssp *cpssp)
{
	sig_shugart_bus_step_in(cpssp->port_shugart_bus, cpssp);
}

static void
fdc_step_out(struct cpssp *cpssp)
{
	sig_shugart_bus_step_out(cpssp->port_shugart_bus, cpssp);
}

static void
fdc_read_start(struct cpssp *cpssp)
{
	sig_shugart_bus_read_start(cpssp->port_shugart_bus, cpssp);
}

static void
fdc_writedata(struct cpssp *cpssp, unsigned char *buf, unsigned int bufsize)
{
	sig_shugart_bus_writedata(cpssp->port_shugart_bus, cpssp,
			buf, bufsize);
}

static void
fdc_motor_set(struct cpssp *cpssp, unsigned int drive, unsigned int val)
{
	sig_shugart_bus_motor_set(cpssp->port_shugart_bus, cpssp,
			drive, val);
}

static void
fdc_select_set(struct cpssp *cpssp, unsigned int drive, unsigned int val)
{
	sig_shugart_bus_select_set(cpssp->port_shugart_bus, cpssp,
			drive, val);
}

static void
fdc_dma_req(struct cpssp *cpssp)
{
	switch (cpssp->fdc_dma) {
	case 0:
		sig_isa_bus_dma_req(cpssp->port_dma0, cpssp);
		break;
	case 1:
		sig_isa_bus_dma_req(cpssp->port_dma1, cpssp);
		break;
	case 2:
		sig_isa_bus_dma_req(cpssp->port_dma2, cpssp);
		break;
	case 3:
		sig_isa_bus_dma_req(cpssp->port_dma3, cpssp);
		break;
	default:
		/* Do nothing... */
		break;
	}
}

static void
par_irq_set(struct cpssp *cpssp, unsigned int val)
{
	switch (cpssp->par_irq) {
	case 0:
		/* Do nothing... */
		break;
	case 3:
		sig_boolean_or_set(cpssp->port_int3, cpssp, val);
		break;
	case 4:
		sig_boolean_or_set(cpssp->port_int4, cpssp, val);
		break;
	case 5:
		sig_boolean_or_set(cpssp->port_int5, cpssp, val);
		break;
	case 6:
		sig_boolean_or_set(cpssp->port_int6, cpssp, val);
		break;
	case 7:
		sig_boolean_or_set(cpssp->port_int7, cpssp, val);
		break;
	case 10:
		sig_boolean_or_set(cpssp->port_int10, cpssp, val);
		break;
	case 11:
		sig_boolean_or_set(cpssp->port_int11, cpssp, val);
		break;
	default:
		fixme();
	}
}

static void
ser0_irq_set(struct cpssp *cpssp, unsigned int val)
{
	switch (cpssp->ser0_irq) {
	case 0:
		/* Do nothing... */
		break;
	case 3:
		sig_boolean_or_set(cpssp->port_int3, cpssp, val);
		break;
	case 4:
		sig_boolean_or_set(cpssp->port_int4, cpssp, val);
		break;
	case 5:
		sig_boolean_or_set(cpssp->port_int5, cpssp, val);
		break;
	case 6:
		sig_boolean_or_set(cpssp->port_int6, cpssp, val);
		break;
	case 7:
		sig_boolean_or_set(cpssp->port_int7, cpssp, val);
		break;
	case 10:
		sig_boolean_or_set(cpssp->port_int10, cpssp, val);
		break;
	case 11:
		sig_boolean_or_set(cpssp->port_int11, cpssp, val);
		break;
	default:
		fixme();
	}
}

static void
ser1_irq_set(struct cpssp *cpssp, unsigned int val)
{
	switch (cpssp->ser1_irq) {
	case 0:
		/* Do nothing... */
		break;
	case 3:
		sig_boolean_or_set(cpssp->port_int3, cpssp, val);
		break;
	case 4:
		sig_boolean_or_set(cpssp->port_int4, cpssp, val);
		break;
	case 5:
		sig_boolean_or_set(cpssp->port_int5, cpssp, val);
		break;
	case 6:
		sig_boolean_or_set(cpssp->port_int6, cpssp, val);
		break;
	case 7:
		sig_boolean_or_set(cpssp->port_int7, cpssp, val);
		break;
	case 10:
		sig_boolean_or_set(cpssp->port_int10, cpssp, val);
		break;
	case 11:
		sig_boolean_or_set(cpssp->port_int11, cpssp, val);
		break;
	default:
		fixme();
	}
}

static void
par_send(struct cpssp *cpssp, struct sig_parallel_msg *msg)
{
	sig_parallel_send(cpssp->port_par, cpssp, msg);
}

static void
ser0_send(struct cpssp *cpssp, uint8_t c)
{
	sig_serial_send(cpssp->port_ser0, cpssp, c);
}

static void
ser1_send(struct cpssp *cpssp, uint8_t c)
{
	sig_serial_send(cpssp->port_ser1, cpssp, c);
}

#include "arch_fdc.c"

#define BEHAVIOR

#define NAME		fdc
#define NAME_(x)	fdc_ ## x
#define SNAME		"fdc"
#include "arch_fdc.c"
#undef SNAME
#undef NAME_
#undef NAME

#define NAME		par
#define NAME_(x)	par_ ## x
#define SNAME		"par"
#include "arch_par.c"
#undef SNAME
#undef NAME_
#undef NAME

#define NAME		ser0
#define NAME_(x)	ser0_ ## x
#define SNAME		"ser0"
#include "arch_ser.c"
#undef SNAME
#undef NAME_
#undef NAME
#define NAME		ser1
#define NAME_(x)	ser1_ ## x
#define SNAME		"ser1"
#include "arch_ser.c"
#undef SNAME
#undef NAME_
#undef NAME

#undef BEHAVIOR

static int
chip_ite_8661F_inb(void *_cpssp, uint8_t *valp, uint16_t port)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (1 /* FIXME VOSSI */
	 && cpssp->fdc_port <= port
	 && port != cpssp->fdc_port + 6
	 && port < cpssp->fdc_port + 8) {
		/* Floppy Disk Controller */
		*valp = fdc_inb(cpssp, port & 0x7);
		return 0;
	}

	if (1 /* FIXME */
	 && cpssp->par_port <= port
	 && port < cpssp->par_port + 8) {
		/* Parallel Port */
		*valp = par_inb(cpssp, port & 0x7);
		return 0;
	}

	if (1 /* FIXME */
	 && cpssp->ser0_port <= port
	 && port < cpssp->ser0_port + 8) {
		/* Serial Port 0 */
		*valp = ser0_inb(cpssp, port & 0x7);
		return 0;
	}
	if (1 /* FIXME */
	 && cpssp->ser1_port <= port
	 && port < cpssp->ser1_port + 8) {
		/* Serial Port 1 */
		*valp = ser1_inb(cpssp, port & 0x7);
		return 0;
	}

	return -1;
}

static int
chip_ite_8661F_outb(void *_cpssp, uint8_t val, uint16_t port)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (1 /* FIXME VOSSI */
	 && cpssp->fdc_port <= port
	 && port != cpssp->fdc_port + 6
	 && port < cpssp->fdc_port + 8) {
		/* Floppy Disk Controller */
		fdc_outb(cpssp, val, port & 0x7);
		return 0;
	}

	if (1 /* FIXME */
	 && cpssp->par_port <= port
	 && port < cpssp->par_port + 8) {
		/* Parallel Port */
		par_outb(cpssp, val, port & 0x7);
		return 0;
	}

	if (1 /* FIXME */
	 && cpssp->ser0_port <= port
	 && port < cpssp->ser0_port + 8) {
		/* Serial Port 0 */
		ser0_outb(cpssp, val, port & 0x7);
		return 0;

	}
	if (1 /* FIXME */
	 && cpssp->ser1_port <= port
	 && port < cpssp->ser1_port + 8) {
		/* Serial Port 1 */
		ser1_outb(cpssp, val, port & 0x7);
		return 0;
	}

	return -1;
}

static int
chip_ite_8661F_ack0_inb(void *_cpssp, unsigned int tc, unsigned char *valp)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (cpssp->fdc_dma == 0) {
		fdc_ack_inb(cpssp, tc, valp);
		return 0;
	}
	return 1;
}

static int
chip_ite_8661F_ack0_outb(void *_cpssp, unsigned int tc, unsigned char val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (cpssp->fdc_dma == 0) {
		fdc_ack_outb(cpssp, tc, val);
		return 0;
	}
	return 1;
}

static int
chip_ite_8661F_ack0_verifyb(void *_cpssp, unsigned int tc)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (cpssp->fdc_dma == 0) {
		fdc_ack_verifyb(cpssp, tc);
		return 0;
	}
	return 1;
}

static int
chip_ite_8661F_ack1_inb(void *_cpssp, unsigned int tc, unsigned char *valp)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (cpssp->fdc_dma == 1) {
		fdc_ack_inb(cpssp, tc, valp);
		return 0;
	}
	return 1;
}

static int
chip_ite_8661F_ack1_outb(void *_cpssp, unsigned int tc, unsigned char val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (cpssp->fdc_dma == 1) {
		fdc_ack_outb(cpssp, tc, val);
		return 0;
	}
	return 1;
}

static int
chip_ite_8661F_ack1_verifyb(void *_cpssp, unsigned int tc)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (cpssp->fdc_dma == 1) {
		fdc_ack_verifyb(cpssp, tc);
		return 0;
	}
	return 1;
}

static int
chip_ite_8661F_ack2_inb(void *_cpssp, unsigned int tc, unsigned char *valp)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (cpssp->fdc_dma == 2) {
		fdc_ack_inb(cpssp, tc, valp);
		return 0;
	}
	return 1;
}

static int
chip_ite_8661F_ack2_outb(void *_cpssp, unsigned int tc, unsigned char val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (cpssp->fdc_dma == 2) {
		fdc_ack_outb(cpssp, tc, val);
		return 0;
	}
	return 1;
}

static int
chip_ite_8661F_ack2_verifyb(void *_cpssp, unsigned int tc)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (cpssp->fdc_dma == 2) {
		fdc_ack_verifyb(cpssp, tc);
		return 0;
	}
	return 1;
}

static int
chip_ite_8661F_ack3_inb(void *_cpssp, unsigned int tc, unsigned char *valp)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (cpssp->fdc_dma == 3) {
		fdc_ack_inb(cpssp, tc, valp);
		return 0;
	}
	return 1;
}

static int
chip_ite_8661F_ack3_outb(void *_cpssp, unsigned int tc, unsigned char val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (cpssp->fdc_dma == 3) {
		fdc_ack_outb(cpssp, tc, val);
		return 0;
	}
	return 1;
}

static int
chip_ite_8661F_ack3_verifyb(void *_cpssp, unsigned int tc)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (cpssp->fdc_dma == 3) {
		fdc_ack_verifyb(cpssp, tc);
		return 0;
	}
	return 1;
}

static void
chip_ite_8661F_trk0_set(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	fdc_trk0_set(cpssp, val);
}

static void
chip_ite_8661F_index_set(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	fdc_index_set(cpssp, val);
}

static void
chip_ite_8661F_wp_set(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	fdc_wp_set(cpssp, val);
}

static void
chip_ite_8661F_dskchg_set(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	fdc_dskchg_set(cpssp, val);
}

static void
chip_ite_8661F_readid(void *_cpssp, unsigned char cyl, unsigned char sec)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	fdc_readid(cpssp, cyl, sec);
}

static void
chip_ite_8661F_readdata(void *_cpssp, unsigned char *buf, unsigned int bufsize)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	fdc_readdata(cpssp, buf, bufsize);
}

static void
chip_ite_8661F_par_recv(void *_cpssp, struct sig_parallel_msg *msg)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (! cpssp->state_power) {
		return;
	}

	par_recv(cpssp, msg);
}

static void
chip_ite_8661F_ser0_recv(void *_cpssp, uint8_t c)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (! cpssp->state_power) {
		return;
	}

	ser0_recv(cpssp, c);
}

static void
chip_ite_8661F_ser1_recv(void *_cpssp, uint8_t c)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (! cpssp->state_power) {
		return;
	}

	ser1_recv(cpssp, c);
}

static void
chip_ite_8661F_power_set(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	cpssp->state_power = val;
}

static void
chip_ite_8661F_n_reset_set(void *_cpssp, unsigned int n_val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	fdc_reset(cpssp);
	par_reset(cpssp);
	ser0_reset(cpssp);
	ser1_reset(cpssp);

	/* FIXME */
	if (1) {
		cpssp->fdc_irq = 6;
		cpssp->fdc_port = 0x03f0;
		cpssp->fdc_dma = 2;
	} else {
		cpssp->fdc_irq = 0;
		cpssp->fdc_port = 0;
		cpssp->fdc_dma = 4;
	}

	/* FIXME */
	if (1) {
		cpssp->par_irq = 7;
		cpssp->par_port = 0x0378;
	} else {
		cpssp->par_irq = 0;
		cpssp->par_port = 0;
	}

	/* FIXME */
	if (1) {
		cpssp->ser0_irq = 4;
		cpssp->ser0_port = 0x03f8;
	} else {
		cpssp->ser0_irq = 0;
		cpssp->ser0_port = 0;
	}
	/* FIXME */
	if (1) {
		cpssp->ser1_irq = 3;
		cpssp->ser1_port = 0x02f8;
	} else {
		cpssp->ser1_irq = 0;
		cpssp->ser1_port = 0;
	}
}

void *
chip_ite_8661F_create(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_boolean *port_power,
	struct sig_boolean *port_reset_hash_,
	struct sig_isa_bus_main *port_isa_bus,
	struct sig_boolean_or *port_int3,
	struct sig_boolean_or *port_int4,
	struct sig_boolean_or *port_int5,
	struct sig_boolean_or *port_int6,
	struct sig_boolean_or *port_int7,
	struct sig_boolean_or *port_int10,
	struct sig_boolean_or *port_int11,
	struct sig_isa_bus_dma *port_dma0,
	struct sig_isa_bus_dma *port_dma1,
	struct sig_isa_bus_dma *port_dma2,
	struct sig_isa_bus_dma *port_dma3,
	struct sig_shugart_bus *port_shugart_bus,
	struct sig_parallel *port_par,
	struct sig_serial *port_ser0,
	struct sig_serial *port_ser1
)
{
	static const struct sig_boolean_funcs power_funcs = {
		.set = chip_ite_8661F_power_set,
	};
	static const struct sig_boolean_funcs reset_hash__funcs = {
		.set = chip_ite_8661F_n_reset_set,
	};
	static const struct sig_isa_bus_main_funcs isa_bus_funcs = {
		.inb = chip_ite_8661F_inb,
		.outb = chip_ite_8661F_outb,
	};
	static const struct sig_isa_bus_dma_funcs dma0_funcs = {
		.ack_inb = chip_ite_8661F_ack0_inb,
		.ack_outb = chip_ite_8661F_ack0_outb,
		.ack_verifyb = chip_ite_8661F_ack0_verifyb,
	};
	static const struct sig_isa_bus_dma_funcs dma1_funcs = {
		.ack_inb = chip_ite_8661F_ack1_inb,
		.ack_outb = chip_ite_8661F_ack1_outb,
		.ack_verifyb = chip_ite_8661F_ack1_verifyb,
	};
	static const struct sig_isa_bus_dma_funcs dma2_funcs = {
		.ack_inb = chip_ite_8661F_ack2_inb,
		.ack_outb = chip_ite_8661F_ack2_outb,
		.ack_verifyb = chip_ite_8661F_ack2_verifyb,
	};
	static const struct sig_isa_bus_dma_funcs dma3_funcs = {
		.ack_inb = chip_ite_8661F_ack3_inb,
		.ack_outb = chip_ite_8661F_ack3_outb,
		.ack_verifyb = chip_ite_8661F_ack3_verifyb,
	};
	static const struct sig_boolean_or_funcs shugart_trk0_funcs = {
		.set = chip_ite_8661F_trk0_set,
	};
	static const struct sig_boolean_or_funcs shugart_dskchg_funcs = {
		.set = chip_ite_8661F_dskchg_set,
	};
	static const struct sig_shugart_bus_funcs shugart_funcs = {
		.index_set = chip_ite_8661F_index_set,
		.wp_set = chip_ite_8661F_wp_set,
		.readid = chip_ite_8661F_readid,
		.readdata = chip_ite_8661F_readdata,
	};
	static const struct sig_parallel_funcs par_funcs = {
		.recv = chip_ite_8661F_par_recv,
	};
	static const struct sig_serial_funcs ser0_funcs = {
		.recv = chip_ite_8661F_ser0_recv,
	};
	static const struct sig_serial_funcs ser1_funcs = {
		.recv = chip_ite_8661F_ser1_recv,
	};
	struct cpssp *cpssp;

	cpssp = malloc(sizeof(*cpssp));
	assert(cpssp);

	fdc_init(cpssp);
	par_init(cpssp);
	ser0_init(cpssp);
	ser1_init(cpssp);

	/* Out */
	cpssp->port_int3 = port_int3;
	sig_boolean_or_connect_out(port_int3, cpssp, 0);

	cpssp->port_int4 = port_int4;
	sig_boolean_or_connect_out(port_int4, cpssp, 0);

	cpssp->port_int5 = port_int5;
	sig_boolean_or_connect_out(port_int5, cpssp, 0);

	cpssp->port_int6 = port_int6;
	sig_boolean_or_connect_out(port_int6, cpssp, 0);

	cpssp->port_int7 = port_int7;
	sig_boolean_or_connect_out(port_int7, cpssp, 0);

	cpssp->port_int10 = port_int10;
	sig_boolean_or_connect_out(port_int10, cpssp, 0);

	cpssp->port_int11 = port_int11;
	sig_boolean_or_connect_out(port_int11, cpssp, 0);

	/* Call */
	sig_isa_bus_main_connect(port_isa_bus, cpssp, &isa_bus_funcs);

	cpssp->port_dma0 = port_dma0;
	sig_isa_bus_dma_connect(port_dma0, cpssp, &dma0_funcs);

	cpssp->port_dma1 = port_dma1;
	sig_isa_bus_dma_connect(port_dma1, cpssp, &dma1_funcs);

	cpssp->port_dma2 = port_dma2;
	sig_isa_bus_dma_connect(port_dma2, cpssp, &dma2_funcs);

	cpssp->port_dma3 = port_dma3;
	sig_isa_bus_dma_connect(port_dma3, cpssp, &dma3_funcs);

	cpssp->sig_shugart_bus_trk0 = port_shugart_bus->trk0;
	sig_boolean_or_connect_in(port_shugart_bus->trk0, cpssp,
			&shugart_trk0_funcs);
	cpssp->sig_shugart_bus_dskchg = port_shugart_bus->dskchg;
	sig_boolean_or_connect_in(port_shugart_bus->dskchg, cpssp,
			&shugart_dskchg_funcs);
	cpssp->port_shugart_bus = port_shugart_bus;
	sig_shugart_bus_connect_controller(port_shugart_bus, cpssp, &shugart_funcs);

	cpssp->port_par = port_par;
	sig_parallel_connect(port_par, cpssp, &par_funcs);

	cpssp->port_ser0 = port_ser0;
	sig_serial_connect(port_ser0, cpssp, &ser0_funcs);

	cpssp->port_ser1 = port_ser1;
	sig_serial_connect(port_ser1, cpssp, &ser1_funcs);

	/* In */
	cpssp->state_power = 0;
	sig_boolean_connect_in(port_power, cpssp, &power_funcs);

	sig_boolean_connect_in(port_reset_hash_, cpssp, &reset_hash__funcs);

	return cpssp;
}

void
chip_ite_8661F_destroy(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	free(cpssp);
}
