/*
 * $Id: sig_sound.c,v 1.45 2009-05-21 16:38:09 vrsieh Exp $
 *
 * Copyright (C) 2004-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

#include "fixme.h"
#include "glue-main.h"

#include "sig_sound.h"

static void
_sig_sound_send(struct sig_sound *b, void *s, int16_t *samples)
{
	unsigned int nr;

	for (nr = 0; nr < b->nmembers; nr++) {
		if (! b->member[nr].f
		 || ! b->member[nr].f->samples_set
		 || b->member[nr].s == s) {
			continue;
		}
		(b->member[nr].f->samples_set)(b->member[nr].s, samples);
	}
}

static void
sig_sound_event(void *_b)
{
	struct sig_sound *b = (struct sig_sound *) _b;

	const unsigned long long delta = TIME_HZ / SIG_SOUND_RATE;
	const unsigned long long now = time_virt();

	while (b->time < now) {
		int16_t val;

		if (! b->on) {
			/* Generate '0'. */
			val = 0x8000;
		} else if (b->period == (unsigned long long) -1) {
			/* Generate '1'. */
			val = 0x7fff;
		} else {
			/* Use frequency. */
			if (b->phase < b->period / 2) {
				/* Generate '0'. */
				val = 0x8000;
			} else {
				/* Generate '1'. */
				val = 0x7fff;
			}

			b->phase += delta;
			while (b->period <= b->phase) {
				b->phase -= b->period;
			}
		}
		b->sample[b->nsamples * 2 + 0] = val;
		b->sample[b->nsamples * 2 + 1] = val;
		b->nsamples++;

		if (SIG_SOUND_RATE / SIG_SOUND_PROCESS_FREQ <= b->nsamples) {
			/* (void *) 0 - FIXME */
			_sig_sound_send(b, (void *) 0, b->sample);
			b->nsamples = 0;
		}

		b->time += delta;
	}

	time_call_at(b->time + TIME_HZ / SIG_SOUND_PROCESS_FREQ,
			sig_sound_event, b);
}

void
sig_sound_attr_set(
	struct sig_sound *b,
	void *s,
	unsigned int on,
	unsigned long long period
)
{
	if (! b->attr_mode) {
		/* Switch to attribute mode. */
		b->attr_mode = 1;

		b->time = time_virt();

		b->phase = 0;
		b->nsamples = 0;

		time_call_at(b->time + TIME_HZ / SIG_SOUND_PROCESS_FREQ,
				sig_sound_event, b);
	}

	b->on = on;
	b->period = period;
}

void
sig_sound_samples_set(
	struct sig_sound *b,
	void *s,
	int16_t *samples
)
{
	assert(! b->attr_mode);

	_sig_sound_send(b, s, samples);
}

void
sig_sound_connect(
	struct sig_sound *b,
	void *s,
	const struct sig_sound_funcs *f
)
{
	assert(b);
	assert(b->type == SIG_GEN_SOUND);
	assert(b->nmembers < sizeof(b->member) / sizeof(b->member[0]));

	b->member[b->nmembers].s = s;
	b->member[b->nmembers].f = f;
	b->nmembers++;
}

struct sig_sound *
sig_sound_create(const char *name)
{
	struct sig_sound *b;

	b = malloc(sizeof(*b));
	assert(b);

	b->type = SIG_GEN_SOUND;
	b->attr_mode = 0;
	b->nmembers = 0;

	return b;
}

void
sig_sound_destroy(struct sig_sound *sig)
{
	assert(sig);
	assert(sig->type == SIG_GEN_SOUND);

	free(sig);
}
