/*
 * dsyslog - a dumb syslog (e.g. syslog for people who have a clue)
 * Copyright (c) 2008 William Pitcock <nenolod@sacredspiral.co.uk>
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice is present in all copies.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <glib.h>

#include <stdio.h>
#include <stdlib.h>

#include <sys/types.h>
#include <dirent.h>

#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/poll.h>
#include <sys/stat.h>

#include <errno.h>

#include <unistd.h>

#ifndef __DSYSLOG_INCLUDE_RECURSION_GUARD
#define __DSYSLOG_INCLUDE_RECURSION_GUARD

#include "sysconf.h"
#include "confparse.h"

typedef struct dsyslog_source_ dsyslog_source_t;

typedef void (*dsyslog_source_constructor_t)(dsyslog_source_t *src);
typedef void (*dsyslog_source_destructor_t)(dsyslog_source_t *src);

/* A syslog source. */
struct dsyslog_source_ {
	gchar *type;
	gchar *path;
	gchar *host;
	gint port;
	gpointer opaque;
	dsyslog_source_destructor_t destructor;
};

extern void dsyslog_source_register(gchar *type, dsyslog_source_constructor_t func);
extern void dsyslog_source_unregister(gchar *type);
extern void dsyslog_source_setup(dsyslog_source_t *src);
extern void dsyslog_source_remove(dsyslog_source_t *src);
extern void dsyslog_source_clear(void);

/* A syslog event. */
typedef struct {
	gulong uid;				/* a message id ... doesn't repeat too often */
	gint logcode;				/* syslog-style logcode, if -1, ignored */
	gchar *datestamp;			/* syslog-style datestamp */
	gchar *source;				/* source of message */
	gchar *program;				/* program sending message */
	gchar *message;				/* message */
} dsyslog_event_t;

typedef gboolean (*dsyslog_message_handler_t)(dsyslog_event_t *event);

extern void dsyslog_event_dispatch(gint logcode, gchar *datestamp, gchar *source, gchar *program, gchar *message);
extern void dsyslog_event_delete(dsyslog_event_t *event);
extern void dsyslog_event_init(void);

/* A syslog filter. */
typedef struct dsyslog_filter_ dsyslog_filter_t;
typedef gboolean (*dsyslog_filter_handler_t)(dsyslog_event_t *event, dsyslog_filter_t *filter);
typedef void (*dsyslog_filter_destructor_t)(dsyslog_filter_t *filter);

struct dsyslog_filter_ {
	gchar *type;
	gchar *search;
	gchar *replace;
	gchar *program;
	gchar *message;
	gint logcode;
	gpointer opaque;
	dsyslog_filter_handler_t handler;
	dsyslog_filter_destructor_t destructor;
	GList *conditions;
};

extern void dsyslog_filter_type_register(gchar *type, dsyslog_filter_handler_t hdl);
extern void dsyslog_filter_type_unregister(gchar *type);
extern void dsyslog_filter_add(dsyslog_filter_t *filter);
extern void dsyslog_filter_remove(dsyslog_filter_t *filter);
extern gboolean dsyslog_filter_process(dsyslog_event_t *event);
extern void dsyslog_filter_clear(void);

/* A syslog output. */
typedef struct dsyslog_output_ dsyslog_output_t;
typedef void (*dsyslog_output_handler_t)(dsyslog_event_t *event, dsyslog_output_t *output);
typedef void (*dsyslog_output_destructor_t)(dsyslog_output_t *output);

struct dsyslog_output_ {
	gchar *type;
	gchar *path;
	gchar *host;
	gint port;
	gchar *dbuser;
	gchar *dbpass;
	gchar *dbhost;
	gchar *dbname;
	gint dbport;
	gpointer opaque;
	dsyslog_output_handler_t handler;
	dsyslog_output_destructor_t destructor;
	GList *conditions;
};

extern void dsyslog_output_type_register(gchar *type, dsyslog_output_handler_t hdl);
extern void dsyslog_output_type_unregister(gchar *type);
extern void dsyslog_output_add(dsyslog_output_t *output);
extern void dsyslog_output_remove(dsyslog_output_t *output);
extern void dsyslog_output_process(dsyslog_event_t *event);
extern void dsyslog_output_clear(void);

/* Logging conditionals. */
typedef struct dsyslog_cond_ dsyslog_cond_t;
typedef gboolean (*dsyslog_cond_eval_t)(dsyslog_event_t *event, dsyslog_cond_t *cond);
typedef void (*dsyslog_cond_destructor_t)(dsyslog_cond_t *cond);

struct dsyslog_cond_ {
	gchar *type;
	gchar *cond;
	gchar *value;
	gpointer opaque;
	dsyslog_cond_eval_t evaluator;
	dsyslog_cond_destructor_t destructor;
};

extern void dsyslog_cond_type_register(gchar *type, dsyslog_cond_eval_t hdl);
extern void dsyslog_cond_type_unregister(gchar *type);
extern GList *dsyslog_cond_add(GList *conditions, dsyslog_cond_t *cond);
extern GList *dsyslog_cond_remove(GList *conditions, dsyslog_cond_t *cond);
extern gboolean dsyslog_cond_process(dsyslog_event_t *event, GList *conditions);

/* module stuff */
typedef struct {
	gpointer handle;
	void (*modinit)();
	void (*moddeinit)();
} dsyslog_module_t;

/* module loader routine. */
extern void dsyslog_module_load(const char *path);
extern void dsyslog_module_unload(dsyslog_module_t *module);
extern void dsyslog_module_clear(void);

/* severity stuff */
extern guint dsyslog_facility_to_logcode(const gchar *facility);
extern guint dsyslog_severity_to_logcode(const gchar *facility);

#define DSYSLOG_SEVMASK		0x07
#define DSYSLOG_FACMASK		0x03f8
#define DSYSLOG_GET_SEVERITY(p) ((p) & DSYSLOG_SEVMASK)
#define DSYSLOG_GET_FACILITY(p)	(((p) & DSYSLOG_FACMASK) >> 3)

/* handy debugging macros. */
#define _DMSG(tag, msg, ...) fprintf(stderr, "(%-5s) %s:%d [%s]: " msg "\n", tag, __FILE__, __LINE__, __PRETTY_FUNCTION__, ##__VA_ARGS__)

#ifndef NDEBUG
# define _DEBUG(...)         _DMSG("DEBUG", __VA_ARGS__)
#else
# define _DEBUG(...)         {}
#endif

#define _WARN(...)           _DMSG("WARN", __VA_ARGS__)
#define _INFO(...)           _DMSG("INFO", __VA_ARGS__)
#define _ERROR(...)          _DMSG("ERROR", __VA_ARGS__)
#define _ENTER		     _DEBUG("enter")
#define _LEAVE               _DEBUG("leave"); return 

#endif
