/*
** Copyright 2007-2009 Double Precision, Inc.  See COPYING for
** distribution information.
*/

/*
** $Id: msg2html.c,v 1.9 2009/11/18 03:38:50 mrsam Exp $
*/

#include "msg2html.h"
#include "unicode/unicode.h"
#include "numlib/numlib.h"
#include "gpglib/gpglib.h"
#include "cgi/cgi.h"
#include "rfc822/rfc2047.h"
#include "rfc2045/rfc2646.h"
#include "rfc2646html.h"
#include "md5/md5.h"

#include "filter.h"
#include "html.h"

#include <ctype.h>

static void (*get_known_handler(struct rfc2045 *mime,
				struct msg2html_info *info))
	(FILE *, struct rfc2045 *, struct rfc2045id *, struct msg2html_info *);

static void (*get_handler(struct rfc2045 *mime,
			  struct msg2html_info *info))
	(FILE *, struct rfc2045 *,
	 struct rfc2045id *,
	 struct msg2html_info *);

static void addbuf(int c, char **buf, size_t *bufsize, size_t *buflen)
{
	if (*buflen == *bufsize)
	{
		char	*newbuf= *buf ?
			realloc(*buf, *bufsize+512):malloc(*bufsize+512);

		if (!newbuf)
			return;
		*buf=newbuf;
		*bufsize += 512;
	}
	(*buf)[(*buflen)++]=c;
}

static char *get_next_header(FILE *fp, char **value,
			     int preserve_nl,
			     off_t *mimepos, const off_t *endpos)
{
	int	c;
	int	eatspaces=0;

	size_t bufsize=256;
	char *buf=malloc(bufsize);
	size_t buflen=0;

	if (!buf)
		return NULL;

	if (mimepos && *mimepos >= *endpos)	return (NULL);

	while (mimepos == 0 || *mimepos < *endpos)
	{
		if ((c=getc(fp)) != '\n' && c >= 0)
		{
			if (c != ' ' && c != '\t' && c != '\r')
				eatspaces=0;

			if (!eatspaces)
				addbuf(c, &buf, &bufsize, &buflen);
			if (mimepos)	++ *mimepos;
			continue;
		}
		if ( c == '\n' && mimepos)	++ *mimepos;
		if (buflen == 0)	return (0);
		if (c < 0)	break;
		c=getc(fp);
		if (c >= 0)	ungetc(c, fp);
		if (c < 0 || c == '\n' || !isspace(c))	break;
		addbuf(preserve_nl ? '\n':' ', &buf, &bufsize, &buflen);
		if (!preserve_nl)
			eatspaces=1;
	}
	addbuf(0, &buf, &bufsize, &buflen);
	buf[buflen-1]=0;  /* Make sure, in outofmem situations */

	for ( *value=buf; **value; (*value)++)
	{
		if (**value == ':')
		{
			**value='\0';
			++*value;
			break;
		}
		**value=tolower(**value);
	}
	while (**value && isspace((int)(unsigned char)**value))	++*value;
	return(buf);
}

struct msg2html_info *msg2html_alloc(const char *charset)
{
	struct msg2html_info *p=malloc(sizeof(struct msg2html_info));

	if (!p)
		return NULL;
	memset(p, 0, sizeof(*p));

	p->output_character_set=charset;
	return p;
}

void msg2html_add_smiley(struct msg2html_info *i,
			 const char *txt, const char *imgurl)
{
	char buf[2];
	struct msg2html_smiley_list *l;

	buf[0]=*txt;
	buf[1]=0;

	if (strlen(i->smiley_index) < sizeof(i->smiley_index)-1)
		strcat(i->smiley_index, buf);


	if ((l=malloc(sizeof(struct msg2html_smiley_list))) != NULL)
	{
		if ((l->code=strdup(txt)) != NULL)
		{
			if ((l->url=strdup(imgurl)) != NULL)
			{
				l->next=i->smileys;
				i->smileys=l;
				return;
			}
			free(l->code);
		}
		free(l);
	}
}

void msg2html_free(struct msg2html_info *p)
{
	struct msg2html_smiley_list *sl;

	while ((sl=p->smileys) != NULL)
	{
		p->smileys=sl->next;
		free(sl->code);
		free(sl->url);
		free(sl);
	}
	free(p);
}

static void html_escape(const char *p, size_t n)
{
	char	buf[10];
	const	char *q=p;

	while (n)
	{
		--n;
		if (*p == '<')	strcpy(buf, "&lt;");
		else if (*p == '>') strcpy(buf, "&gt;");
		else if (*p == '&') strcpy(buf, "&amp;");
		else if (*p == ' ') strcpy(buf, "&nbsp;");
		else if (*p == '\n') strcpy(buf, "<br />");
		else if ((unsigned char)(*p) < ' ')
			sprintf(buf, "&#%d;", (int)(unsigned char)*p);
		else
		{
			p++;
			continue;
		}

		fwrite(q, p-q, 1, stdout);
		printf("%s", buf);
		p++;
		q=p;
	}
	fwrite(q, p-q, 1, stdout);
}

/*
** Consider header name: all lowercase, except the very first character,
** and the first character after every "-"
*/

static void header_uc(char *h)
{
	while (*h)
	{
		*h=toupper( (int)(unsigned char) *h);
		++h;
		while (*h)
		{
			*h=tolower((int)(unsigned char) *h);
			if (*h++ == '-')	break;
		}
	}
}

static void show_email_header(const char *h)
{
	html_escape(h, strlen(h));
}

static void print_header_uc(struct msg2html_info *info, char *h)
{	
	header_uc(h);

	printf("<tr valign=\"baseline\"><th align=\"right\" class=\"message-rfc822-header-name\">");

	if (info->email_header)
		(*info->email_header)(h, show_email_header);
	else
		show_email_header(h);
	printf(":<span class=\"tt\">&nbsp;</span></th>");

}

struct showaddrinfo {
	struct msg2html_info *info;
	struct rfc822a *a;
	int curindex;
	int isfirstchar;
} ;

static void showaddressheader_printc(char c, void *p)
{
	struct showaddrinfo *sai= (struct showaddrinfo *)p;

	if (sai->isfirstchar)
	{
		char *name=0;
		char *addr=0;

		if (sai->curindex < sai->a->naddrs &&
		    sai->a->addrs[sai->curindex].tokens)
		{
			name=rfc822_display_name_tobuf(sai->a,
						       sai->curindex,
						       sai->info->
						       output_character_set);
			addr=rfc822_display_addr_tobuf(sai->a,
						       sai->curindex,
						       sai->info->
						       output_character_set);
		}

		if (sai->info->email_address_start)
			(*sai->info->email_address_start)(name, addr);

		if (addr)
			free(addr);
		if (name)
			free(name);

		sai->isfirstchar=0;
	}

	html_escape(&c, 1);
}

static void showaddressheader_printsep(const char *sep, void *p)
{
	struct showaddrinfo *sai= (struct showaddrinfo *)p;

	if (sai && !sai->isfirstchar)
		printf("</span>");

	if (sai->info->email_address_end)
		(*sai->info->email_address_end)();

	if (sai)
	{
		sai->curindex++;
		sai->isfirstchar=1;
	}

	printf("%s<span class=\"message-rfc822-header-contents\">", sep);
}

static void showaddressheader_printsep_plain(const char *sep, void *p)
{
	printf("%s", sep);
}

static void showmsgrfc822_addressheader(struct msg2html_info *info,
					const char *p)
{
	struct	rfc822t *rfcp;
	struct  rfc822a *rfca;

	struct showaddrinfo sai;

	rfcp=rfc822t_alloc_new(p, NULL, NULL);
	if (!rfcp)
		return;

	rfca=rfc822a_alloc(rfcp);
	if (!rfca)
	{
		rfc822t_free(rfcp);
		return;
	}

	sai.info=info;
	sai.a=rfca;
	sai.curindex=0;
	sai.isfirstchar=1;

	rfc2047_print_unicodeaddr(rfca, info->output_character_set,
				  showaddressheader_printc,
				  showaddressheader_printsep, &sai);
	if (!sai.isfirstchar)
		showaddressheader_printsep("", &sai);
	/* This closes the final </a> */


	rfc822a_free(rfca);
	rfc822t_free(rfcp);
}

static void showrfc2369_printheader(char c, void *p)
{
	p=p;
	putchar(c);
}

static void showmsgrfc2369_header(struct msg2html_info *info, const char *p)
{
struct	rfc822t *rfcp;
struct  rfc822a *rfca;
int	i;

	rfcp=rfc822t_alloc_new(p, NULL, NULL);
	if (!rfcp)
		return;

	rfca=rfc822a_alloc(rfcp);
	if (!rfca)
	{
		rfc822t_free(rfcp);
		return;
	}

	for (i=0; i<rfca->naddrs; i++)
	{
		char	*p=rfc822_getaddr(rfca, i);
		char	*q=info->get_textlink ?
			(*info->get_textlink)(p, info->arg):NULL;

		if (q && rfca->addrs[i].tokens)
		{
			rfca->addrs[i].tokens->token=0;
			if (*q)
				free(p);
			else
			{
			struct	buf b;

				buf_init(&b);
				free(q);
				for (q=p; *q; q++)
				{
				char	c[2];

					switch (*q)	{
					case '<':
						buf_cat(&b, "&lt;");
						break;
					case '>':
						buf_cat(&b, "&gt;");
						break;
					case '&':
						buf_cat(&b, "&amp;");
						break;
					case ' ':
						buf_cat(&b, "&nbsp;");
						break;
					default:
						c[1]=0;
						c[0]=*q;
						buf_cat(&b, c);
						break;
					}
				}
				free(p);
				q=strdup(b.ptr ? b.ptr:"");
				buf_free(&b);
			}
			rfca->addrs[i].tokens->ptr=q;
			rfca->addrs[i].tokens->len=q ? strlen(q):0;
			rfca->addrs[i].tokens->next=0;
		}
		else
		{
			if (q)
				free(q);
			free(p);
		}
	}

	rfc822_print(rfca, showrfc2369_printheader,
				showaddressheader_printsep_plain, NULL);
	for (i=0; i<rfca->naddrs; i++)
		if (rfca->addrs[i].tokens)
			free((char *)rfca->addrs[i].tokens->ptr);

	rfc822a_free(rfca);
	rfc822t_free(rfcp);
}

static int isaddressheader(const char *header)
{
	return (strcmp(header, "to") == 0 ||
		strcmp(header, "cc") == 0 ||
		strcmp(header, "from") == 0 ||
		strcmp(header, "sender") == 0 ||
		strcmp(header, "resent-to") == 0 ||
		strcmp(header, "resent-cc") == 0 ||
		strcmp(header, "reply-to") == 0);
}


static void showmsgrfc822_headerp(const char *p, size_t l, void *dummy)
{
	if (fwrite(p, l, 1, stdout) != 1)
	    ; /* ignore */
}

static void showmsgrfc822_header(const char *p)
{
	struct filter_info info;

	filter_start(&info, FILTER_FOR_DISPLAY, showmsgrfc822_headerp, NULL);
	filter(&info, p, strlen(p));
	filter_end(&info);
}

static void showmsgrfc822_body(FILE *fp, struct rfc2045 *rfc,
			       struct rfc2045id *idptr, int flag,
			       struct msg2html_info *info)
{
char	*header, *value;
char	*save_subject=0;
char	*save_date=0;
off_t	start_pos, end_pos, start_body;
struct	rfc2045id *p, newpart;
off_t	dummy;
off_t	pos;
const struct unicode_info *uiptr=unicode_find(info->output_character_set);

	if (!uiptr)
		uiptr=&unicode_ISO8859_1;

	rfc2045_mimepos(rfc, &start_pos, &end_pos, &start_body, &dummy, &dummy);
	if (fseek(fp, start_pos, SEEK_SET) < 0)
		return;

	printf("<table border=\"0\" cellpadding=\"0\" cellspacing=\"0\" class=\"message-rfc822-header\">\n");

	pos=start_pos;
	while ((header=get_next_header(fp, &value, 1,
		&pos, &start_body)) != 0)
	{
		if (strcmp(header, "list-help") == 0 ||
			strcmp(header, "list-subscribe") == 0 ||
			strcmp(header, "list-unsubscribe") == 0 ||
			strcmp(header, "list-owner") == 0 ||
			strcmp(header, "list-archive") == 0 ||
			strcmp(header, "list-post") == 0)
		{
			print_header_uc(info, header);
			printf("<td><span class=\"message-rfc822-header-contents\">");
			showmsgrfc2369_header(info, value);
			printf("</span></td></tr>\n");
			free(header);
			continue;
		}

		if (info->fullheaders)
		{
			int	isaddress=isaddressheader(header);

			print_header_uc(info, header);
			printf("<td><span class=\"message-rfc822-header-contents\">");
			if (isaddress)
				showmsgrfc822_addressheader(info, value);
			else
				showmsgrfc822_header(value);
			printf("</span></td></tr>\n");
			free(header);
			continue;
		}
		if (strcmp(header, "subject") == 0)
		{
			if (save_subject)	free(save_subject);

			save_subject=
				rfc822_display_hdrvalue_tobuf(header, value,
							      uiptr->chset,
							      NULL,
							      NULL);

			if (!save_subject)
				save_subject=strdup(value);

			free(header);
			continue;
		}
		if (strcmp(header, "date") == 0)
		{
			if (save_date)	free(save_date);
			save_date=strdup(value);
			free(header);
			continue;
		}
		if (isaddressheader(header))
		{
			print_header_uc(info, header);
			printf("<td><span class=\"message-rfc822-header-contents\">");
			showmsgrfc822_addressheader(info, value);
			printf("</span></td></tr>\n");
		}
		free(header);
	}

	if (save_date)
	{
		time_t	t=rfc822_parsedt(save_date);
		struct tm *tmp=t ? localtime(&t):0;
		char	date_buf[256];

		if (tmp)
		{
			char date_header[10];
			const char *date_fmt="%d %b %Y, %I:%M:%S %p";

			if (info->email_header_date_fmt)
				date_fmt=(*info->email_header_date_fmt)
					(date_fmt);

			strcpy(date_header, "Date");
			print_header_uc(info, date_header);

			strftime(date_buf, sizeof(date_buf)-1, date_fmt, tmp);
			date_buf[sizeof(date_buf)-1]=0;
			printf("<td><span class=\"message-rfc822-header-contents\">");
			showmsgrfc822_header(date_buf);
			printf("</span></td></tr>\n");
		}
		free(save_date);
	}

	if (save_subject)
	{
		char subj_header[20];

		strcpy(subj_header, "Subject");
		print_header_uc(info, subj_header);

		printf("<td><span class=\"message-rfc822-header-contents\">");
		showmsgrfc822_header(save_subject);
		printf("</span></td></tr>\n");
	}

	if (flag && info->message_rfc822_action)
		(*info->message_rfc822_action)(idptr);

	printf("</table>\n<hr width=\"100%%\" />\n");

	if (!flag && info->gpgdir && libmail_gpg_has_gpg(info->gpgdir) == 0
	    && libmail_gpgmime_has_mimegpg(rfc)
	    && info->gpg_message_action)
		(*info->gpg_message_action)();

	if (!idptr)
	{
		idptr= &newpart;
		p=0;
	}
	else
	{
		for (p=idptr; p->next; p=p->next)
			;
		p->next=&newpart;
	}
	newpart.idnum=1;
	newpart.next=0;
	(*get_handler(rfc, info))(fp, rfc, idptr, info);
	if (p)
		p->next=0;
}

void msg2html(FILE *fp, struct rfc2045 *rfc,
	      struct msg2html_info *info)
{
	if (!info->mimegpgfilename)
		info->mimegpgfilename="";

	showmsgrfc822_body(fp, rfc, NULL, 0, info);
}

static void showmsgrfc822(FILE *fp, struct rfc2045 *rfc, struct rfc2045id *id,
			  struct msg2html_info *info)
{
	if (rfc->firstpart)
		showmsgrfc822_body(fp, rfc->firstpart, id, 1, info);
}

static void showunknown(FILE *fp, struct rfc2045 *rfc, struct rfc2045id *id,
			struct msg2html_info *info)
{
const char	*content_type, *cn;
const char	*dummy;
off_t start_pos, end_pos, start_body;
off_t dummy2;
char	*content_name;
const struct unicode_info *uiptr=unicode_find(info->output_character_set);

	id=id;
	rfc2045_mimeinfo(rfc, &content_type, &dummy, &dummy);

	/* Punt for image/ MIMEs */

	if (strncmp(content_type, "image/", 6) == 0 &&
		(rfc->content_disposition == 0
		 || strcmp(rfc->content_disposition, "attachment")))
	{
		if (info->inline_image_action)
			(*info->inline_image_action)(id, content_type,
						     info->arg);
		return;
	}

	if (rfc2231_udecodeType(rfc, "name", uiptr, &content_name) < 0 ||
	    rfc2231_udecodeDisposition(rfc, "filename", uiptr,
				       &content_name) < 0)
		content_name=NULL;
	if (!content_name &&
	    ((cn=rfc2045_getattr(rfc->content_type_attr, "name")) ||
	     (cn=rfc2045_getattr(rfc->content_disposition_attr,
				 "filename"))) &&
	    strstr(cn, "=?") && strstr(cn, "?="))
	    /* RFC2047 header encoding (not compliant to RFC2047) */
	{
		content_name = rfc822_display_hdrvalue_tobuf("subject",
							     cn,
							     uiptr->chset,
							     NULL, NULL);
	}

	rfc2045_mimepos(rfc, &start_pos, &end_pos, &start_body,
			&dummy2, &dummy2);

	if (info->unknown_attachment_action)
		(*info->unknown_attachment_action)(id, content_type,
						   content_name,
						   end_pos-start_body,
						   info->arg);


	if (content_name)
		free(content_name);
}

void showmultipartdecoded_start(int status, const char **styleptr)
{
	const char *style= status ? "message-gpg-bad":"message-gpg-good";

	printf("<table border=\"0\" cellpadding=\"2\" class=\"%s\"><tr><td>"
	       "<table border=\"0\" class=\"message-gpg\"><tr><td>", style);
	*styleptr=status ? "message-gpg-bad-text":"message-gpg-good-text";

}

void showmultipartdecoded_end()
{
	printf("</td></tr></table></td></tr></table>\n");
}

static void showmultipart(FILE *fp, struct rfc2045 *rfc, struct rfc2045id *id,
			  struct msg2html_info *info)
{
const char	*content_type, *dummy;
struct	rfc2045 *q;
struct	rfc2045id	nextpart, nextnextpart;
struct	rfc2045id	*p;
int gpg_status;

	for (p=id; p->next; p=p->next)
		;
	p->next=&nextpart;
	nextpart.idnum=0;
	nextpart.next=0;

	rfc2045_mimeinfo(rfc, &content_type, &dummy, &dummy);

	if (info->is_gpg_enabled && (*info->is_gpg_enabled)() &&
	    libmail_gpgmime_is_decoded(rfc, &gpg_status))
	{
		const char *style;
		showmultipartdecoded_start(gpg_status, &style);
		for (q=rfc->firstpart; q; q=q->next, ++nextpart.idnum)
		{
			if (q->isdummy)	continue;

			
			if (nextpart.idnum == 1)
			{
				printf("<blockquote class=\"%s\">",
				       style);
			}

			(*get_handler(q, info))(fp, q, id, info);
			if (nextpart.idnum == 1)
			{
				printf("</blockquote>");
			}
			else
				if (q->next)
					printf("<hr width=\"100%%\" />\n");
		}
		showmultipartdecoded_end();
	}
	else if (strcmp(content_type, "multipart/alternative") == 0)
	{
		struct	rfc2045 *q, *r=0, *s;
	int	idnum=0;
	int	dummy;

		for (q=rfc->firstpart; q; q=q->next, ++idnum)
		{
			int found=0;
			if (q->isdummy)	continue;

			/*
			** We pick this multipart/related section if:
			**
			** 1) This is the first section, or
			** 2) We know how to display this section, or
			** 3) It's a multipart/signed section and we know
			**    how to display the signed content.
			** 4) It's a decoded section, and we know how to
			**    display the decoded section.
			*/

			if (!r)
				found=1;
			else if ((s=libmail_gpgmime_is_multipart_signed(q))
				 != 0)
			{
				if (get_known_handler(s, info))
					found=1;
			}
			else if ( *info->mimegpgfilename
				  && libmail_gpgmime_is_decoded(q, &dummy))
			{
				if ((s=libmail_gpgmime_decoded_content(q)) != 0
				    && get_known_handler(s, info))
					found=1;
			}
			else if (get_known_handler(q, info))
			{
				found=1;
			}

			if (found)
			{
				r=q;
				nextpart.idnum=idnum;
			}
		}

		if (r)
			(*get_handler(r, info))(fp, r, id, info);
	}
	else if (strcmp(content_type, "multipart/related") == 0)
	{
	char *sid=rfc2045_related_start(rfc);

		/*
		** We can't just walts in, search for the Content-ID:,
		** and skeddaddle, that's because we need to keep track of
		** our MIME section.  So we pretend that we're multipart/mixed,
		** see below, and abort at the first opportunity.
		*/

		for (q=rfc->firstpart; q; q=q->next, ++nextpart.idnum)
		{
		const char *cid;

			if (q->isdummy)	continue;

			cid=rfc2045_content_id(q);

			if (sid && *sid && strcmp(sid, cid))
			{
				struct rfc2045 *qq;

				qq=libmail_gpgmime_is_multipart_signed(q);

				if (!qq) continue;

				/* Don't give up just yet */

				cid=rfc2045_content_id(qq);

				if (sid && *sid && strcmp(sid, cid))
				{
					/* Not yet, check for MIME/GPG stuff */



					/* Ok, we can give up now */
					continue;
				}
				nextnextpart.idnum=1;
				nextnextpart.next=0;
				nextpart.next= &nextnextpart;
			}
			(*get_handler(q, info))(fp, q, id, info);

			break;
			/* In all cases, we stop after dumping something */
		}
		if (sid)	free(sid);
	}
	else
	{
		for (q=rfc->firstpart; q; q=q->next, ++nextpart.idnum)
		{
			if (q->isdummy)	continue;
			(*get_handler(q, info))(fp, q, id, info);
			if (q->next)
				printf("<hr width=\"100%%\" />\n");
		}
	}
	p->next=0;
}

static void text_to_stdout(const char *p, size_t n)
{
	while (n)
	{
		--n;
		putchar(*p++);
	}
}

static int htmlfilter_stub(const char *ptr, size_t cnt, void *voidptr)
{
	htmlfilter((struct htmlfilter_info *)voidptr, ptr, cnt);
	return (0);
}


/* Recursive search for a Content-ID: header that we want */

static struct rfc2045 *find_cid(struct rfc2045 *p, const char *cidurl)
{
const char *cid=rfc2045_content_id(p);

	if (cid && strcmp(cid, cidurl) == 0)
		return (p);

	for (p=p->firstpart; p; p=p->next)
	{
	struct rfc2045 *q;

		if (p->isdummy)	continue;

		q=find_cid(p, cidurl);
		if (q)	return (q);
	}
	return (0);
}

/*
** Given an rfc2045 ptr, return the mime reference that will resolve to
** this MIME part.
*/

static char *rfc2mimeid(struct rfc2045 *p)
{
char	buf[MAXLONGSIZE+1];
char	*q=0;
unsigned n=p->pindex+1;	/* mime counts start at one */
char	*r;

	if (p->parent)
	{
		q=rfc2mimeid(p->parent);
		if (p->parent->firstpart->isdummy)
			--n;	/* ... except let's ignore the dummy part */
	}
	else	n=1;

	sprintf(buf, "%u", n);
	r=malloc( (q ? strlen(q)+1:0)+strlen(buf)+1);
	if (!r)
	{
		if (q)
			free(q);
		return NULL;
	}
	*r=0;
	if (q)
	{
		strcat(strcat(r, q), ".");
		free(q);
	}
	strcat(r, buf);
	return (r);
}

/*
** Convert cid: url to a http:// reference that will access the indicated
** MIME section.
*/

struct convert_cid_info {
	struct rfc2045 *rfc;
	struct msg2html_info *info;
};

static void add_decoded_link(struct rfc2045 *, const char *, int);

static char *convertcid(const char *cidurl, void *voidp)
{
	struct convert_cid_info *cid_info=
		(struct convert_cid_info *)voidp;

	struct	rfc2045 *rfc=cid_info->rfc;
	struct	rfc2045 *savep;

	char	*mimeid;
	char	*p;
	char *mimegpgfilename=cgiurlencode(cid_info->info->mimegpgfilename);
	int dummy;

	if (!mimegpgfilename)
		return NULL;

	if (rfc->parent)	rfc=rfc->parent;
	if (rfc->parent)
	{
		if (libmail_gpgmime_is_multipart_signed(rfc) ||
		    (*mimegpgfilename
		     && libmail_gpgmime_is_decoded(rfc, &dummy)))
			rfc=rfc->parent;
	}

	savep=rfc;
	rfc=find_cid(rfc, cidurl);

	if (!rfc)
		/* Sometimes broken MS software needs to go one step higher */
	{
		while ((savep=savep->parent) != NULL)
		{
			rfc=find_cid(savep, cidurl);
			if (rfc)
				break;
		}
	}

	if (!rfc)	/* Not found, punt */
	{
		free(mimegpgfilename);
		return strdup("");
	}

	mimeid=rfc2mimeid(rfc);

	if (!mimeid)
		p=NULL;
	else if (!cid_info->info->get_url_to_mime_part)
		p=strdup("");
	else
		p=(*cid_info->info->get_url_to_mime_part)(mimeid,
							  cid_info->info);
	free(mimeid);

	if (*mimegpgfilename && rfc->parent &&
	    libmail_gpgmime_is_decoded(rfc->parent, &dummy))
		add_decoded_link(rfc->parent, mimeid, dummy);

	return p;
}

/*
** When we output a multipart/related link to some content that has been
** signed/encoded, we save the decoding status, for later.
**
** Note -- we collapse multiple links to the same content.
*/

static struct decoded_list {
	struct decoded_list *next;
	struct rfc2045 *ptr;
	char *mimeid;
	int status;
} *decoded_first=0, *decoded_last=0;

static void add_decoded_link(struct rfc2045 *ptr, const char *mimeid,
			     int status)
{
	struct decoded_list *p;

	for (p=decoded_first; p; p=p->next)
	{

		if (strcmp(p->mimeid, mimeid) == 0)
			return;	/* Dupe */
	}

	p=(struct decoded_list *)malloc(sizeof(*p));

	if (!p)
		return;

	p->mimeid=strdup(mimeid);

	if (!p->mimeid)
	{
		free(p);
		return;
	}
	p->next=0;

	if (decoded_last)
		decoded_last->next=p;
	else
		decoded_first=p;

	decoded_last=p;

	p->ptr=ptr;
	p->status=status;
}

static void showtexthtml(FILE *fp, struct rfc2045 *rfc, struct rfc2045id *id,
			 struct msg2html_info *info)
{
	char	*content_base;
	const char *mime_charset, *dummy_s;

	struct htmlfilter_info *hf_info;

	id=id;


	content_base=rfc2045_content_base(rfc);

	rfc2045_mimeinfo(rfc, &dummy_s, &dummy_s, &mime_charset);

	hf_info=htmlfilter_alloc(&text_to_stdout);

	if (hf_info)
	{
		struct convert_cid_info cid_info;

		cid_info.rfc=rfc;
		cid_info.info=info;

		htmlfilter_washlink(hf_info, info->wash_http_prefix);
		htmlfilter_convertcid(hf_info, &convertcid, &cid_info);

		htmlfilter_contentbase(hf_info, content_base);

		htmlfilter_washlinkmailto(hf_info, info->wash_mailto_prefix);

		if (info->character_set_follows)
			(*info->character_set_follows)(mime_charset);

		if (info->html_content_follows)
			(*info->html_content_follows)();

		printf("<table border=\"0\" cellpadding=\"0\" cellspacing=\"0\" width=\"100%%\"><tr><td>\n");

		rfc2045_decodetextmimesection(fileno(fp), rfc,
					      info->output_character_set,
					      &htmlfilter_stub,
					      hf_info);

		printf("</td></tr>");

		htmlfilter_free(hf_info);
	}

	free(content_base);

	while (decoded_first)
	{
		struct decoded_list *p=decoded_first;
		const char *style;

		struct rfc2045 *q;

		printf("<tr><td>");

		showmultipartdecoded_start(p->status, &style);

		for (q=p->ptr->firstpart; q; q=q->next)
		{
			if (q->isdummy)
				continue;

			printf("<div class=\"%s\">", style);
			(*get_handler(q, info))(fp, q, NULL, info);
			printf("</div>\n");
			break;
		}
		showmultipartdecoded_end();
		decoded_first=p->next;
		free(p->mimeid);
		free(p);
		printf("</td></tr>\n");
	}
	printf("</table>\n");

}

static void showdsn(FILE *fp, struct rfc2045 *rfc, struct rfc2045id *id,
		    struct msg2html_info *info)
{
off_t	start_pos, end_pos, start_body;
off_t	dummy;

	id=id;
	rfc2045_mimepos(rfc, &start_pos, &end_pos, &start_body, &dummy, &dummy);
	if (fseek(fp, start_body, SEEK_SET) < 0)
	{
		printf("Seek error.");
		return;
	}
	printf("<table border=\"0\" cellpadding=\"0\" cellspacing=\"0\">\n");
	while (start_body < end_pos)
	{
	int	c=getc(fp);
	char	*header, *value;

		if (c == EOF)	break;
		if (c == '\n')
		{
			printf("<tr><td colspan=\"2\"><hr /></td></tr>\n");
			++start_body;
			continue;
		}
		ungetc(c, fp);

		if ((header=get_next_header(fp, &value, 1,
			&start_body, &end_pos)) == 0)
			break;

		print_header_uc(info, header);
		printf("<td><span class=\"message-rfc822-header-contents\">");
		/* showmsgrfc822_addressheader(value); */
		printf("%s", value);
		printf("</span></td></tr>\n");
		free(header);
	}
	printf("</table>\n");
}


static struct buf showtextplain_buf;

const char *skip_text_url(const char *r, const char *end)
{
const char *q=r;

	for (; r < end && (isalnum(*r) || *r == ':' || *r == '/'
		|| *r == '.' || *r == '~' || *r == '%'
		|| *r == '+' || *r == '?' || *r == '&' || *r == '#'
		|| *r == '=' || *r == '@' || *r == ';'
		|| *r == '-' || *r == '_' || *r == ','); r++)
	{
		if (*r == '&' && (end-r < 5 || strncmp(r, "&amp;", 5)))
			break;
	}
	if (r > q && (r[-1] == ',' || r[-1] == '.' || r[-1] == ';'))	--r;
	return (r);
}

static void showtextplain_oneline(const char *p, size_t n,
				  struct msg2html_info *info)
{
const char *q, *r;
char	*s, *t;

	for (q=r=p; q < p+n; q++)
	{
		if ( p+n-q > 7 &&
				(strncmp(q, "http://", 7) == 0
					||
				strncmp(q, "https:/", 7) == 0
					||
				strncmp(q, "mailto:", 7) == 0)
				)
		{
			if (fwrite(r, q-r, 1, stdout) != 1)
			    ; /* ignore */

			r=skip_text_url(q, p+n);

			if ((s=malloc(r+1-q)) != NULL)
			{
				memcpy(s, q, r-q);
				s[r-q]=0;

				t=info->get_textlink ?
					(*info->get_textlink)(s, info->arg)
					:NULL;

				if (t)
				{
					printf("%s", t);
					free(t);
				}
				free(s);
			}
			q=r;
		}

		if (strchr(info->smiley_index, *q))
		{
			struct msg2html_smiley_list *sm;
			size_t l=0;

			for (sm=info->smileys; sm; sm=sm->next)
			{
				l=strlen(sm->code);

				if (l <= p+n-q &&
				    strncmp(q, sm->code, l) == 0)
					break;
			}

			if (sm)
			{
				if (fwrite(r, 1, q-r, stdout) != q-r)
					;
				printf("%s", sm->url);
				q += l;
				r=q;
				--q;
				continue;
			}
		}

	}
	if (fwrite(r, 1, q-r, stdout) != q-r)
		; /* ignore */
}

static void showtextplainfunc(const char *txt, size_t l, void *dummy)
{
	const	char *p;
	size_t	n;

	struct msg2html_info *info=dummy;

	if (txt)
	{
		buf_catn(&showtextplain_buf, txt, l);
		while ((p=strchr(showtextplain_buf.ptr, '\n')) != 0)
		{
			n= p+1 - showtextplain_buf.ptr;
			showtextplain_oneline(showtextplain_buf.ptr, n, info);
			buf_trimleft(&showtextplain_buf, n);
		}
	}
	else if (showtextplain_buf.cnt)
		showtextplain_oneline(showtextplain_buf.ptr,
				      showtextplain_buf.cnt, info);
}

struct textplain_info {

	struct rfc2646parser *flowparser;
	struct filter_info info;
};


static int filter_stub(const char *ptr, size_t cnt, void *voidptr)
{
	struct textplain_info *tinfo=(struct textplain_info *)voidptr;

	if (tinfo->flowparser)
		return ( rfc2646_parse(tinfo->flowparser, ptr, cnt));

	filter(&tinfo->info, ptr, cnt);
	return (0);
}

static int filter_flowed(const char *ptr, int cnt, void *voidptr)
{
	showtextplainfunc(ptr, cnt, voidptr);
	return (0);
}

static int preview_callback(const char *ptr, size_t cnt, void *voidptr)
{
        filter((struct filter_info *)voidptr, ptr, cnt);
        return (0);
}

static void preview_show_func_s(const char *p, size_t n, void *dummy)
{
        if (fwrite(p, 1, n, stdout) != n)
                ; /* ignore */
}


static void showtextplain(FILE *fp, struct rfc2045 *rfc, struct rfc2045id *id,
			  struct msg2html_info *info)
{
	int rc;

	const char *mime_charset, *dummy;

	int isflowed;

	struct textplain_info tinfo;
	struct rfc2646tohtml *flowtohtml=0;

	id=id;

	tinfo.flowparser=NULL;

	if (info->is_preview_mode && (*info->is_preview_mode)())
	{
		const char *cb=rfc2045_getattr(rfc->content_type_attr,
					       "format");
		if (cb && strcasecmp(cb, "xdraft") == 0)
		{
			struct filter_info preview_filter_info;

			filter_start(&preview_filter_info,
				     FILTER_FOR_PREVIEW,
				     &preview_show_func_s, NULL);
			printf("<pre class=\"message-text-plain\">");

			rfc2045_decodemimesection(fileno(fp), rfc,
						  &preview_callback,
						  &preview_filter_info);
			filter_end(&preview_filter_info);
			printf("</pre>\n");
			return;
		}
	}

	rfc2045_mimeinfo(rfc, &dummy, &dummy, &mime_charset);

	if (info->character_set_follows)
		(*info->character_set_follows)(mime_charset);

	isflowed=rfc2045_isflowed(rfc);

	if (info->noflowedtext)
		isflowed=0;

	buf_init(&showtextplain_buf);

	if (isflowed)
	{
		flowtohtml=rfc2646tohtml_alloc(filter_flowed, info);

		if (!flowtohtml)
			isflowed=0;
	}

	if (isflowed)
	{
		tinfo.flowparser=RFC2646TOHTML_PARSEALLOC(flowtohtml);

		if (!tinfo.flowparser)
		{
			rfc2646tohtml_free(flowtohtml);
			flowtohtml=NULL;
			isflowed=0;
		}
	}

	if (isflowed)
	{
		printf("<div class=\"message-text-plain\">");
	}
	else
	{
		printf("<pre class=\"message-text-plain\">");
		filter_start(&tinfo.info, FILTER_FOR_DISPLAY,
			     &showtextplainfunc, info);
	}

	rc=rfc2045_decodetextmimesection(fileno(fp), rfc,
					 info->output_character_set,
					 &filter_stub,
					 &tinfo);
	fseek(fp, 0L, SEEK_END);
	fseek(fp, 0L, SEEK_SET);	/* Resync stdio with uio */

	if (isflowed)
	{
		rfc2646tohtml_cleanup(flowtohtml);
		rfc2646_free(tinfo.flowparser);
		rfc2646tohtml_free(flowtohtml);
	}
	else
	{
		filter_end(&tinfo.info);
	}
	showtextplainfunc(0, 0, NULL);
	buf_free(&showtextplain_buf);

	if (isflowed)
		printf("</div><br />\n");
	else
		printf("</pre><br />\n");
}

static void showkey(FILE *fp, struct rfc2045 *rfc, struct rfc2045id *id,
		    struct msg2html_info *info)
{
	if (info->application_pgp_keys_action)
		(*info->application_pgp_keys_action)(id);
}

static void (*get_known_handler(struct rfc2045 *mime,
				struct msg2html_info *info))
	(FILE *, struct rfc2045 *, struct rfc2045id *,
	 struct msg2html_info *)
{
const char	*content_type, *dummy;

	rfc2045_mimeinfo(mime, &content_type, &dummy, &dummy);
	if (strncmp(content_type, "multipart/", 10) == 0)
		return ( &showmultipart );

	if (strcmp(content_type, "application/pgp-keys") == 0
	    && info->gpgdir && libmail_gpg_has_gpg(info->gpgdir) == 0)
		return ( &showkey );

	if (mime->content_disposition
	    && strcmp(mime->content_disposition, "attachment") == 0)
		return (0);

	if (strcmp(content_type, "text/plain") == 0 ||
	    strcmp(content_type, "text/rfc822-headers") == 0 ||
	    strcmp(content_type, "text/x-gpg-output") == 0)
		return ( &showtextplain );
	if (strcmp(content_type, "message/delivery-status") == 0)
		return ( &showdsn);
	if (info->showhtml && strcmp(content_type, "text/html") == 0)
		return ( &showtexthtml );
	if (strcmp(content_type, "message/rfc822") == 0)
		return ( &showmsgrfc822);

	return (0);
}

static void (*get_handler(struct rfc2045 *mime,
			  struct msg2html_info *info))
	(FILE *, struct rfc2045 *,
	 struct rfc2045id *,
	 struct msg2html_info *)
{
	void (*func)(FILE *, struct rfc2045 *, struct rfc2045id *,
		     struct msg2html_info *);

	if ((func=get_known_handler(mime, info)) == 0)
		func= &showunknown;

	return (func);
}

static int download_func(const char *, size_t, void *);

static void disposition_attachment(FILE *fp, const char *p, int attachment)
{
	fprintf(fp, "Content-Disposition: %s; filename=\"", 
		attachment ? "attachment":"inline");
	while (*p)
	{
		if (*p == '"' || *p == '\\')
			putc('\\', fp);
		if (!((unsigned char)(*p) < (unsigned char)' '))
			putc(*p, fp);
		p++;
	}
	fprintf(fp, "\"\n");
}


void msg2html_download(FILE *fp, const char *mimeid, int dodownload)
{
	struct	rfc2045 *rfc, *part;
	char	buf[BUFSIZ];
	int	n,cnt;
	const char	*content_type, *dummy, *charset;
	off_t	start_pos, end_pos, start_body;
	char	*content_name;
	off_t	ldummy;

	rfc=rfc2045_alloc();

	while ((n=fread(buf, 1, sizeof(buf), fp)) > 0)
		rfc2045_parse(rfc, buf, n);
	rfc2045_parse_partial(rfc);

	part=*mimeid ? rfc2045_find(rfc, mimeid):rfc;
	if (!part)
	{
		rfc2045_free(rfc);
		return;
	}

	rfc2045_mimeinfo(part, &content_type, &dummy, &charset);

	if (rfc2231_udecodeType(part, "name", NULL,
				&content_name) < 0)
		content_name=NULL;

	if (dodownload)
	{
		char *disposition_filename;
		const char *p;

		if (rfc2231_udecodeDisposition(part, "filename", NULL,
					       &disposition_filename) < 0)
		{
			if (content_name)
				free(content_name);
			rfc2045_free(rfc);
			return;
		}


		p=disposition_filename;

		if (!p || !*p) p=content_name;
		if (!p || !*p) p="message.dat";
		disposition_attachment(stdout, p, 1);
		content_type="application/octet-stream";
		free(disposition_filename);
	} else {
		if (content_name && *content_name)
			disposition_attachment(stdout, content_name, 0);
	}

	printf(
		content_name && *content_name ?
		"Content-Type: %s; charset=%s; name=\"%s\"\n\n":
		"Content-Type: %s; charset=%s\n\n",
			content_type, charset, content_name ? content_name:"");
	if (content_name)
		free(content_name);

	rfc2045_mimepos(part, &start_pos, &end_pos, &start_body,
		&ldummy, &ldummy);

	if (*mimeid == 0)	/* Download entire message */
	{
		if (fseek(fp, start_pos, SEEK_SET) < 0)
		{
			rfc2045_free(rfc);
			return;
		}

		while (start_pos < end_pos)
		{
			cnt=sizeof(buf);
			if (cnt > end_pos-start_pos)
				cnt=end_pos-start_pos;
			cnt=fread(buf, 1, cnt, fp);
			if (cnt <= 0)	break;
			start_pos += cnt;
			download_func(buf, cnt, NULL);
		}
	}
	else
	{
		if (fseek(fp, start_body, SEEK_SET) < 0)
		{
			rfc2045_free(rfc);
			return;
		}

		rfc2045_cdecode_start(part, &download_func, 0);

		while (start_body < end_pos)
		{
			cnt=sizeof(buf);
			if (cnt > end_pos-start_body)
				cnt=end_pos-start_body;
			cnt=fread(buf, 1, cnt, fp);
			if (cnt <= 0)	break;
			start_body += cnt;
			rfc2045_cdecode(part, buf, cnt);
		}
		rfc2045_cdecode_end(part);
	}
	rfc2045_free(rfc);
}

static int download_func(const char *p, size_t cnt, void *voidptr)
{
	if (fwrite(p, 1, cnt, stdout) != cnt)
		return (-1);
	return (0);
}

void msg2html_showmimeid(struct rfc2045id *idptr, const char *p)
{
	if (!p)
		p="&amp;mimeid=";

	while (idptr)
	{
		printf("%s%d", p, idptr->idnum);
		idptr=idptr->next;
		p=".";
	}
}
