# ui_gtk.py - graphical user interface implemented using GTK+
# Copyright (C) 2008, 2009  Canonical, Ltd.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


import logging
import os
import sys
import threading
import time
import traceback

import computerjanitor
import computerjanitorapp
_ = computerjanitorapp.setup_gettext()

# We can't import gtk here, since that would mean it gets imported
# even when we run in command line mode. Thus, we do the imports in
# the class, when we know we need them.


import os
GLADE = os.environ.get('COMPUTER_JANITOR_GLADE', 
                       '/usr/share/computer-janitor/ComputerJanitor.ui')


STATE_COL = 0
NAME_COL = 1
CRUFT_COL = 2
EXPANDED_COL = 3
SHOW_COL = 4


def ui(method):
    """Decorator function for UI methods.
    
    Use @ui to decorate all methods in the controller class that
    call GTK stuff, to ensure correct locking.
    
    """
    
    def new(self, *args, **kwargs):
        self.gtk.gdk.threads_enter()
        ret = method(self, *args, **kwargs)
        self.gtk.gdk.threads_leave()
        return ret
        
    return new
    
    
class ListUpdater(threading.Thread):

    """Find cruft in the background, update user interface when done."""

    def __init__(self, ui):
        threading.Thread.__init__(self)
        self.ui = ui

    def run(self):
        self.ui.finding = True
        self.ui.set_sensitive()
        status = self.ui.widgets['statusbar']
        context_id = status.get_context_id('ListUpdater')
        status.push(context_id, _('Analyzing system...'))
        for plugin in self.ui.pm.get_plugins():
            for cruft in self.ui.app.remove_whitelisted(plugin.get_cruft()):
                self.ui.add_cruft(cruft)
        status.pop(context_id)
        self.ui.finding = False
        self.ui.set_sensitive()
        self.ui.done_updating_list()


class Cleaner(threading.Thread):

    """Actually clean up the cruft."""
    
    def __init__(self, ui):
        threading.Thread.__init__(self)
        self.ui = ui
        self.gtk = self.ui.gtk
        self.crufts = ui.get_crufts()
        self.plugins = ui.pm.get_plugins()
        # progress reporting dialog/bar
        # FIXME: the progress bar is a bit silly, it takes
        #        items like fix-fstab (0.1s) and deb_package (5min)
        #        and assigns them the same slice - use a pulsing one 
        #        instead?
        self.dialog = ui.widgets['cleanup_dialog']
        self.pbar = ui.widgets['cleanup_progressbar']
        # add status bar context
        self.status = self.ui.widgets['statusbar']
        self.context_id = self.status.get_context_id('ListUpdater')
        self.status.push(self.context_id, _('Cleaning up...'))
        # cancel button
        button = ui.widgets['cleanup_cancel']
        button.connect('clicked', self.cancel)
        self.cancel_event = threading.Event()

    def cancel(self, *args):
        self.cancel_event.set()
        
    @ui
    def inc_done(self):
        self.done_work += 1.0
        self.pbar.set_fraction(self.done_work / self.total_work)

    @ui
    def start_reporting(self):
        self.total_work = len(self.crufts) + len(self.plugins)
        self.done_work = 0.0
        self.pbar.set_fraction(0.0)

        self.dialog.show()
    
    @ui
    def end_reporting(self):
        self.pbar.set_fraction(1.0)
        self.dialog.hide()
        self.status.pop(self.context_id)

    def run(self):
        self.ui.cleaning = True
        self.ui.set_sensitive()

        self.start_reporting()
        
        for cruft in self.crufts:
            if self.cancel_event.isSet():
                break
            name = cruft.get_name()
            if self.ui.app.state.is_enabled(name):
                if self.ui.options.no_act:
                    time.sleep(0.1)
                    logging.info(_("Pretending to remove cruft: %s") % name)
                else:
                    logging.info(_("Removing cruft: %s") % name)
                    cruft.cleanup()
            self.inc_done()

        for plugin in self.plugins:
            if self.cancel_event.isSet():
                break
            if self.ui.options.no_act:
                logging.info(_("Pretending to post-cleanup: %s") % plugin)
            else:
                logging.info(_("Post-cleanup: %s") % plugin)
                error = None
                try:
                    plugin.post_cleanup()
                except Exception, e:
                    logging.debug(unicode(traceback.format_exc()))
                    self.ui.show_error(_("Could not clean up properly"),
                                       unicode(e))
                    break
            self.inc_done()
                
        # do not run it directly (thread deadlocks), register for running
        # in the gtk main thread instead
        self.ui.glib.timeout_add(100, self.ui.show_cruft)
        self.ui.cleaning = False

        self.end_reporting()


class GtkUserInterface(computerjanitorapp.UserInterface):

    """The GTK+ user interface of Computer Janitor."""
    
    # This acts as the controller in MVC. To simplify binding of callbacks
    # to GTK signals, we follow a strict convention: any method named
    # on_WIDGETNAME_SIGNALNAME is a callback and will be bound automatically. 
    # This way, we don't need do add each binding by hand, either in the 
    # code or in the .glade file. See the find_and_bind_widgets method
    # for details.

    def run(self, options, args):
        # This is where UI execution starts.
        
        import gtk
        self.gtk = gtk
        
        import gobject
        self.gobject = gobject

        import glib
        self.glib = glib

        self.options = options
        self.app.state.load(options.state_file)

        builder = gtk.Builder()
        builder.set_translation_domain('computerjanitor')
        builder.add_from_file(GLADE)
        self.find_and_bind_widgets(builder)
        
        self.store = self.create_cruft_store()
        self.name_cols = set()
        self.popup_menus = dict()
        self.create_column('unused_treeview', self.unused_filter)
        self.create_column('recommended_treeview', self.recommended_filter)
        self.create_column('optimize_treeview', self.optimize_filter)
        
        self.show_previously_ignored = False
        
        self.first_map = True
        
        self.set_default_window_size()
        self.widgets['window'].show()
        
        self.sort_crufts_by_current_order = self.sort_crufts_by_name
        
        self.finding = False
        self.cleaning = False

        # set thread switches interval to make it more UI friendly
        sys.setcheckinterval(0)
        gtk.gdk.threads_init()
        gtk.main()

    def find_and_bind_widgets(self, builder):
        """Bind widgets and callbacks."""
        import gtk
        self.widgets = {}
        for o in builder.get_objects():
            if issubclass(type(o), gtk.Buildable):
                name = gtk.Buildable.get_name(o)
                self.widgets[name] = o
                for attr in dir(self):
                    prefix = 'on_%s_' % name
                    if attr.startswith(prefix):
                        signal_name = attr[len(prefix):]
                        method = getattr(self, attr)
                        o.connect(signal_name, method)
                        
    
    def set_default_window_size(self):
        w = self.widgets['window']
        width = 900
        height = 700
        self.widgets['window'].set_default_size(width, height)
    
    def create_cruft_store(self):
        """Create a gtk.ListStore for holding all the cruft."""
        pairs = ((NAME_COL, self.gobject.TYPE_STRING),
                 (STATE_COL, self.gobject.TYPE_BOOLEAN),
                 (CRUFT_COL, self.gobject.TYPE_PYOBJECT),
                 (EXPANDED_COL, self.gobject.TYPE_BOOLEAN),
                 (SHOW_COL, self.gobject.TYPE_BOOLEAN))
        column_types = [pair[1] for pair in sorted(pairs)]
        store = self.gtk.ListStore(*column_types)

        return store

    def get_crufts(self):
        def get(model, path, iter, crufts):
            cruft = model.get_value(iter, CRUFT_COL)
            crufts.append(cruft)
        crufts = []
        self.store.foreach(get, crufts)
        return crufts

    def sort_crufts(self, get_key):
        crufts = self.get_crufts()
        crufts = [(get_key(c), i, c) for i, c in enumerate(crufts)]
        crufts.sort()
        crufts = [i for key, i, c in crufts]
        self.store.reorder(crufts)
        
    def sort_crufts_by_name(self):
        def get_key(cruft):
            return cruft.get_name()
        self.sort_crufts(get_key)
        
    def sort_crufts_by_size(self):
        def get_key(cruft):
            return -cruft.get_disk_usage()
        self.sort_crufts(get_key)

    def create_column(self, widget_name, filterfunc):
        """Add gtk.TreeViewColumn to the desired widget."""
        treeview = self.widgets[widget_name]
        treeview.set_rules_hint(True)
        
        toggle_cr = self.gtk.CellRendererToggle()
        toggle_cr.connect('toggled', self.toggled, treeview)
        toggle_cr.set_property("yalign", 0)
        toggle_col = self.gtk.TreeViewColumn()
        toggle_col.pack_start(toggle_cr)
        toggle_col.add_attribute(toggle_cr, 'active', STATE_COL)
        treeview.append_column(toggle_col)

        name_cr = self.gtk.CellRendererText()
        name_cr.set_property("yalign", 0)
        import pango
        name_cr.set_property("wrap-mode", pango.WRAP_WORD)
        name_col = self.gtk.TreeViewColumn()
        name_col.pack_start(name_cr)
        name_col.add_attribute(name_cr, 'markup', NAME_COL)
        treeview.append_column(name_col)
        self.name_cols.add(name_col)
        
        filter_store = self.store.filter_new()
        filter_store.set_visible_func(filterfunc)
        treeview.set_model(filter_store)
        
        self.create_popup_menu_for_treeview(treeview)

    def create_popup_menu_for_treeview(self, treeview):
        select_all = self.gtk.MenuItem(label='Select all')
        select_all.connect('activate', self.popup_menu_select_all, treeview)

        unselect_all = self.gtk.MenuItem(label='Unselect all')
        unselect_all.connect('activate', self.popup_menu_unselect_all, treeview)

        menu = self.gtk.Menu()
        menu.append(select_all)
        menu.append(unselect_all)
        menu.show_all()
        
        self.popup_menus[treeview] = menu

    def unused_filter(self, store, iter):
        cruft = store.get_value(iter, CRUFT_COL)
        shown = store.get_value(iter, SHOW_COL)
        return shown and isinstance(cruft, computerjanitor.PackageCruft)

    def recommended_filter(self, store, iter):
        return False
        
    def optimize_filter(self, store, iter): 
        shown = store.get_value(iter, SHOW_COL)
        return (shown and
                not self.unused_filter(store, iter) and
                not self.recommended_filter(store, iter))

    def error_dialog(self, msg, secondary_msg=None):
        dialog = self.gtk.MessageDialog(parent=self.widgets["window"],
                                        type=self.gtk.MESSAGE_ERROR,
                                        buttons=self.gtk.BUTTONS_OK,
                                        message_format=msg)
        if secondary_msg:
            dialog.format_secondary_text(secondary_msg)

        return dialog

    @ui
    def show_error(self, msg, secondary_msg=None):
        dialog = self.error_dialog(msg, secondary_msg)
        dialog.show()
        dialog.run()
        dialog.hide()

    def require_root(self):
        if os.getuid() != 0:
            dialog = self.error_dialog(_("Root access required."), 
                                       _("You must run computer-janitor-gtk "
                                         "as root. Sorry."))
            dialog.show()
            dialog.run()
            sys.exit(1)

    def require_working_apt_cache(self):
        """ensure that the apt cache is in good state and error/exit 
           otherwise
        """
        try:
            self.app.verify_apt_cache()
        except computerjanitor.Exception, e:
            logging.error(unicode(traceback.format_exc()))
            dialog = self.error_dialog(str(e))
            dialog.show()
            dialog.run()
            sys.exit(1)

    def pulse(self):
        """pulse callback that shows a progress pulse until finding is False"""
        progress = self.widgets['progressbar_status']
        if self.finding or self.cleaning:
            progress.show()
            progress.pulse()
            return True
        else:
            progress.hide()
            return False

    def show_cruft(self):
        """clear the cruft store and update it again via a thread """
        self.store.clear()
        # run as "daemon" thread to ensure that the main app exist
        # if the user presses "quit" before the ListUpdater thread 
        # has finished
        t = ListUpdater(self)
        t.daemon = True
        t.start()
        # run a glib handler to shows a pulse progress
        self.glib.timeout_add(150, self.pulse)

    @ui
    def add_cruft(self, cruft):
        state = self.app.state.is_enabled(cruft.get_name())
        shown = (self.show_previously_ignored or
                 not self.app.state.was_previously_ignored(cruft.get_name()))
        sort_index = 0
        values = ((CRUFT_COL, cruft),
                  (NAME_COL, cruft.get_shortname()),
                  (STATE_COL, state),
                  (EXPANDED_COL, False),
                  (SHOW_COL, shown))
        values = [pair[1] for pair in sorted(values)]
        self.store.append(values)
        self.sort_crufts_by_current_order()

    @ui
    def done_updating_list(self):
        if not self.find_visible_cruft():
            dialog = self.widgets['borednow_messagedialog']
            dialog.show()
            dialog.run()
            dialog.hide()
            
    def foreach_set_state(self, treeview, enabled):
        def set_state(model, path, iter, user_data):
            iter2 = model.convert_iter_to_child_iter(iter)
            cruft = self.store.get_value(iter2, CRUFT_COL)
            cruft_name = cruft.get_name()
            if enabled:
                self.app.state.enable(cruft_name)
            else:
                self.app.state.disable(cruft_name)
            self.store.set_value(iter2, STATE_COL, enabled)
        treeview.get_model().foreach(set_state, None)
        self.app.state.save(self.options.state_file)
        self.set_sensitive_unlocked()

    def format_name(self, cruft):
        return self.gobject.markup_escape_text(cruft.get_shortname())

    def format_size(self, bytes):
        table = ((1000**3, "GB"),
                 (1000**2, "MB"),
                 (1000**1, "kB"),
                 (      1, "B"))
        for factor, unit in table:
            if bytes >= factor or factor == 1:
                return '%d %s' % (bytes / factor, unit)
        
    def format_description(self, cruft):
        esc = self.gobject.markup_escape_text

        lines = [esc(cruft.get_shortname())]
        
        # FIXME: The action verbs should come from the crufts themselves.
        action_verbs = {
            computerjanitor.PackageCruft: 'uninstall',
            computerjanitor.FileCruft: 'remove',
            computerjanitor.MissingPackageCruft: 'install',
        }
        action_descriptions = {
            'uninstall': _('Package will be <b>removed</b>.'),
            'install': _('Package will be <b>installed</b>.'),
            'remove': _('File will be <b>removed</b>.'),
        }
        action_verb = action_verbs.get(type(cruft))
        if action_verb:
            lines += [action_descriptions[action_verb]]
            
        size = cruft.get_disk_usage()
        if size is not None:
            lines += [_('Size: %s.') % self.format_size(size)]

        desc = cruft.get_description()
        if desc:
            lines += ['', esc(desc)]
        
        return '\n'.join(lines)

    def toggle_long_description(self, treeview):
        """Toggle the showing of the long description of some cruft."""
        
        selection = treeview.get_selection()
        filtermodel, selected = selection.get_selected()
        if not selected:
            return
        model = filtermodel.get_model()
        iter = filtermodel.convert_iter_to_child_iter(selected)
        cruft = model.get_value(iter, CRUFT_COL)
        expanded = model.get_value(iter, EXPANDED_COL)
        expanded = not expanded
        model.set_value(iter, EXPANDED_COL, expanded)
        if expanded:
            value = self.format_description(cruft)
        else:
            value = self.format_name(cruft)
        model.set_value(iter, NAME_COL, value)

    @ui
    def set_sensitive(self):
        self.set_sensitive_unlocked()
        
    def set_sensitive_unlocked(self):
        do = self.widgets['do_button']
        
        names = ['unused_treeview', 'recommended_treeview', 
                 'optimize_treeview']
        cleanable_cruft = self.find_visible_cruft()
        do.set_sensitive(not self.finding and
                         not self.cleaning and 
                         len(cleanable_cruft) > 0)

    def find_visible_cruft(self):
        names = ['unused_treeview', 'recommended_treeview', 
                 'optimize_treeview']
        cleanable_cruft = []
        for name in names:
            w = self.widgets[name]
            model = w.get_model()
            it = model.get_iter_first()
            while it is not None:
                cruft = model.get_value(it, CRUFT_COL)
                if self.app.state.is_enabled(cruft.get_name()):
                    cleanable_cruft.append(cruft)
                it = model.iter_next(it)
        return cleanable_cruft

    def really_cleanup(self):
        """Ask user if they really mean to clean up.
        
        Be especially insistent (in wording) if they are removing
        packages.
        
        """
        
        crufts = self.get_crufts()
        crufts = [c 
                  for c in crufts 
                  if self.app.state.is_enabled(c.get_name())]
        packages = [c 
                    for c in crufts 
                    if isinstance(c, computerjanitor.PackageCruft)]
        others = [c for c in crufts if c not in packages]

        # The following messages are a bit vague, since we need to handle
        # cases where we remove packages, and don't remove packages, and
        # so on, given that "clean up cruft" is such a general concept.
        # My apologies to anyone who thinks this is confusing. Please
        # provide a patch that a) works b) is not specific to removing
        # packages.

        msg = _('Are you sure you want to clean up?')
        dialog = self.gtk.MessageDialog(parent=self.widgets['window'],
                                        type=self.gtk.MESSAGE_WARNING,
                                        buttons=self.gtk.BUTTONS_NONE,
                                        message_format=msg)
        dialog.set_title(_('Clean up'))                                        

        if packages:
            msg = (_('You have chosen to <b>remove %d software packages.</b> '
                     'Removing packages that are still needed can cause '
                     'errors.') % 
                   len(packages))
        else:
            msg = _('Do you want to continue?')
        dialog.format_secondary_markup(msg)

        dialog.add_button(self.gtk.STOCK_CANCEL, self.gtk.RESPONSE_CLOSE)
        if others:
            dialog.add_button(_('Clean up'), self.gtk.RESPONSE_YES)
        else:
            dialog.add_button(_('Remove packages'), self.gtk.RESPONSE_YES)

        dialog.show_all()
        response = dialog.run()
        dialog.hide()

        return response == self.gtk.RESPONSE_YES

    # The rest of this class is callbacks for GTK signals.

    def on_about_menuitem_activate(self, *args):
        w = self.widgets['about_dialog']
        w.set_name(_('Computer Janitor'))
        w.set_version(computerjanitorapp.VERSION)
        w.show()
        w.run()
        w.hide()

    def on_do_button_clicked(self, *args):
        if self.really_cleanup():
            Cleaner(self).start()
            self.glib.timeout_add(150, self.pulse)

    def on_show_previously_ignored_toggled(self, menuitem):
        self.show_previously_ignored = menuitem.get_active()
        iter = self.store.get_iter_first()
        while iter:
            cruft = self.store.get_value(iter, CRUFT_COL)
            name = cruft.get_name()
            shown = (self.show_previously_ignored or
                     not self.app.state.was_previously_ignored(name))
            self.store.set_value(iter, SHOW_COL, shown)
            iter = self.store.iter_next(iter)
    
    def treeview_size_allocate(self, treeview, *args):
        column = treeview.get_column(NAME_COL)
        name_cr = column.get_cell_renderers()[0]
        x, y, width, height = name_cr.get_size(treeview, None)
        width = column.get_width()
        name_cr.set_property("wrap-width", width)
        
    on_unused_treeview_size_allocate = treeview_size_allocate
    on_recommended_treeview_size_allocate = treeview_size_allocate
    on_optimize_treeview_size_allocate = treeview_size_allocate

    def on_quit_menuitem_activate(self, *args):
        self.gtk.main_quit()

    on_window_delete_event = on_quit_menuitem_activate

    def on_window_map_event(self, *args):
        if self.first_map:
            self.first_map = False
            self.require_root()
            self.require_working_apt_cache()
            self.show_cruft()

    def treeview_button_press_event(self, treeview, event):
        # We handle mouse button presses ourselves so that we can either
        # toggle the long description (button 1, typically left) or 
        # pop up a menu (button 3, typically right).
        #
        # This is slightly tricky and probably a source of bugs.
        # Oh well.
        
        if event.button == 1:
            # Select row being clicked on. Also show/hide its long 
            # description. But only if click is on the name
            # portion of the column, not the toggle button.
            x = int(event.x)
            y = int(event.y)
            time = event.time
            pathinfo = treeview.get_path_at_pos(x, y)
            if pathinfo:
                path, col, cellx, celly = pathinfo
                if col in self.name_cols:
                    treeview.set_cursor(path, col, False)
                    self.toggle_long_description(treeview)
                else:
                    return False
            return True
        if event.button == 3:
            # Popup a menu
            x = int(event.x)
            y = int(event.y)
            time = event.time
            pathinfo = treeview.get_path_at_pos(x, y)
            if pathinfo:
                path, col, cellx, celly = pathinfo
                treeview.grab_focus()
                treeview.set_cursor(path, col, False)
                menu = self.popup_menus[treeview]
                menu.popup(None, None, None, event.button, time)
            return True

    on_unused_treeview_button_press_event = treeview_button_press_event
    on_recommended_treeview_button_press_event = treeview_button_press_event
    on_optimize_treeview_button_press_event = treeview_button_press_event

    def popup_menu_select_all(self, menuitem, treeview):
        self.foreach_set_state(treeview, True)

    def popup_menu_unselect_all(self, menuitem, treeview):
        self.foreach_set_state(treeview, False)

    def toggled(self, cr, path, treeview):
        model = treeview.get_model()
        filter_iter = model.get_iter(path)
        iter = model.convert_iter_to_child_iter(filter_iter)
        cruft = self.store.get_value(iter, CRUFT_COL)
        cruft_name = cruft.get_name()
        enabled = self.app.state.is_enabled(cruft.get_name())
        enabled = not enabled
        if enabled:
            self.app.state.enable(cruft_name)
        else:
            self.app.state.disable(cruft_name)
        self.app.state.save(self.options.state_file)
        self.store.set_value(iter, STATE_COL, enabled)
        self.set_sensitive_unlocked()

    def on_sort_by_name_toggled(self, menuitem):
        if menuitem.get_active():
            self.sort_crufts_by_current_order = self.sort_crufts_by_name
        else:
            self.sort_crufts_by_current_order = self.sort_crufts_by_size
        self.sort_crufts_by_current_order()

    def on_borednow_messagedialog_close(self, dialog):
        dialog.hide()

    def on_borednow_messagedialog_response(self, dialog, response):
        dialog.hide()

