/* GData plugin for Claws-Mail
 * Copyright (C) 2011 Holger Berndt
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif
#include "pluginconfig.h"

#include "cm_gdata_contacts.h"
#include "cm_gdata_prefs.h"

#include <gtk/gtk.h>
#include "alertpanel.h"
#include "addr_compl.h"
#include "main.h"
#include "prefs_common.h"

#include <gdata/gdata.h>

#include "gettext.h"


typedef struct
{
  const gchar *family_name;
  const gchar *given_name;
  const gchar *full_name;
  const gchar *address;
} Contact;

typedef struct
{
  GDateTime *datetime;
  GSList *contacts;
} CmGDataContactsCache;


CmGDataContactsCache contacts_cache;


static void add_gdata_contact_to_cache(GDataContactsContact *contact)
{
  GList *walk;
  for(walk = gdata_contacts_contact_get_email_addresses(contact); walk; walk = walk->next) {
    const gchar *email_address;
    GDataGDEmailAddress *address = GDATA_GD_EMAIL_ADDRESS(walk->data);

    email_address = gdata_gd_email_address_get_address(address);
    if(email_address && (*email_address != '\0')) {
      GDataGDName *name;
      Contact *cached_contact;

      name = gdata_contacts_contact_get_name(contact);

      cached_contact = g_new0(Contact, 1);
      cached_contact->full_name = g_strdup(gdata_gd_name_get_full_name(name));
      cached_contact->given_name = g_strdup(gdata_gd_name_get_given_name(name));
      cached_contact->family_name = g_strdup(gdata_gd_name_get_family_name(name));
      cached_contact->address = g_strdup(email_address);

      contacts_cache.contacts = g_slist_prepend(contacts_cache.contacts, cached_contact);

      debug_print("GData plugin: Added %s <%s>\n", cached_contact->full_name, cached_contact->address);
    }
  }
}

static void free_contact(Contact *contact)
{
  g_free((gpointer)contact->full_name);
  g_free((gpointer)contact->family_name);
  g_free((gpointer)contact->given_name);
  g_free((gpointer)contact->address);
  g_free(contact);
}

static void clear_contacts_cache(void)
{
  GSList *walk;
  for(walk = contacts_cache.contacts; walk; walk = walk->next)
    free_contact(walk->data);
  g_slist_free(contacts_cache.contacts);
  contacts_cache.contacts = NULL;
}

static void query(void)
{
  GDataContactsService *service;
  GDataContactsQuery *query;
  GDataFeed *feed;
  GList *walk;
  GError *error = NULL;

  service = gdata_contacts_service_new(CM_GDATA_CLIENT_ID);

  if(!gdata_service_authenticate(GDATA_SERVICE(service), cm_gdata_config.username, cm_gdata_config.password, NULL, &error)) {
    g_object_unref(service);
    alertpanel_error(_("GData plugin: Authentification error: %s"), error->message);
    g_error_free(error);
    return;
  }

  query = gdata_contacts_query_new(NULL);
  gdata_query_set_max_results(GDATA_QUERY(query), cm_gdata_config.max_num_results);
  feed = gdata_contacts_service_query_contacts(service, GDATA_QUERY(query), NULL, NULL, NULL, &error);
  g_object_unref(query);
  g_object_unref(service);
  if(error) {
    g_object_unref(feed);
    alertpanel_error(_("GData plugin: Error querying for contacts: %s"), error->message);
    g_error_free(error);
    return;
  }

  /* clear cache */
  clear_contacts_cache();

  /* Iterate through the returned contacts and fill the cache */
  for(walk = gdata_feed_get_entries(feed); walk; walk = walk->next)
    add_gdata_contact_to_cache(GDATA_CONTACTS_CONTACT(walk->data));
  g_object_unref(feed);
}


static void add_contacts_to_list(GList **address_list, GSList *contacts)
{
  GSList *walk;

  for(walk = contacts; walk; walk = walk->next)
  {
    address_entry *ae;
    Contact *contact = walk->data;

    ae = g_new0(address_entry, 1);
    ae->name = g_strdup(contact->full_name);
    ae->address = g_strdup(contact->address);
    ae->grp_emails = NULL;

    *address_list = g_list_prepend(*address_list, ae);
    addr_compl_add_address1(ae->address, ae);

    if(contact->given_name && *(contact->given_name) != '\0')
      addr_compl_add_address1(contact->given_name, ae);

    if(contact->family_name && *(contact->family_name) != '\0')
      addr_compl_add_address1(contact->family_name, ae);
  }
}

void cm_gdata_add_contacts(GList **address_list)
{
  GDateTime *now;

  if(prefs_common.work_offline)
  {
    debug_print("GData plugin: Offline mode\n");
    return;
  }

  /* noop if no credentials are given */
  if(!cm_gdata_config.username || *(cm_gdata_config.username) == '\0' || !cm_gdata_config.password)
  {
    debug_print("GData plugin: Empty username or password\n");
    return;
  }

  /* If no query has been done yet, or last query is old, perform a new query to fill the cache. */
  now = g_date_time_new_now_local();
  if(!contacts_cache.datetime || g_date_time_difference(now, contacts_cache.datetime) > (cm_gdata_config.max_cache_age*1000000))
  {
    debug_print("GData plugin: Querying contacts\n");
    query();
    if(contacts_cache.datetime)
      g_date_time_unref(contacts_cache.datetime);
    g_date_time_ref(now);
    contacts_cache.datetime = now;
  }
  g_date_time_unref(now);

  add_contacts_to_list(address_list, contacts_cache.contacts);
}


void cm_gdata_contacts_done(void)
{
  if(contacts_cache.contacts)
    clear_contacts_cache();
  if(contacts_cache.datetime)
  {
    g_date_time_unref(contacts_cache.datetime);
    contacts_cache.datetime = NULL;
  }
}
