// Copyright (c) 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "net/base/x509_certificate.h"

#include <stdlib.h>

#include <algorithm>
#include <map>
#include <string>
#include <vector>

#include "base/lazy_instance.h"
#include "base/logging.h"
#include "base/memory/singleton.h"
#include "base/metrics/histogram.h"
#include "base/pickle.h"
#include "base/sha1.h"
#include "base/string_piece.h"
#include "base/string_util.h"
#include "base/synchronization/lock.h"
#include "base/time.h"
#include "googleurl/src/url_canon_ip.h"
#include "net/base/cert_status_flags.h"
#include "net/base/cert_verify_result.h"
#include "net/base/net_errors.h"
#include "net/base/net_util.h"
#include "net/base/pem_tokenizer.h"

namespace net {

namespace {

// Indicates the order to use when trying to decode binary data, which is
// based on (speculation) as to what will be most common -> least common
const X509Certificate::Format kFormatDecodePriority[] = {
  X509Certificate::FORMAT_SINGLE_CERTIFICATE,
  X509Certificate::FORMAT_PKCS7
};

// The PEM block header used for DER certificates
const char kCertificateHeader[] = "CERTIFICATE";
// The PEM block header used for PKCS#7 data
const char kPKCS7Header[] = "PKCS7";

// A thread-safe cache for OS certificate handles.
//
// Within each of the supported underlying crypto libraries, a certificate
// handle is represented as a ref-counted object that contains the parsed
// data for the certificate. In addition, the underlying OS handle may also
// contain a copy of the original ASN.1 DER used to constructed the handle.
//
// In order to reduce the memory usage when multiple SSL connections exist,
// with each connection storing the server's identity certificate plus any
// intermediates supplied, the certificate handles are cached. Any two
// X509Certificates that were created from the same ASN.1 DER data,
// regardless of where that data came from, will share the same underlying
// OS certificate handle.
class X509CertificateCache {
 public:
  // Performs a compare-and-swap like operation. If an OS certificate handle
  // for the same certificate data as |*cert_handle| already exists in the
  // cache, the original |*cert_handle| will be freed and |cert_handle|
  // will be updated to point to a duplicated reference to the existing cached
  // certificate, with the caller taking ownership of this duplicated handle.
  // If an equivalent OS certificate handle is not found, a duplicated
  // reference to |*cert_handle| will be added to the cache. In either case,
  // upon return, the caller fully owns |*cert_handle| and is responsible for
  // calling FreeOSCertHandle(), after first calling Remove().
  void InsertOrUpdate(X509Certificate::OSCertHandle* cert_handle);

  // Decrements the cache reference count for |cert_handle|, a handle that was
  // previously obtained by calling InsertOrUpdate(). If this is the last
  // cached reference held, this will remove the handle from the cache. The
  // caller retains ownership of |cert_handle| and remains responsible for
  // calling FreeOSCertHandle() to release the underlying OS certificate
  void Remove(X509Certificate::OSCertHandle cert_handle);

 private:
  // A single entry in the cache. Certificates will be keyed by their SHA1
  // fingerprints, but will not be considered equivalent unless the entire
  // certificate data matches.
  struct Entry {
    Entry() : cert_handle(NULL), ref_count(0) {}

    X509Certificate::OSCertHandle cert_handle;

    // Increased by each call to InsertOrUpdate(), and balanced by each call
    // to Remove(). When it equals 0, all references created by
    // InsertOrUpdate() have been released, so the cache entry will be removed
    // the cached OS certificate handle will be freed.
    int ref_count;
  };
  typedef std::map<SHA1Fingerprint, Entry, SHA1FingerprintLessThan> CertMap;

  // Obtain an instance of X509CertificateCache via a LazyInstance.
  X509CertificateCache() {}
  ~X509CertificateCache() {}
  friend struct base::DefaultLazyInstanceTraits<X509CertificateCache>;

  // You must acquire this lock before using any private data of this object
  // You must not block while holding this lock.
  base::Lock lock_;

  // The certificate cache.  You must acquire |lock_| before using |cache_|.
  CertMap cache_;

  DISALLOW_COPY_AND_ASSIGN(X509CertificateCache);
};

base::LazyInstance<X509CertificateCache,
                   base::LeakyLazyInstanceTraits<X509CertificateCache> >
    g_x509_certificate_cache(base::LINKER_INITIALIZED);

void X509CertificateCache::InsertOrUpdate(
    X509Certificate::OSCertHandle* cert_handle) {
  DCHECK(cert_handle);
  SHA1Fingerprint fingerprint =
      X509Certificate::CalculateFingerprint(*cert_handle);

  X509Certificate::OSCertHandle old_handle = NULL;
  {
    base::AutoLock lock(lock_);
    CertMap::iterator pos = cache_.find(fingerprint);
    if (pos == cache_.end()) {
      // A cached entry was not found, so initialize a new entry. The entry
      // assumes ownership of the current |*cert_handle|.
      Entry cache_entry;
      cache_entry.cert_handle = *cert_handle;
      cache_entry.ref_count = 0;
      CertMap::value_type cache_value(fingerprint, cache_entry);
      pos = cache_.insert(cache_value).first;
    } else {
      bool is_same_cert =
          X509Certificate::IsSameOSCert(*cert_handle, pos->second.cert_handle);
      if (!is_same_cert) {
        // Two certificates don't match, due to a SHA1 hash collision. Given
        // the low probability, the simplest solution is to not cache the
        // certificate, which should not affect performance too negatively.
        return;
      }
      // A cached entry was found and will be used instead of the caller's
      // handle. Ensure the caller's original handle will be freed, since
      // ownership is assumed.
      old_handle = *cert_handle;
    }
    // Whether an existing cached handle or a new handle, increment the
    // cache's reference count and return a handle that the caller can own.
    ++pos->second.ref_count;
    *cert_handle = X509Certificate::DupOSCertHandle(pos->second.cert_handle);
  }
  // If the caller's handle was replaced with a cached handle, free the
  // original handle now. This is done outside of the lock because
  // |old_handle| may be the only handle for this particular certificate, so
  // freeing it may be complex or resource-intensive and does not need to
  // be guarded by the lock.
  if (old_handle) {
    X509Certificate::FreeOSCertHandle(old_handle);
    DHISTOGRAM_COUNTS("X509CertificateReuseCount", 1);
  }
}

void X509CertificateCache::Remove(X509Certificate::OSCertHandle cert_handle) {
  SHA1Fingerprint fingerprint =
      X509Certificate::CalculateFingerprint(cert_handle);
  base::AutoLock lock(lock_);

  CertMap::iterator pos = cache_.find(fingerprint);
  if (pos == cache_.end())
    return;  // A hash collision where the winning cert was already freed.

  bool is_same_cert = X509Certificate::IsSameOSCert(cert_handle,
                                                    pos->second.cert_handle);
  if (!is_same_cert)
    return;  // A hash collision where the winning cert is still around.

  if (--pos->second.ref_count == 0) {
    // The last reference to |cert_handle| has been removed, so release the
    // Entry's OS handle and remove the Entry. The caller still holds a
    // reference to |cert_handle| and is responsible for freeing it.
    X509Certificate::FreeOSCertHandle(pos->second.cert_handle);
    cache_.erase(pos);
  }
}

// CompareSHA1Hashes is a helper function for using bsearch() with an array of
// SHA1 hashes.
int CompareSHA1Hashes(const void* a, const void* b) {
  return memcmp(a, b, base::SHA1_LENGTH);
}

// Utility to split |src| on the first occurrence of |c|, if any. |right| will
// either be empty if |c| was not found, or will contain the remainder of the
// string including the split character itself.
void SplitOnChar(const base::StringPiece& src,
                 char c,
                 base::StringPiece* left,
                 base::StringPiece* right) {
  size_t pos = src.find(c);
  if (pos == base::StringPiece::npos) {
    *left = src;
    right->clear();
  } else {
    *left = src.substr(0, pos);
    *right = src.substr(pos);
  }
}

#if defined(OS_WIN)
X509Certificate::OSCertHandle CreateOSCert(base::StringPiece der_cert) {
  X509Certificate::OSCertHandle cert_handle = NULL;
  BOOL ok = CertAddEncodedCertificateToStore(
      X509Certificate::cert_store(), X509_ASN_ENCODING | PKCS_7_ASN_ENCODING,
      reinterpret_cast<const BYTE*>(der_cert.data()), der_cert.size(),
      CERT_STORE_ADD_USE_EXISTING, &cert_handle);
  return ok ? cert_handle : NULL;
}
#else
X509Certificate::OSCertHandle CreateOSCert(base::StringPiece der_cert) {
  return X509Certificate::CreateOSCertHandleFromBytes(
      const_cast<char*>(der_cert.data()), der_cert.size());
}
#endif

}  // namespace

bool X509Certificate::LessThan::operator()(X509Certificate* lhs,
                                           X509Certificate* rhs) const {
  if (lhs == rhs)
    return false;

  SHA1FingerprintLessThan fingerprint_functor;
  return fingerprint_functor(lhs->chain_fingerprint_, rhs->chain_fingerprint_);
}

X509Certificate::X509Certificate(const std::string& subject,
                                 const std::string& issuer,
                                 base::Time start_date,
                                 base::Time expiration_date)
    : subject_(subject),
      issuer_(issuer),
      valid_start_(start_date),
      valid_expiry_(expiration_date),
      cert_handle_(NULL) {
  memset(fingerprint_.data, 0, sizeof(fingerprint_.data));
  memset(chain_fingerprint_.data, 0, sizeof(chain_fingerprint_.data));
}

// static
X509Certificate* X509Certificate::CreateFromHandle(
    OSCertHandle cert_handle,
    const OSCertHandles& intermediates) {
  DCHECK(cert_handle);
  return new X509Certificate(cert_handle, intermediates);
}

// static
X509Certificate* X509Certificate::CreateFromDERCertChain(
    const std::vector<base::StringPiece>& der_certs) {
  if (der_certs.empty())
    return NULL;

  X509Certificate::OSCertHandles intermediate_ca_certs;
  for (size_t i = 1; i < der_certs.size(); i++) {
    OSCertHandle handle = CreateOSCert(der_certs[i]);
    if (!handle)
      break;
    intermediate_ca_certs.push_back(handle);
  }

  OSCertHandle handle = NULL;
  // Return NULL if we failed to parse any of the certs.
  if (der_certs.size() - 1 == intermediate_ca_certs.size())
    handle = CreateOSCert(der_certs[0]);

  X509Certificate* cert = NULL;
  if (handle) {
    cert = CreateFromHandle(handle, intermediate_ca_certs);
    FreeOSCertHandle(handle);
  }

  for (size_t i = 0; i < intermediate_ca_certs.size(); i++)
    FreeOSCertHandle(intermediate_ca_certs[i]);

  return cert;
}

// static
X509Certificate* X509Certificate::CreateFromBytes(const char* data,
                                                  int length) {
  OSCertHandle cert_handle = CreateOSCertHandleFromBytes(data, length);
  if (!cert_handle)
    return NULL;

  X509Certificate* cert = CreateFromHandle(cert_handle, OSCertHandles());
  FreeOSCertHandle(cert_handle);
  return cert;
}

// static
X509Certificate* X509Certificate::CreateFromPickle(const Pickle& pickle,
                                                   void** pickle_iter,
                                                   PickleType type) {
  OSCertHandle cert_handle = ReadOSCertHandleFromPickle(pickle, pickle_iter);
  if (!cert_handle)
    return NULL;

  OSCertHandles intermediates;
  size_t num_intermediates = 0;
  if (type == PICKLETYPE_CERTIFICATE_CHAIN) {
    if (!pickle.ReadSize(pickle_iter, &num_intermediates)) {
      FreeOSCertHandle(cert_handle);
      return NULL;
    }

    for (size_t i = 0; i < num_intermediates; ++i) {
      OSCertHandle intermediate = ReadOSCertHandleFromPickle(pickle,
                                                             pickle_iter);
      if (!intermediate)
        break;
      intermediates.push_back(intermediate);
    }
  }

  X509Certificate* cert = NULL;
  if (intermediates.size() == num_intermediates)
    cert = CreateFromHandle(cert_handle, intermediates);
  FreeOSCertHandle(cert_handle);
  for (size_t i = 0; i < intermediates.size(); ++i)
    FreeOSCertHandle(intermediates[i]);

  return cert;
}

// static
CertificateList X509Certificate::CreateCertificateListFromBytes(
    const char* data, int length, int format) {
  OSCertHandles certificates;

  // Check to see if it is in a PEM-encoded form. This check is performed
  // first, as both OS X and NSS will both try to convert if they detect
  // PEM encoding, except they don't do it consistently between the two.
  base::StringPiece data_string(data, length);
  std::vector<std::string> pem_headers;

  // To maintain compatibility with NSS/Firefox, CERTIFICATE is a universally
  // valid PEM block header for any format.
  pem_headers.push_back(kCertificateHeader);
  if (format & FORMAT_PKCS7)
    pem_headers.push_back(kPKCS7Header);

  PEMTokenizer pem_tok(data_string, pem_headers);
  while (pem_tok.GetNext()) {
    std::string decoded(pem_tok.data());

    OSCertHandle handle = NULL;
    if (format & FORMAT_PEM_CERT_SEQUENCE)
      handle = CreateOSCertHandleFromBytes(decoded.c_str(), decoded.size());
    if (handle != NULL) {
      // Parsed a DER encoded certificate. All PEM blocks that follow must
      // also be DER encoded certificates wrapped inside of PEM blocks.
      format = FORMAT_PEM_CERT_SEQUENCE;
      certificates.push_back(handle);
      continue;
    }

    // If the first block failed to parse as a DER certificate, and
    // formats other than PEM are acceptable, check to see if the decoded
    // data is one of the accepted formats.
    if (format & ~FORMAT_PEM_CERT_SEQUENCE) {
      for (size_t i = 0; certificates.empty() &&
           i < arraysize(kFormatDecodePriority); ++i) {
        if (format & kFormatDecodePriority[i]) {
          certificates = CreateOSCertHandlesFromBytes(decoded.c_str(),
              decoded.size(), kFormatDecodePriority[i]);
        }
      }
    }

    // Stop parsing after the first block for any format but a sequence of
    // PEM-encoded DER certificates. The case of FORMAT_PEM_CERT_SEQUENCE
    // is handled above, and continues processing until a certificate fails
    // to parse.
    break;
  }

  // Try each of the formats, in order of parse preference, to see if |data|
  // contains the binary representation of a Format, if it failed to parse
  // as a PEM certificate/chain.
  for (size_t i = 0; certificates.empty() &&
       i < arraysize(kFormatDecodePriority); ++i) {
    if (format & kFormatDecodePriority[i])
      certificates = CreateOSCertHandlesFromBytes(data, length,
                                                  kFormatDecodePriority[i]);
  }

  CertificateList results;
  // No certificates parsed.
  if (certificates.empty())
    return results;

  for (OSCertHandles::iterator it = certificates.begin();
       it != certificates.end(); ++it) {
    X509Certificate* result = CreateFromHandle(*it, OSCertHandles());
    results.push_back(scoped_refptr<X509Certificate>(result));
    FreeOSCertHandle(*it);
  }

  return results;
}

void X509Certificate::Persist(Pickle* pickle) {
  DCHECK(cert_handle_);
  if (!WriteOSCertHandleToPickle(cert_handle_, pickle)) {
    NOTREACHED();
    return;
  }

  if (!pickle->WriteSize(intermediate_ca_certs_.size())) {
    NOTREACHED();
    return;
  }

  for (size_t i = 0; i < intermediate_ca_certs_.size(); ++i) {
    if (!WriteOSCertHandleToPickle(intermediate_ca_certs_[i], pickle)) {
      NOTREACHED();
      return;
    }
  }
}

void X509Certificate::GetDNSNames(std::vector<std::string>* dns_names) const {
  GetSubjectAltName(dns_names, NULL);
  if (dns_names->empty())
    dns_names->push_back(subject_.common_name);
}

bool X509Certificate::HasExpired() const {
  return base::Time::Now() > valid_expiry();
}

bool X509Certificate::Equals(const X509Certificate* other) const {
  return IsSameOSCert(cert_handle_, other->cert_handle_);
}

bool X509Certificate::HasIntermediateCertificate(OSCertHandle cert) {
  for (size_t i = 0; i < intermediate_ca_certs_.size(); ++i) {
    if (IsSameOSCert(cert, intermediate_ca_certs_[i]))
      return true;
  }
  return false;
}

bool X509Certificate::HasIntermediateCertificates(const OSCertHandles& certs) {
  for (size_t i = 0; i < certs.size(); ++i) {
    if (!HasIntermediateCertificate(certs[i]))
      return false;
  }
  return true;
}

// static
bool X509Certificate::VerifyHostname(
    const std::string& hostname,
    const std::string& cert_common_name,
    const std::vector<std::string>& cert_san_dns_names,
    const std::vector<std::string>& cert_san_ip_addrs) {
  DCHECK(!hostname.empty());
  // Perform name verification following http://tools.ietf.org/html/rfc6125.
  // The terminology used in this method is as per that RFC:-
  // Reference identifier == the host the local user/agent is intending to
  //                         access, i.e. the thing displayed in the URL bar.
  // Presented identifier(s) == name(s) the server knows itself as, in its cert.

  // CanonicalizeHost requires surrounding brackets to parse an IPv6 address.
  const std::string host_or_ip = hostname.find(':') != std::string::npos ?
      "[" + hostname + "]" : hostname;
  url_canon::CanonHostInfo host_info;
  const std::string reference_name = CanonicalizeHost(host_or_ip, &host_info);
  if (reference_name.empty())
    return false;

  // Allow fallback to Common name matching?
  const bool common_name_fallback = cert_san_dns_names.empty() &&
                                    cert_san_ip_addrs.empty();

  // Fully handle all cases where |hostname| contains an IP address.
  if (host_info.IsIPAddress()) {
    if (common_name_fallback &&
        host_info.family == url_canon::CanonHostInfo::IPV4) {
      // Fallback to Common name matching. As this is deprecated and only
      // supported for compatibility refuse it for IPv6 addresses.
      return reference_name == cert_common_name;
    }
    base::StringPiece ip_addr_string(
        reinterpret_cast<const char*>(host_info.address),
        host_info.AddressLength());
    return std::find(cert_san_ip_addrs.begin(), cert_san_ip_addrs.end(),
                     ip_addr_string) != cert_san_ip_addrs.end();
  }

  // |reference_domain| is the remainder of |host| after the leading host
  // component is stripped off, but includes the leading dot e.g.
  // "www.f.com" -> ".f.com".
  // If there is no meaningful domain part to |host| (e.g. it contains no dots)
  // then |reference_domain| will be empty.
  base::StringPiece reference_host, reference_domain;
  SplitOnChar(reference_name, '.', &reference_host, &reference_domain);
  bool allow_wildcards = false;
  if (!reference_domain.empty()) {
    DCHECK(reference_domain.starts_with("."));
    // We required at least 3 components (i.e. 2 dots) as a basic protection
    // against too-broad wild-carding.
    // Also we don't attempt wildcard matching on a purely numerical hostname.
    allow_wildcards = reference_domain.rfind('.') != 0 &&
        reference_name.find_first_not_of("0123456789.") != std::string::npos;
  }

  // Now step through the DNS names doing wild card comparison (if necessary)
  // on each against the reference name. If subjectAltName is empty, then
  // fallback to use the common name instead.
  std::vector<std::string> common_name_as_vector;
  const std::vector<std::string>* presented_names = &cert_san_dns_names;
  if (common_name_fallback) {
    // Note: there's a small possibility cert_common_name is an international
    // domain name in non-standard encoding (e.g. UTF8String or BMPString
    // instead of A-label). As common name fallback is deprecated we're not
    // doing anything specific to deal with this.
    common_name_as_vector.push_back(cert_common_name);
    presented_names = &common_name_as_vector;
  }
  for (std::vector<std::string>::const_iterator it =
           presented_names->begin();
       it != presented_names->end(); ++it) {
    // Catch badly corrupt cert names up front.
    if (it->empty() || it->find('\0') != std::string::npos) {
      DVLOG(1) << "Bad name in cert: " << *it;
      continue;
    }
    std::string presented_name(StringToLowerASCII(*it));

    // Remove trailing dot, if any.
    if (*presented_name.rbegin() == '.')
      presented_name.resize(presented_name.length() - 1);

    // The hostname must be at least as long as the cert name it is matching,
    // as we require the wildcard (if present) to match at least one character.
    if (presented_name.length() > reference_name.length())
      continue;

    base::StringPiece presented_host, presented_domain;
    SplitOnChar(presented_name, '.', &presented_host, &presented_domain);

    if (presented_domain != reference_domain)
      continue;

    base::StringPiece pattern_begin, pattern_end;
    SplitOnChar(presented_host, '*', &pattern_begin, &pattern_end);

    if (pattern_end.empty()) {  // No '*' in the presented_host
      if (presented_host == reference_host)
        return true;
      continue;
    }
    pattern_end.remove_prefix(1);  // move past the *

    if (!allow_wildcards)
      continue;

    // * must not match a substring of an IDN A label; just a whole fragment.
    if (reference_host.starts_with("xn--") &&
        !(pattern_begin.empty() && pattern_end.empty()))
      continue;

    if (reference_host.starts_with(pattern_begin) &&
        reference_host.ends_with(pattern_end))
      return true;
  }
  return false;
}

int X509Certificate::Verify(const std::string& hostname, int flags,
                            CertVerifyResult* verify_result) const {
  verify_result->Reset();
  verify_result->verified_cert = const_cast<X509Certificate*>(this);

  if (IsBlacklisted()) {
    verify_result->cert_status |= CERT_STATUS_REVOKED;
    return ERR_CERT_REVOKED;
  }

  int rv = VerifyInternal(hostname, flags, verify_result);

  // This check is done after VerifyInternal so that VerifyInternal can fill in
  // the list of public key hashes.
  if (IsPublicKeyBlacklisted(verify_result->public_key_hashes)) {
    verify_result->cert_status |= CERT_STATUS_REVOKED;
    rv = MapCertStatusToNetError(verify_result->cert_status);
  }

  return rv;
}

#if !defined(USE_NSS)
bool X509Certificate::VerifyNameMatch(const std::string& hostname) const {
  std::vector<std::string> dns_names, ip_addrs;
  GetSubjectAltName(&dns_names, &ip_addrs);
  return VerifyHostname(hostname, subject_.common_name, dns_names, ip_addrs);
}
#endif

X509Certificate::X509Certificate(OSCertHandle cert_handle,
                                 const OSCertHandles& intermediates)
    : cert_handle_(DupOSCertHandle(cert_handle)) {
  X509CertificateCache* cache = g_x509_certificate_cache.Pointer();
  cache->InsertOrUpdate(&cert_handle_);
  for (size_t i = 0; i < intermediates.size(); ++i) {
    // Duplicate the incoming certificate, as the caller retains ownership
    // of |intermediates|.
    OSCertHandle intermediate = DupOSCertHandle(intermediates[i]);
    // Update the cache, which will assume ownership of the duplicated
    // handle and return a suitable equivalent, potentially from the cache.
    cache->InsertOrUpdate(&intermediate);
    intermediate_ca_certs_.push_back(intermediate);
  }
  // Platform-specific initialization.
  Initialize();
}

X509Certificate::~X509Certificate() {
  // We might not be in the cache, but it is safe to remove ourselves anyway.
  X509CertificateCache* cache = g_x509_certificate_cache.Pointer();
  if (cert_handle_) {
    cache->Remove(cert_handle_);
    FreeOSCertHandle(cert_handle_);
  }
  for (size_t i = 0; i < intermediate_ca_certs_.size(); ++i) {
    cache->Remove(intermediate_ca_certs_[i]);
    FreeOSCertHandle(intermediate_ca_certs_[i]);
  }
}

bool X509Certificate::IsBlacklisted() const {
  static const unsigned kNumSerials = 256;
  static const unsigned kSerialBytes = 16;
  static const uint8 kSerials[kNumSerials][kSerialBytes] = {
    // Not a real certificate. For testing only.
    {0x07,0x7a,0x59,0xbc,0xd5,0x34,0x59,0x60,0x1c,0xa6,0x90,0x72,0x67,0xa6,0xdd,0x1c},

    // The next nine certificates all expire on Fri Mar 14 23:59:59 2014.
    // Some serial numbers actually have a leading 0x00 byte required to
    // encode a positive integer in DER if the most significant bit is 0.
    // We omit the leading 0x00 bytes to make all serial numbers 16 bytes.

    // Subject: CN=mail.google.com
    // subjectAltName dNSName: mail.google.com, www.mail.google.com
    {0x04,0x7e,0xcb,0xe9,0xfc,0xa5,0x5f,0x7b,0xd0,0x9e,0xae,0x36,0xe1,0x0c,0xae,0x1e},
    // Subject: CN=global trustee
    // subjectAltName dNSName: global trustee
    // Note: not a CA certificate.
    {0xd8,0xf3,0x5f,0x4e,0xb7,0x87,0x2b,0x2d,0xab,0x06,0x92,0xe3,0x15,0x38,0x2f,0xb0},
    // Subject: CN=login.live.com
    // subjectAltName dNSName: login.live.com, www.login.live.com
    {0xb0,0xb7,0x13,0x3e,0xd0,0x96,0xf9,0xb5,0x6f,0xae,0x91,0xc8,0x74,0xbd,0x3a,0xc0},
    // Subject: CN=addons.mozilla.org
    // subjectAltName dNSName: addons.mozilla.org, www.addons.mozilla.org
    {0x92,0x39,0xd5,0x34,0x8f,0x40,0xd1,0x69,0x5a,0x74,0x54,0x70,0xe1,0xf2,0x3f,0x43},
    // Subject: CN=login.skype.com
    // subjectAltName dNSName: login.skype.com, www.login.skype.com
    {0xe9,0x02,0x8b,0x95,0x78,0xe4,0x15,0xdc,0x1a,0x71,0x0a,0x2b,0x88,0x15,0x44,0x47},
    // Subject: CN=login.yahoo.com
    // subjectAltName dNSName: login.yahoo.com, www.login.yahoo.com
    {0xd7,0x55,0x8f,0xda,0xf5,0xf1,0x10,0x5b,0xb2,0x13,0x28,0x2b,0x70,0x77,0x29,0xa3},
    // Subject: CN=www.google.com
    // subjectAltName dNSName: www.google.com, google.com
    {0xf5,0xc8,0x6a,0xf3,0x61,0x62,0xf1,0x3a,0x64,0xf5,0x4f,0x6d,0xc9,0x58,0x7c,0x06},
    // Subject: CN=login.yahoo.com
    // subjectAltName dNSName: login.yahoo.com
    {0x39,0x2a,0x43,0x4f,0x0e,0x07,0xdf,0x1f,0x8a,0xa3,0x05,0xde,0x34,0xe0,0xc2,0x29},
    // Subject: CN=login.yahoo.com
    // subjectAltName dNSName: login.yahoo.com
    {0x3e,0x75,0xce,0xd4,0x6b,0x69,0x30,0x21,0x21,0x88,0x30,0xae,0x86,0xa8,0x2a,0x71},

    // Bad DigiNotar leaf certificates for non-Google sites.
    {0x95,0x32,0x18,0xb7,0x04,0xcc,0x62,0xb2,0x40,0xa4,0xd0,0x18,0x0e,0x81,0x93,0xf2},
    {0x5f,0x41,0xfe,0x52,0x93,0x75,0xb1,0x4b,0xd1,0x56,0x37,0x53,0xa2,0xc9,0xc6,0x49},
    {0x22,0xec,0x36,0x83,0x9f,0x44,0x65,0xe7,0xa1,0x34,0x7b,0xae,0x69,0x2f,0x5d,0x2a},
    {0x46,0xaf,0xa2,0x75,0x58,0x2e,0x27,0x34,0xe2,0xa5,0x22,0xfc,0x6c,0x2e,0x80,0x55},
    {0xc8,0x63,0x14,0x89,0xc3,0xa6,0xaf,0xde,0xdc,0x70,0xd6,0xc1,0xb5,0x81,0xc0,0x19},
    {0x26,0xf6,0x2a,0xee,0x2b,0xae,0x06,0xee,0x71,0xe7,0xc4,0x8a,0x2b,0x4d,0x60,0xfe},
    {0xb6,0x73,0x9c,0x1a,0xdd,0x2b,0x52,0x19,0x6c,0x53,0x36,0xef,0xb6,0x7b,0x26,0x0f},
    {0x9e,0x4e,0x87,0x07,0x55,0xd9,0x4c,0xf5,0xc4,0x67,0x81,0xe6,0x83,0x49,0xe3,0xd3},
    {0x7e,0x82,0x25,0xf6,0xb4,0xbe,0x3d,0x1c,0x13,0x72,0xbb,0xc3,0x65,0xf5,0xf6,0xb6},
    {0xff,0x2e,0x1b,0xda,0xa2,0xf4,0xc9,0x51,0x02,0x65,0xf7,0x26,0x11,0xdd,0x0c,0x44},
    {0x1f,0xe8,0x4c,0xbe,0x99,0xe5,0xd2,0x68,0x4c,0x67,0x8d,0xef,0x7f,0xc0,0xbd,0xaa},
    {0x4b,0x72,0xe8,0x49,0x97,0x38,0x82,0x99,0xf5,0x83,0xcb,0x37,0x60,0xab,0x1b,0x20},
    {0xcf,0x61,0x90,0x78,0xde,0x6c,0x9d,0x9a,0x50,0x72,0xb5,0xe3,0xb1,0x3c,0xa9,0x9f},
    {0xa8,0x22,0xeb,0x37,0xb7,0x35,0xa6,0x3e,0x6c,0xa9,0xa9,0x96,0x80,0x08,0x21,0xfc},
    {0xe3,0xb6,0x00,0xb1,0x07,0x39,0xb0,0x94,0xab,0x47,0x4c,0x88,0x8d,0x97,0x25,0x0d},
    {0x36,0xe8,0x57,0x1d,0x39,0x46,0x34,0x2c,0x74,0x20,0x81,0xc0,0xfa,0xab,0x34,0x3a},
    {0x77,0x17,0x07,0x87,0x3d,0xa4,0x91,0xa8,0xd5,0xcc,0x5b,0x07,0xdc,0x50,0x8c,0x72},
    {0x1e,0x94,0xcf,0x79,0x06,0x58,0x7b,0xdf,0x45,0xc4,0x61,0xfb,0xdd,0x74,0x3e,0x25},
    {0x58,0x7c,0xd9,0xe3,0xd9,0xa8,0xab,0x10,0xfd,0xb0,0xc7,0x0b,0xcd,0x8a,0xe7,0x28},
    {0x7e,0xd8,0x0b,0x93,0x20,0x50,0x8d,0xa6,0xcf,0x5d,0x96,0xba,0x53,0x30,0x43,0xd7},
    {0xf8,0x6d,0x1e,0x22,0xbd,0x20,0x68,0x47,0x1e,0x20,0x13,0x6b,0x30,0x21,0x0e,0xae},
    {0x2c,0x86,0x94,0x86,0xec,0x04,0x75,0x97,0x82,0x91,0x6e,0x68,0x04,0xdb,0x48,0xfd},
    {0x35,0x19,0x81,0x98,0xd1,0xbc,0xaf,0xcf,0x43,0x86,0xe6,0xb8,0x87,0xba,0xe6,0x1b},
    {0x4a,0x66,0x2b,0xa9,0xe4,0x5d,0x06,0xff,0xc5,0x0d,0x1d,0xbb,0xe2,0x25,0xa8,0xbb},
    {0x89,0x9a,0xe1,0x20,0xcd,0x44,0xfc,0xec,0x0f,0xfc,0xd6,0x2f,0x6f,0xc4,0xbb,0x81},
    {0x7d,0xd1,0x6c,0x03,0xdf,0x04,0x38,0xb2,0xbe,0x5f,0xc1,0xd3,0xe1,0x9f,0x13,0x8b},
    {0xaf,0xe2,0x66,0x30,0x8f,0x50,0x1d,0x49,0x37,0xc1,0xaa,0xf0,0x00,0xc5,0x68,0x97},
    {0x53,0xd9,0xf7,0x50,0x84,0xf2,0x62,0xfb,0x40,0x91,0x27,0xab,0x1e,0x04,0x44,0x2e},
    {0x35,0xec,0x66,0x4c,0xbd,0x19,0x42,0xa2,0x36,0xc3,0xba,0x0e,0xb2,0xc6,0x81,0x3f},
    {0x5c,0x94,0xc8,0xbd,0x9d,0x50,0x4a,0xd8,0x8d,0x96,0x59,0x36,0xef,0x5f,0xa0,0x21},
    {0x88,0x82,0xc0,0xa1,0xd6,0x4b,0xab,0xb7,0xdb,0x82,0x07,0x3e,0xcd,0x77,0x72,0x2f},
    {0x81,0xc4,0x84,0x5a,0x97,0x2f,0x37,0x4d,0x1c,0xe5,0x83,0x08,0x44,0x54,0xdd,0x50},
    {0x04,0x1d,0x36,0xbb,0x4d,0xd3,0x9e,0x4a,0xce,0x8c,0x5b,0x4e,0x72,0x27,0x79,0xa4},
    {0xe6,0xc0,0xb6,0x11,0xb2,0x84,0x0a,0xac,0xc4,0x00,0xd0,0x3f,0xf2,0x7c,0x86,0xb7},
    {0xb5,0x94,0xa1,0x02,0xf8,0x9a,0xe2,0x9a,0xf6,0x2d,0x68,0x96,0x5f,0x49,0x0d,0x65},
    {0xa5,0xbe,0x65,0x4d,0xcc,0xf8,0xaa,0x3b,0xd3,0x15,0x64,0x8d,0x95,0x79,0x12,0x29},
    {0x5b,0x1d,0xd6,0x43,0xb9,0x80,0xa8,0x17,0xf7,0x86,0x71,0x16,0x97,0xd2,0x84,0x0d},
    {0x4c,0x06,0x16,0x26,0x9a,0x64,0xa3,0xe5,0x7b,0xe9,0xd0,0xeb,0x9d,0x1b,0x59,0x97},
    {0x10,0x2c,0x47,0x9d,0x70,0x61,0xdb,0x7e,0xf1,0x55,0xba,0xa6,0x41,0x0b,0x11,0x20},
    {0xd1,0x62,0x65,0x70,0x77,0x9a,0x69,0x32,0x66,0x28,0x2f,0x9f,0xdc,0x5a,0x72,0x3f},
    {0x0f,0x50,0x0a,0xe6,0x0b,0xf8,0x69,0x67,0x16,0xf2,0x67,0xc8,0x0e,0x13,0xdc,0x09},
    {0x8b,0x22,0xbf,0xf4,0x66,0xe4,0x15,0x26,0x65,0x8a,0x56,0x8d,0x40,0x00,0x70,0xfd},
    {0x4e,0xf7,0x05,0xc3,0x1e,0xff,0xc8,0x5a,0x0f,0x6f,0x74,0x52,0x8c,0x84,0xa7,0x72},
    {0x77,0x1d,0x89,0xbc,0xa1,0xd2,0x53,0xd1,0xb5,0x6a,0x61,0x41,0xcf,0x29,0xe1,0xbb},
    {0x33,0x73,0x97,0x6f,0xb7,0xe8,0x21,0xdc,0x3d,0x68,0xf9,0x47,0xfa,0xed,0xb2,0xb0},
    {0x30,0xc2,0x0f,0xff,0x9b,0xe7,0xc3,0xd5,0x82,0xc8,0x0b,0xc2,0xcb,0xeb,0x04,0x62},
    {0xc5,0x2e,0xde,0x8f,0x68,0xb3,0x98,0x38,0x83,0x34,0x44,0x25,0x2c,0xb6,0x4f,0x73},
    {0x94,0x53,0x95,0x11,0x88,0x67,0x56,0x53,0x13,0xa2,0x66,0x98,0xb9,0x5f,0x1e,0x3e},
    {0xc3,0xac,0x62,0x96,0xff,0x90,0xf0,0x9b,0xf1,0x82,0xfb,0x81,0xf8,0x65,0x00,0xda},
    {0x92,0x72,0xf9,0x8c,0xfe,0x35,0x00,0x00,0xd6,0xbc,0xaf,0x30,0x63,0x74,0x72,0x98},
    {0x36,0x9a,0xaa,0x6a,0x97,0x3d,0x0f,0x76,0x32,0x1c,0x53,0x2d,0xed,0xb3,0xbf,0xc6},
    {0x73,0x02,0x4e,0x7c,0x99,0x8b,0x3d,0xdd,0x24,0x4c,0xfd,0x31,0x3d,0x5e,0x43,0xb6},
    {0xd7,0xfb,0x96,0xe0,0x29,0x1e,0xbe,0x3e,0x01,0x5e,0x93,0xe5,0x86,0xcf,0xc7,0xd1},
    {0xb0,0x1d,0x8c,0x6f,0x2d,0x53,0x73,0xea,0xbf,0x0c,0x00,0x31,0x9e,0x92,0xae,0x95},
    {0xd7,0x06,0x98,0xd2,0x7e,0x7c,0xfa,0xfd,0x81,0x8d,0xe7,0xe2,0xea,0x31,0x9f,0x7f},
    {0x96,0x43,0x10,0xa7,0xec,0x83,0x47,0x6c,0xcd,0x99,0xd8,0x89,0x63,0x6a,0xc2,0xba},
    {0x54,0x17,0x57,0xb5,0x43,0x6f,0x1d,0x0b,0x3e,0xea,0x7d,0xce,0x49,0x8f,0x8a,0x0a},
    {0x89,0x86,0xa0,0xfc,0x6f,0xd1,0xe6,0x1b,0xef,0x2f,0xc7,0x45,0x64,0xa7,0x0b,0x18},
    {0x57,0x38,0x70,0x4f,0x75,0xa3,0xff,0x3b,0x86,0xa6,0x0a,0x55,0x96,0xed,0x09,0xb3},
    {0x09,0xe1,0xc3,0xd2,0x6b,0xaf,0x74,0xe6,0xf9,0xa1,0x2f,0x9f,0x3f,0x2d,0xe9,0xa5},
    {0x81,0xf7,0xb1,0x28,0xcd,0x71,0xbd,0x1a,0x59,0x50,0x74,0x45,0x00,0xc7,0x25,0xd4},
    {0x26,0xc6,0x83,0xa0,0x88,0x71,0x2b,0x79,0x3f,0xf4,0x1d,0x8a,0x0c,0x0e,0xc0,0x45},
    {0xd0,0x37,0xe2,0x1b,0x2f,0xe6,0xf1,0x9c,0x00,0x74,0x6e,0xb5,0xd6,0x32,0x9a,0xa2},
    {0x06,0xdb,0x21,0x87,0x57,0x84,0xe5,0x35,0x6a,0x5b,0x97,0x99,0x15,0x0f,0x84,0x4c},
    {0x2b,0x33,0xcb,0x52,0xae,0xff,0xb2,0x2e,0x23,0x30,0x9d,0x8b,0xc1,0x97,0x66,0x85},
    {0xb9,0xb4,0x53,0x1c,0xf7,0x1c,0xb2,0x60,0x72,0x8e,0x42,0x43,0x85,0xcd,0x5c,0x7e},
    {0xc4,0xfb,0xd4,0xa1,0x4c,0x59,0x5a,0xdf,0xb3,0xd2,0x44,0xfd,0x6d,0x46,0x9a,0x53},
    {0xe6,0x56,0xed,0x4a,0xa2,0x98,0xed,0xb4,0x91,0xb0,0xb7,0x20,0x86,0x8a,0x11,0x2b},
    {0x75,0x07,0x6e,0xc6,0xa2,0x1d,0x7e,0x4d,0xa8,0x7a,0x30,0x4e,0x9e,0xdd,0x02,0x62},
    {0x27,0xdf,0x84,0xc1,0x2b,0xf8,0x37,0xac,0x68,0x1f,0xfd,0x26,0x63,0x93,0x88,0xf7},
    {0x94,0xc7,0x5f,0x32,0xf1,0x00,0xa3,0xc3,0x2d,0x47,0xd0,0x1a,0xa9,0x8c,0xf9,0xa8},
    {0x83,0x51,0x2c,0xc6,0x89,0x5b,0x44,0x15,0x14,0xa0,0xa4,0x15,0xf3,0x14,0x2a,0x6c},
    {0x37,0x47,0xfe,0x8b,0x81,0xcb,0x6e,0x65,0x29,0x87,0x2d,0xe5,0x07,0x5c,0x3b,0xcb},
    {0x33,0x87,0x0a,0xc7,0x1c,0x7a,0x8e,0xc6,0x68,0xcd,0x68,0x65,0xb1,0x15,0x84,0xb6},
    {0x8a,0xa4,0x4a,0x11,0x84,0xdc,0x25,0xf9,0xa5,0x44,0x2c,0xd6,0xe1,0x39,0x16,0x5d},
    {0xdc,0x7e,0x23,0xb3,0x8c,0x34,0xea,0x5c,0xd1,0x24,0xc6,0x74,0x00,0x46,0xd4,0x3c},
    {0xdb,0x8f,0x84,0x41,0x38,0x05,0x89,0xf4,0x49,0xef,0xb9,0xe8,0xf0,0x8e,0x21,0x0c},
    {0x86,0x63,0x3b,0x95,0x72,0x80,0xbc,0x65,0xa5,0xad,0xfd,0x1d,0x15,0x3b,0xde,0x52},
    {0xb0,0x62,0x55,0xfa,0xe2,0x88,0xba,0xa6,0x6c,0x83,0xcb,0xf0,0x3f,0x5d,0x16,0x09},
    {0x1a,0x07,0xd8,0xd6,0xdd,0xc7,0xe6,0x23,0xe7,0x12,0x05,0x07,0x4a,0x05,0xce,0xa2},
    {0x97,0x00,0x4b,0x3d,0x85,0x6f,0x36,0x2c,0x3d,0x7c,0xd6,0x46,0xf6,0x05,0xcf,0xa5},
    {0x2c,0xd3,0xaa,0x47,0x3e,0x64,0x48,0x1a,0xbd,0x68,0xe0,0xb1,0x0d,0x28,0x70,0xdf},
    {0x3e,0x83,0x78,0xf1,0xe2,0x48,0x4c,0xe2,0x31,0x91,0x06,0x16,0xe4,0xb9,0x10,0x30},
    {0xd4,0x48,0x44,0xe3,0x38,0xa1,0x02,0x53,0x6c,0x0c,0x23,0x8f,0x76,0xe0,0x7e,0xd2},
    {0xed,0x52,0x13,0x57,0xb7,0x34,0x2f,0xb6,0xb5,0x6e,0x6b,0x7e,0x03,0xec,0xec,0x21},
    {0x7f,0x3d,0x39,0x52,0x19,0xc7,0xb9,0xb8,0xc6,0x0e,0x4f,0x52,0xd9,0x33,0x50,0xb2},
    {0xbe,0x69,0x1e,0x23,0x68,0x70,0xd9,0x01,0x4e,0xda,0x67,0xd3,0x9b,0x3f,0x80,0xa9},
    {0x4b,0xf3,0x02,0x0d,0x00,0xff,0xb8,0x31,0xf3,0x8a,0xf3,0x1e,0x26,0xb9,0x04,0x9a},
    {0x44,0x0e,0xea,0x70,0x5a,0x9f,0x79,0x15,0x35,0x60,0xbf,0xc4,0x4a,0x13,0x5d,0x99},
    {0xe9,0x92,0x00,0x82,0x28,0x03,0xad,0x53,0x65,0x27,0xb5,0x90,0x4d,0xac,0x04,0x6f},
    {0x63,0xf8,0x79,0x61,0x7a,0x3d,0x48,0xc0,0xc9,0x39,0x06,0xb6,0xa5,0xc9,0x99,0x8f},
    {0xd6,0xbe,0x78,0x3f,0x1f,0x34,0x35,0x79,0x63,0x0a,0xa4,0xbe,0x60,0xc8,0xb4,0xc8},
    {0xbb,0x83,0x3b,0x51,0xa8,0x2f,0x7e,0xb9,0xab,0x5c,0xff,0x4b,0x6f,0x0a,0x58,0x9e},
    {0x85,0x36,0x66,0x57,0x3c,0xdb,0xc6,0x80,0x74,0x0c,0xad,0x26,0x25,0x05,0x4a,0x22},
    {0xd6,0x2c,0x3d,0x36,0xc3,0xa9,0xa5,0xd8,0xcf,0xcb,0x76,0x0d,0x5b,0xcb,0xb1,0xf2},
    {0x46,0x0c,0x33,0x8e,0x89,0xba,0xa1,0x45,0x26,0x1f,0x2a,0x06,0x6c,0x56,0x86,0x2e},
    {0x5d,0x27,0x8c,0x8f,0xcd,0x78,0x9e,0xda,0x9d,0xef,0x09,0x6c,0x91,0xe1,0x15,0xe1},
    {0xcb,0xae,0xad,0xd5,0x76,0x4b,0x56,0x91,0xa1,0xb0,0xef,0x87,0xae,0x2b,0xb7,0xc6},
    {0x33,0x1e,0x3c,0x0b,0x5b,0x8e,0xd3,0xac,0xbf,0x72,0x7a,0x27,0x4c,0xf9,0xd4,0x32},
    {0xe5,0x66,0xe1,0x8a,0x28,0xc6,0xc4,0x4c,0xb1,0x29,0x55,0x23,0x9a,0x38,0x08,0xe3},
    {0x51,0x68,0xf4,0x42,0xbd,0xd4,0xc4,0x24,0x95,0xdb,0xb1,0xd5,0xaf,0x23,0xf5,0xf1},
    {0xb8,0x05,0x89,0x79,0xa8,0x42,0x6d,0x1b,0xbc,0x23,0x3d,0xa0,0xf7,0x6d,0x59,0x88},
    {0x88,0x60,0xfc,0xbf,0xda,0x75,0xf4,0x3e,0x6f,0x9a,0x1c,0x57,0x6b,0x27,0x23,0x3a},
    {0x39,0x83,0xdf,0x8b,0x4f,0xc3,0x30,0xb5,0x87,0x32,0x9f,0x62,0xab,0x8a,0xbb,0x13},
    {0x79,0xc8,0xe8,0xb7,0xde,0x36,0x53,0x9f,0xfc,0x4b,0x2b,0x58,0x25,0x30,0x53,0x24},
    {0x06,0xcb,0xb1,0xcc,0x51,0x15,0x6c,0x6d,0x46,0x5f,0x14,0x82,0x94,0x53,0xdd,0x68},
    {0x51,0x49,0xe0,0xbb,0xa2,0xc4,0x69,0x3b,0xde,0xe4,0xfe,0x90,0xc1,0x22,0x76,0x27},
    {0xf8,0xe5,0xe0,0xe2,0x02,0x50,0x9a,0x1a,0x21,0x7e,0x72,0x2f,0x14,0xd9,0x0e,0xf2},
    {0x8b,0x10,0x97,0x63,0xa2,0xc5,0x2b,0x0f,0x86,0x04,0x6d,0x18,0xf1,0xba,0x97,0x81},
    {0x49,0x68,0xf4,0xbd,0x42,0x6a,0x68,0xfa,0xd9,0xde,0x67,0x63,0x9c,0xcc,0x63,0x9a},
    {0x09,0xf7,0x2a,0x16,0xcf,0x28,0xd8,0x20,0xb5,0x34,0x8f,0x22,0x9e,0x1b,0x9e,0x51},
    {0x39,0x1d,0xde,0x64,0x85,0x6b,0xee,0xe6,0x9d,0x3b,0x39,0x16,0xcc,0x9a,0x00,0xdb},
    {0x50,0x6f,0xc2,0xc4,0x67,0x9a,0x8f,0x41,0x07,0x19,0x85,0x1b,0xa5,0xaf,0x10,0x6c},
    {0xfa,0x8d,0x23,0xab,0x7f,0x1d,0x4e,0x6b,0xbf,0xd5,0x1b,0x11,0x23,0x62,0x8b,0xe0},
    {0xb0,0xf8,0x09,0xbe,0x90,0xab,0xad,0xb2,0x49,0x64,0x5b,0x46,0x33,0x4d,0x46,0xcf},
    {0x35,0xab,0x29,0xcc,0x8d,0xd8,0xf9,0x5a,0x04,0x75,0x16,0xec,0xbe,0x30,0xa0,0x4f},
    {0x95,0x59,0xb1,0x04,0x8c,0xcd,0x7f,0xf7,0xbe,0x62,0xb7,0xad,0xc2,0x3c,0xd4,0xf2},
    {0x64,0xb2,0xfd,0xad,0x1c,0xd1,0xc7,0xcb,0x5a,0x07,0x87,0x4e,0x50,0xc8,0x44,0xf7},
    {0x2b,0x37,0x47,0x59,0x3e,0x64,0xf7,0xf2,0xfc,0x71,0x37,0x6e,0x94,0x11,0x86,0xd5},
    {0x94,0xcf,0xf9,0x1e,0x95,0x05,0xcb,0xfc,0x51,0x6c,0xcc,0x58,0xd6,0x7e,0x5c,0x5d},
    {0x90,0x59,0x1a,0x1f,0xb8,0x9f,0x19,0x3e,0x47,0x5f,0x65,0xe8,0xbf,0x00,0x8e,0x4c},
    {0xe7,0xd6,0x44,0x11,0xac,0x8d,0xc6,0x4c,0x98,0xaa,0x3b,0x50,0x2f,0xe9,0x11,0x37},
    {0x41,0x5c,0xa8,0xd0,0xe2,0x87,0x10,0x04,0x51,0xc5,0x8e,0xa6,0xf1,0xa9,0xbe,0xb6},
    {0x3f,0x1b,0x34,0xf0,0x9f,0x8b,0x4a,0x1e,0xad,0xb4,0x90,0xbe,0x47,0xa8,0xd0,0x62},
    {0x89,0xdb,0xb9,0x5e,0x30,0xdd,0x1f,0xe8,0x37,0xed,0xc5,0x05,0x89,0x7b,0x65,0x02},
    {0xa4,0x13,0xd7,0x25,0xf5,0x67,0xfa,0xc9,0x93,0x74,0xa6,0x89,0xb6,0xce,0xd8,0x68},
    {0x8c,0xdd,0xef,0x7b,0x6b,0x7d,0x7c,0xc9,0x2f,0x87,0xee,0x79,0x98,0x10,0x13,0xc1},
    {0x07,0xfc,0x7a,0x1e,0x11,0x5e,0x83,0xee,0x65,0xd5,0x8d,0x3d,0x81,0x9a,0xb4,0x1e},
    {0x64,0xdc,0x17,0x7e,0x6d,0xf5,0x33,0x40,0x1e,0x74,0xff,0xe8,0xce,0x4a,0x9f,0x40},
    {0x81,0x94,0xde,0x4d,0xbc,0xc9,0xe6,0x18,0x00,0x5e,0x5d,0x50,0x27,0x9f,0xfd,0xe9},
    {0xfd,0x3e,0x6f,0x62,0xbe,0x93,0xf5,0x2a,0x34,0x39,0x67,0x47,0x7a,0x3c,0xe8,0x95},
    {0xb7,0x28,0x7e,0x92,0x5d,0xae,0x57,0x1c,0xb5,0x3a,0xbb,0xa8,0x1c,0xd4,0x72,0x1b},
    {0x1a,0xc1,0x24,0xbd,0xc1,0x67,0xce,0x45,0x55,0x50,0x40,0xe4,0xfe,0xc1,0x5a,0x85},
    {0xe7,0x44,0xb2,0x33,0xa1,0xfb,0x1e,0x84,0x9b,0x58,0x73,0xca,0x2c,0x6d,0xcd,0x0b},
    {0x96,0x17,0x8d,0x5a,0x67,0xb1,0x0a,0x91,0x62,0xb2,0x48,0x01,0x96,0x9f,0x21,0xf5},
    {0x6b,0xcd,0x5f,0xf3,0x5a,0x6c,0x8f,0xbe,0xff,0x86,0x82,0x22,0xbf,0x93,0x93,0x3e},
    {0x3f,0x2a,0x5e,0xd2,0x6d,0x98,0xd8,0x6b,0x0a,0xe7,0xe8,0x76,0x67,0x11,0xdb,0x80},
    {0x7c,0x70,0x1a,0x08,0x98,0xd1,0xb2,0xbf,0xe9,0x10,0xde,0xb5,0xec,0x5f,0x90,0x95},
    {0x4b,0x50,0x25,0xd8,0xdf,0xad,0x34,0x47,0xf1,0x47,0x41,0x9a,0x3f,0x91,0x56,0x1a},
    {0x03,0x87,0x4a,0x58,0xe3,0x65,0xef,0x78,0x44,0xfb,0x92,0x42,0xb4,0x41,0xe9,0xea},
    {0x94,0xaf,0xcb,0xaa,0xa3,0x65,0x5c,0xb1,0x4e,0x34,0xc9,0x2b,0xdd,0xa3,0xf7,0x86},
    {0x13,0x22,0x01,0xd8,0x0c,0x07,0xe9,0xe2,0x99,0xb5,0x3b,0xc3,0x52,0xca,0x15,0x3b},
    {0xc5,0x7f,0x2c,0x1e,0xc6,0xb1,0x34,0x9c,0x6a,0x7e,0x92,0x04,0x43,0xac,0x5e,0x3a},
    {0x2d,0x2b,0xd9,0x79,0x3e,0xde,0x74,0x82,0x05,0x8d,0xb6,0xf7,0xdf,0xb1,0x25,0x90},
    {0x6f,0xf9,0xc0,0x8b,0xfc,0x8c,0x81,0xe7,0x05,0x1e,0x07,0xef,0x4f,0xe7,0x26,0x7f},
    {0x3a,0xa3,0x46,0xa0,0xa7,0xbb,0x0c,0x3d,0xce,0x13,0x16,0x19,0xb5,0x22,0x5d,0x8f},
    {0x89,0xe6,0xdd,0xfa,0x63,0x34,0xd3,0x32,0xdd,0x7f,0x98,0x48,0xee,0xd8,0x2d,0xf5},
    {0xed,0xd4,0xae,0x23,0xa3,0x34,0x95,0xcc,0x5d,0x77,0xae,0x7b,0xa7,0x99,0xc0,0x44},
    {0xe4,0x19,0x5f,0x34,0xb8,0x7b,0x27,0xef,0x7e,0x24,0x45,0x67,0x01,0x84,0xb0,0xe9},
    {0x8f,0x1a,0xf7,0xa6,0xb8,0xd3,0x3a,0x34,0x05,0xcb,0xd4,0x50,0xbd,0x13,0xad,0x75},
    {0x67,0xd9,0xb8,0x1d,0xe5,0x2d,0x00,0x80,0x0a,0x41,0x9b,0x1f,0x0c,0xe7,0x2b,0x8a},
    {0x18,0x09,0x91,0xac,0x8e,0xf5,0xbe,0x23,0x5f,0x14,0x5d,0xfb,0xf1,0xc1,0xe1,0x0f},
    {0x24,0x70,0x94,0xde,0x01,0x5a,0xb4,0xd7,0x66,0xe2,0x09,0x1e,0x4d,0x28,0xfd,0xde},
    {0x90,0x5d,0x96,0x0b,0xb9,0x2a,0x4e,0x49,0xd9,0xda,0xb2,0xba,0x00,0x85,0x0e,0x3e},
    {0xce,0x48,0x54,0xb3,0x78,0xd4,0xc3,0xda,0xf0,0x96,0xa5,0xaf,0x24,0x67,0x47,0x6c},
    {0xd7,0xf6,0x3e,0xf0,0x30,0x80,0x17,0x2a,0xd4,0x18,0x2c,0x6d,0xe1,0x18,0x54,0x19},
    {0x1b,0x95,0x59,0x42,0x14,0x96,0x53,0xee,0xc3,0xf4,0xf0,0x94,0xfa,0x96,0x14,0xbe},
    {0x4e,0xa9,0x43,0x41,0xaa,0x3f,0x59,0x09,0x10,0xc7,0x5d,0xca,0x63,0x2b,0x40,0x36},
    {0x1b,0x67,0x00,0xb7,0xc6,0x1d,0x0b,0xe5,0x25,0x47,0xdb,0x19,0x19,0xb7,0xcc,0x0e},
    {0x9b,0x2a,0x3b,0xaa,0xaa,0x00,0x5c,0xa4,0x8b,0xd1,0x39,0xc3,0xa4,0xb1,0xb9,0x39},
    {0x88,0x6b,0x5a,0x1e,0x93,0x22,0x84,0xed,0x20,0x37,0x9d,0x37,0xf7,0xef,0x7a,0xa6},
    {0x89,0x1e,0x8a,0xa7,0x40,0xe0,0x41,0xad,0xe3,0xa2,0xd2,0x20,0x9f,0x0f,0x6c,0x7e},
    {0xb7,0x76,0x75,0x39,0x44,0x08,0x68,0x9d,0xb4,0xb0,0xb7,0x84,0x94,0x5c,0x0f,0xea},
    {0xd6,0x32,0x73,0x2d,0xa5,0x5e,0x9a,0xc7,0x7f,0xfa,0xb9,0x46,0x11,0x35,0x8b,0x9d},
    {0x12,0xf7,0x4f,0x60,0x7b,0x9a,0xdd,0x06,0x20,0x35,0x31,0xfa,0x47,0xfc,0x03,0xd6},
    {0xc4,0x9b,0x60,0x13,0xc5,0x48,0xcc,0xdc,0x49,0x58,0x0a,0x79,0x55,0xfc,0x2a,0x7e},
    {0x53,0x1d,0x22,0x6b,0x47,0x88,0x72,0xbd,0x69,0xc0,0xa6,0x5a,0x36,0x4e,0xb0,0x8c},
    {0x53,0x82,0xcd,0x4c,0xfc,0xe9,0x97,0xc1,0xf4,0x55,0x5f,0x1a,0x08,0x04,0x4d,0x07},
    {0x10,0x37,0x4c,0x98,0x9e,0x1d,0xf5,0x58,0x53,0x15,0x20,0x71,0x06,0x3e,0x3a,0x73},
    {0x6c,0x12,0xb4,0x08,0xf7,0xf1,0x6d,0x2c,0x71,0x6b,0x67,0x02,0xd0,0x55,0x7b,0xcd},
    {0x34,0xd5,0xd1,0x5d,0x36,0x44,0xf5,0x0e,0x0b,0x3e,0xea,0xf0,0x76,0xb5,0xe9,0xee},
    {0xc8,0xa1,0x0c,0xf3,0x34,0xc2,0xae,0x63,0x60,0xb5,0xa3,0x0b,0x3d,0x86,0x41,0x7d},
    {0xa3,0xdc,0x0f,0x15,0x87,0x65,0x4d,0x4b,0xc5,0x12,0x0f,0x91,0x49,0x64,0xdb,0x48},
    {0xee,0x4d,0xcf,0xf5,0x83,0xab,0xba,0x8e,0xec,0x61,0x6e,0xaa,0x11,0x1f,0x86,0x0a},
    {0xc3,0x10,0xca,0x83,0x40,0x19,0xf9,0xac,0xa7,0xce,0xad,0x19,0x75,0xc9,0x50,0xb5},
    {0x79,0x89,0x76,0xb4,0x17,0x6e,0x91,0x66,0xc9,0xdf,0x6b,0xaf,0x07,0xdb,0x4c,0x85},
    {0xf8,0x4e,0x88,0x6c,0xbc,0x5c,0xc3,0x24,0x27,0x16,0xf1,0xcf,0xb0,0x23,0x93,0xf3},
    {0x79,0x51,0x2f,0x7c,0xd7,0x0c,0xe4,0xd7,0xb9,0x2e,0x0c,0x65,0xbd,0xda,0xae,0xa4},
    {0xdb,0xea,0xda,0xdd,0x40,0x68,0x2f,0xb5,0x5d,0x81,0x84,0xfe,0x00,0x73,0xa2,0xd9},
    {0xf5,0x88,0x10,0x6b,0xaf,0x63,0xc5,0x70,0x11,0xd4,0x63,0x93,0x58,0x71,0xba,0x1c},
    {0x6f,0xed,0x44,0x67,0x7e,0x59,0x37,0xdc,0xda,0x85,0x57,0x9f,0xc9,0xdd,0x0e,0xb5},
    {0x32,0x97,0x51,0x8c,0x51,0xd4,0x64,0x5a,0x73,0x50,0xfb,0x90,0xb9,0x3c,0x2d,0xd8},
    {0xe5,0x08,0x0a,0x59,0xb3,0xc4,0xfc,0x10,0x15,0x38,0x35,0xdd,0x45,0xdc,0x55,0x52},
    {0xfe,0xd3,0x7d,0xda,0xbb,0xb0,0xf5,0x07,0x05,0x1f,0x8d,0x86,0x46,0xf1,0x69,0x43},
    {0xd1,0x60,0x46,0xdc,0x2e,0x78,0xcf,0x18,0xce,0xe0,0x1f,0x1d,0x97,0x0e,0xe4,0xb1},
    {0x10,0xb0,0xec,0x09,0xea,0x45,0xe9,0xd1,0xd1,0xf8,0x73,0x72,0xe2,0xd4,0xc6,0x4a},
    {0xf0,0x66,0x18,0xb2,0x79,0x00,0xab,0x31,0x5c,0x8a,0x24,0x6d,0x3c,0xfd,0x11,0x06},
    {0xaf,0x77,0x28,0xb1,0x4d,0x47,0xa6,0xff,0x74,0x30,0x80,0x74,0xe6,0x7b,0x75,0xdc},
    {0xab,0x9f,0xfc,0xb0,0xf7,0xc3,0xe2,0xb6,0xaa,0x06,0x32,0xde,0xe9,0x7f,0xf0,0x4e},
    {0x67,0x91,0x59,0x6f,0x76,0x99,0x7f,0xe1,0xec,0x23,0x53,0xd3,0x21,0xe4,0x3a,0x4a},
    {0xc2,0xfd,0x99,0x93,0x4b,0x0d,0x76,0x77,0x10,0xd1,0x03,0x9e,0x35,0xec,0xaf,0xc4},
    {0x2c,0x26,0x15,0x2c,0xf9,0x93,0x38,0x9b,0x02,0x69,0xe5,0xf5,0xb7,0x63,0xc8,0x7d},
    {0x97,0x6e,0x07,0x58,0xc6,0x22,0x62,0xf5,0x9e,0x97,0xf4,0x20,0x33,0xdc,0x8e,0x1d},
    {0x8b,0x74,0x09,0xd0,0x97,0x91,0x51,0x4d,0x21,0xd4,0xa6,0xf6,0x91,0xeb,0x76,0x58},
    {0x0c,0x01,0xdf,0x15,0xc2,0x9d,0xb4,0x8e,0xe2,0xc5,0xf1,0xda,0x3d,0xab,0x74,0x6d},
    {0xa1,0x79,0x93,0x3b,0x61,0xe9,0x0e,0x45,0xf1,0x32,0xec,0x17,0x80,0x34,0x8f,0x8b},
    {0x41,0x7d,0x99,0xce,0xab,0xbc,0x1a,0xad,0xe3,0x58,0xbf,0x53,0xe1,0x92,0xa6,0xb4},
    {0xd6,0xf3,0x71,0x0a,0x57,0xac,0x0d,0xa9,0xae,0x72,0xf4,0xb8,0x29,0xbd,0x5f,0xe5},
    {0xa0,0xae,0x10,0xca,0x41,0x0d,0x93,0x32,0x9b,0x19,0xb7,0x65,0x5a,0xa3,0x18,0xed},
    {0x91,0xd0,0xa3,0x45,0x4e,0x9d,0xfc,0xeb,0xc4,0xa0,0x3d,0x4c,0xc6,0x67,0x62,0x3b},
    {0x5a,0xa1,0xa1,0x13,0x62,0x69,0x19,0xbb,0x5b,0x62,0x37,0x08,0xc5,0xb2,0x34,0xca},
    {0xd1,0x0c,0x08,0x0d,0xdf,0x1e,0xca,0xda,0x21,0xa0,0xb9,0xdd,0x4e,0x40,0x17,0x1c},
    {0x0f,0x87,0x02,0x6f,0x9f,0x9e,0xa5,0xaa,0x24,0xda,0x79,0x18,0x4e,0x46,0xbe,0x95},
    {0x54,0x32,0xfc,0x98,0x14,0x18,0x83,0xf7,0x80,0x89,0x7b,0xc8,0x29,0xeb,0x90,0x80},
    {0x57,0x44,0x26,0xc4,0x44,0x01,0x25,0x44,0x6f,0xd2,0xd3,0x04,0xcf,0x13,0x37,0x99},
    {0x70,0x71,0x2d,0x06,0x80,0x6d,0x26,0x93,0x60,0x7d,0xf9,0x69,0x79,0xa3,0x95,0xf2},
    {0x0f,0xac,0x31,0x7d,0xd9,0xf9,0x58,0xce,0xac,0x7f,0x42,0xad,0x3c,0xde,0x3e,0x3e},
    {0x23,0x30,0x77,0xd9,0xfd,0x64,0xf6,0xa7,0xff,0x4b,0xad,0x3f,0xa9,0x61,0xd1,0x4d},
    {0x38,0x68,0x36,0x7a,0xd1,0x7a,0x77,0x07,0xed,0x7e,0x19,0xb3,0x71,0x56,0x7e,0xdb},
    {0x0a,0x75,0x86,0x4a,0x93,0xb8,0xaf,0x71,0x09,0xae,0xa3,0x57,0xcc,0x45,0xb6,0x38},
    {0xed,0x15,0xbc,0xce,0x02,0x64,0x46,0x8a,0x58,0xc2,0xef,0xdb,0xc4,0x57,0xa6,0x0f},
    {0xd9,0x13,0xc8,0xae,0xf9,0x7a,0xc7,0x62,0x66,0x4f,0xe0,0x91,0xab,0xdb,0x0b,0x5b},
    {0xa2,0xcb,0xd5,0x32,0xed,0x71,0x07,0x63,0x25,0x59,0xfb,0x06,0xfa,0xb0,0xbd,0x70},
    {0x6e,0x23,0x5e,0x65,0x9c,0xc6,0xcd,0x89,0xba,0x68,0xa2,0x92,0x28,0xb3,0x74,0x10},
    {0x50,0x0d,0x88,0x2a,0x93,0x19,0x9b,0xf0,0x97,0x0e,0x96,0x62,0xbe,0x94,0x6d,0x02},
    {0x0c,0xc7,0x0a,0xc8,0xb1,0xad,0xbf,0xda,0xce,0x8b,0x51,0x34,0x16,0x85,0x8a,0xf2},
    {0xff,0x78,0x96,0x32,0xb8,0xd4,0xae,0xcd,0x94,0xa0,0xaa,0xb3,0x30,0x74,0xa0,0x58},
    {0x8a,0xd2,0x35,0xc1,0x09,0x65,0x23,0x39,0xbd,0xea,0x3a,0xc8,0x32,0x55,0xb2,0x88},
    {0x45,0xcf,0x0b,0x3f,0x9d,0xa8,0xd2,0x17,0x6e,0x5c,0x53,0x0c,0xe2,0xa7,0x4e,0xe6},
    {0xec,0x35,0x8a,0x4b,0xaa,0x38,0x45,0xeb,0x24,0xaf,0x6c,0x0c,0x34,0xef,0xb2,0xeb},
    {0x73,0xfd,0xd6,0xc7,0x65,0x3c,0xde,0xdc,0x9a,0xda,0x88,0xc8,0xe9,0x7e,0x5e,0x77},
    {0xf0,0x88,0x2a,0x36,0x10,0xa5,0xe8,0x5f,0x8a,0x17,0x49,0xb9,0x77,0x9d,0xb7,0x3c},
    {0xfe,0x73,0x90,0xdb,0x0d,0x7f,0x6c,0x64,0xba,0x20,0xfa,0xb8,0x98,0x6b,0x07,0xac},
    {0x87,0x07,0x70,0x09,0xd7,0x96,0xf7,0x0f,0xc1,0x79,0xf3,0x98,0xbc,0x94,0x7a,0x0d},
    {0x44,0x08,0x71,0x99,0x1c,0xf0,0x1e,0x3d,0x7f,0x58,0xc4,0x5e,0x20,0x3e,0xa5,0xed},
    {0x64,0xd2,0x35,0x96,0x94,0x7a,0xe5,0x1a,0xcb,0xc8,0xa2,0x4f,0xaf,0x22,0xd4,0x83},
    {0x8b,0x01,0x1d,0xaa,0x9f,0x3b,0x72,0xda,0x93,0x67,0x03,0x6c,0x8a,0x89,0xe8,0x0d},
    {0x61,0xd3,0x0f,0x6b,0x8a,0x2a,0x10,0x3d,0xcf,0x37,0x1d,0xa4,0xc0,0x04,0x60,0x02},
    {0x66,0xe4,0x6e,0x5b,0x8d,0x57,0x42,0x81,0xc9,0x35,0xe6,0x34,0x03,0x8e,0x87,0x7b},
    {0xf8,0x4b,0x75,0xfc,0xe8,0xbd,0x1f,0xf1,0x3f,0xa0,0x79,0xcc,0x39,0xd4,0x51,0x64},
    {0x62,0xf0,0x91,0x47,0xb1,0xac,0x01,0x4d,0x88,0x92,0xc0,0xf8,0x29,0x4d,0x0b,0x66},
    {0xe7,0xf5,0x86,0x83,0x06,0x61,0x12,0xdc,0x5e,0xb2,0x44,0xfc,0xf2,0x08,0xe8,0x50},
    {0x97,0x0c,0x55,0xe7,0x2e,0x6a,0x7b,0xfc,0x51,0x12,0xd8,0xa0,0xb5,0x7f,0x2e,0x18},
    {0x19,0xf1,0xf7,0xdf,0xba,0xfd,0xd3,0xf1,0x20,0x75,0xb0,0xa2,0xfa,0x78,0x84,0x8d},
    {0x9e,0xaf,0xe6,0xd3,0x83,0x69,0x72,0x39,0x64,0xbe,0x9f,0x4f,0x19,0x8e,0x36,0x16},
    {0x35,0x80,0x30,0xbc,0x92,0xa9,0x9b,0x61,0xe9,0x6a,0xee,0x63,0x0e,0x32,0xa9,0xc3},
    {0x2e,0x34,0xe5,0x92,0xca,0xbd,0xe8,0x9b,0xde,0x01,0xe7,0xfb,0xd4,0x25,0xcc,0x93},
    {0x83,0xef,0x75,0x79,0x05,0xde,0xcd,0x9f,0x97,0x3c,0x60,0x2f,0x4b,0x1e,0x1a,0x96},
    {0x5b,0xec,0xea,0x91,0x8f,0x7a,0x1a,0xe9,0xe8,0xd7,0xc7,0xa4,0xc0,0x25,0x65,0xe1},
    {0x32,0x50,0xb9,0xbe,0x0d,0xd0,0x0a,0x10,0x64,0xab,0xc9,0xbf,0xb3,0xba,0x56,0x92},
    {0x31,0x4b,0xc4,0xfb,0x88,0x16,0x17,0xd7,0x5b,0x7f,0xcc,0x8a,0xbe,0xd1,0xf9,0x19},
    {0x10,0x34,0x3e,0x96,0x72,0x69,0x92,0x7b,0x5d,0x8f,0x92,0xdc,0xae,0x4b,0xf6,0x55},
    {0x65,0x7e,0xfa,0xec,0x2e,0xdf,0x76,0x0b,0x9d,0x59,0xe2,0xd0,0x0b,0x1b,0xb9,0x8a},
    {0xb0,0x31,0x6b,0xd2,0x63,0xb0,0x28,0x79,0xb3,0x64,0x5f,0x45,0xdc,0x98,0x58,0xa0},
    {0xed,0x1a,0x10,0x08,0x19,0x0a,0x5d,0x16,0x54,0xd1,0x38,0xeb,0x8f,0xd1,0x15,0x4a},
    {0xad,0x7c,0x7f,0x0c,0xfe,0x66,0x6f,0xbe,0xde,0xf2,0x0a,0xa9,0x5f,0x22,0xf9,0x0b},
  };

  if (serial_number_.size() == kSerialBytes) {
    for (unsigned i = 0; i < kNumSerials; i++) {
      if (memcmp(kSerials[i], serial_number_.data(), kSerialBytes) == 0) {
        UMA_HISTOGRAM_ENUMERATION("Net.SSLCertBlacklisted", i, kNumSerials + 1);
        return true;
      }
    }
  }

  // Special case for DigiNotar: this serial number had a leading 0x00 byte
  static const uint8 kDigiNotarLeadingZero[15] = {
    0x17,0x7f,0xb6,0x53,0x6b,0x98,0xce,0x40,0xd5,0x4b,0x8b,0x24,0xe3,0x16,0x05
  };

  if (serial_number_.size() == sizeof(kDigiNotarLeadingZero) &&
      memcmp(serial_number_.data(), kDigiNotarLeadingZero,
             sizeof(kDigiNotarLeadingZero)) == 0) {
    UMA_HISTOGRAM_ENUMERATION("Net.SSLCertBlacklisted", kNumSerials,
                              kNumSerials + 1);
    return true;
  }

  return false;
}

// static
bool X509Certificate::IsPublicKeyBlacklisted(
    const std::vector<SHA1Fingerprint>& public_key_hashes) {
  static const unsigned kNumHashes = 7;
  static const uint8 kHashes[kNumHashes][base::SHA1_LENGTH] = {
    // Subject: CN=DigiNotar Root CA
    // Issuer: CN=Entrust.net x2 and self-signed
    {0x41, 0x0f, 0x36, 0x36, 0x32, 0x58, 0xf3, 0x0b, 0x34, 0x7d,
     0x12, 0xce, 0x48, 0x63, 0xe4, 0x33, 0x43, 0x78, 0x06, 0xa8},
    // Subject: CN=DigiNotar Cyber CA
    // Issuer: CN=GTE CyberTrust Global Root
    {0xc4, 0xf9, 0x66, 0x37, 0x16, 0xcd, 0x5e, 0x71, 0xd6, 0x95,
     0x0b, 0x5f, 0x33, 0xce, 0x04, 0x1c, 0x95, 0xb4, 0x35, 0xd1},
    // Subject: CN=DigiNotar Services 1024 CA
    // Issuer: CN=Entrust.net
    {0xe2, 0x3b, 0x8d, 0x10, 0x5f, 0x87, 0x71, 0x0a, 0x68, 0xd9,
     0x24, 0x80, 0x50, 0xeb, 0xef, 0xc6, 0x27, 0xbe, 0x4c, 0xa6},
    // Subject: CN=DigiNotar PKIoverheid CA Organisatie - G2
    // Issuer: CN=Staat der Nederlanden Organisatie CA - G2
    {0x7b, 0x2e, 0x16, 0xbc, 0x39, 0xbc, 0xd7, 0x2b, 0x45, 0x6e,
     0x9f, 0x05, 0x5d, 0x1d, 0xe6, 0x15, 0xb7, 0x49, 0x45, 0xdb},
    // Subject: CN=DigiNotar PKIoverheid CA Overheid en Bedrijven
    // Issuer: CN=Staat der Nederlanden Overheid CA
    {0xe8, 0xf9, 0x12, 0x00, 0xc6, 0x5c, 0xee, 0x16, 0xe0, 0x39,
     0xb9, 0xf8, 0x83, 0x84, 0x16, 0x61, 0x63, 0x5f, 0x81, 0xc5},
    // Subject: O=Digicert Sdn. Bhd.
    // Issuer: CN=GTE CyberTrust Global Root
    // Expires: Jul 17 15:16:54 2012 GMT
    {0x01, 0x29, 0xbc, 0xd5, 0xb4, 0x48, 0xae, 0x8d, 0x24, 0x96,
     0xd1, 0xc3, 0xe1, 0x97, 0x23, 0x91, 0x90, 0x88, 0xe1, 0x52},
    // Subject: O=Digicert Sdn. Bhd.
    // Issuer: CN=Entrust.net Certification Authority (2048)
    // Expires: Jul 16 17:53:37 2015 GMT
    {0xd3, 0x3c, 0x5b, 0x41, 0xe4, 0x5c, 0xc4, 0xb3, 0xbe, 0x9a,
     0xd6, 0x95, 0x2c, 0x4e, 0xcc, 0x25, 0x28, 0x03, 0x29, 0x81},
  };

  for (unsigned i = 0; i < kNumHashes; i++) {
    for (std::vector<SHA1Fingerprint>::const_iterator
         j = public_key_hashes.begin(); j != public_key_hashes.end(); ++j) {
      if (memcmp(j->data, kHashes[i], base::SHA1_LENGTH) == 0)
        return true;
    }
  }

  return false;
}

// static
bool X509Certificate::IsSHA1HashInSortedArray(const SHA1Fingerprint& hash,
                                              const uint8* array,
                                              size_t array_byte_len) {
  DCHECK_EQ(0u, array_byte_len % base::SHA1_LENGTH);
  const unsigned arraylen = array_byte_len / base::SHA1_LENGTH;
  return NULL != bsearch(hash.data, array, arraylen, base::SHA1_LENGTH,
                         CompareSHA1Hashes);
}

}  // namespace net
