// Copyright (c) 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ui/views/sad_tab_view.h"

#include "base/metrics/histogram.h"
#include "base/utf_string_conversions.h"
#include "chrome/browser/google/google_util.h"
#include "chrome/browser/ui/browser.h"
#include "chrome/browser/ui/webui/bug_report_ui.h"
#include "chrome/browser/userfeedback/proto/extension.pb.h"
#include "chrome/common/url_constants.h"
#include "content/browser/tab_contents/tab_contents.h"
#include "grit/generated_resources.h"
#include "grit/theme_resources.h"
#include "ui/base/l10n/l10n_util.h"
#include "ui/base/resource/resource_bundle.h"
#include "ui/gfx/font.h"
#include "views/controls/image_view.h"
#include "views/controls/label.h"
#include "views/controls/link.h"
#include "views/layout/grid_layout.h"

static const int kPadding = 20;
static const float kMessageSize = 0.65f;
static const SkColor kTextColor = SK_ColorWHITE;
static const SkColor kCrashColor = SkColorSetRGB(35, 48, 64);
static const SkColor kKillColor = SkColorSetRGB(57, 48, 88);

// Font size correction.
#if defined(CROS_FONTS_USING_BCI)
static const int kTitleFontSizeDelta = 1;
static const int kMessageFontSizeDelta = 0;
#else
static const int kTitleFontSizeDelta = 2;
static const int kMessageFontSizeDelta = 1;
#endif

SadTabView::SadTabView(TabContents* tab_contents, Kind kind)
    : tab_contents_(tab_contents),
      kind_(kind),
      painted_(false),
      message_(NULL),
      help_link_(NULL),
      feedback_link_(NULL) {
  DCHECK(tab_contents);

  // Sometimes the user will never see this tab, so keep track of the total
  // number of creation events to compare to display events.
  UMA_HISTOGRAM_COUNTS("SadTab.Created", kind_);

  // Set the background color.
  set_background(views::Background::CreateSolidBackground(
      (kind_ == CRASHED) ? kCrashColor : kKillColor));
}

SadTabView::~SadTabView() {}

void SadTabView::LinkClicked(views::Link* source, int event_flags) {
  if (tab_contents_ != NULL && source == help_link_) {
    GURL help_url =
        google_util::AppendGoogleLocaleParam(GURL(kind_ == CRASHED ?
                                                  chrome::kCrashReasonURL :
                                                  chrome::kKillReasonURL));
    tab_contents_->OpenURL(help_url, GURL(), CURRENT_TAB, PageTransition::LINK);
  } else if (tab_contents_ != NULL && source == feedback_link_) {
    browser::ShowHtmlBugReportView(
        Browser::GetBrowserForController(&tab_contents_->controller(), NULL),
        l10n_util::GetStringUTF8(IDS_KILLED_TAB_FEEDBACK_MESSAGE),
        userfeedback::ChromeOsData_ChromeOsCategory_CRASH);
  }
}

void SadTabView::Layout() {
  // Specify the maximum message width explicitly.
  message_->SizeToFit(static_cast<int>(width() * kMessageSize));
  View::Layout();
}

void SadTabView::ViewHierarchyChanged(bool is_add,
                                      views::View* parent,
                                      views::View* child) {
  if (child != this || !is_add)
    return;

  views::GridLayout* layout = views::GridLayout::CreatePanel(this);
  SetLayoutManager(layout);

  const int column_set_id = 0;
  views::ColumnSet* columns = layout->AddColumnSet(column_set_id);
  columns->AddPaddingColumn(1, kPadding);
  columns->AddColumn(views::GridLayout::CENTER, views::GridLayout::CENTER,
                     0, views::GridLayout::USE_PREF, 0, 0);
  columns->AddPaddingColumn(1, kPadding);

  views::ImageView* image = new views::ImageView();
  ResourceBundle& rb = ResourceBundle::GetSharedInstance();
  image->SetImage(
      rb.GetBitmapNamed((kind_ == CRASHED) ? IDR_SAD_TAB : IDR_KILLED_TAB));
  layout->StartRowWithPadding(0, column_set_id, 1, kPadding);
  layout->AddView(image);

  views::Label* title = new views::Label(UTF16ToWide(l10n_util::GetStringUTF16(
      (kind_ == CRASHED) ? IDS_SAD_TAB_TITLE : IDS_KILLED_TAB_TITLE)));
  const gfx::Font& base_font = rb.GetFont(ResourceBundle::BaseFont);
  title->SetFont(base_font.DeriveFont(kTitleFontSizeDelta, gfx::Font::BOLD));
  title->SetColor(kTextColor);
  layout->StartRowWithPadding(0, column_set_id, 0, kPadding);
  layout->AddView(title);

  message_ = new views::Label(UTF16ToWide(l10n_util::GetStringUTF16(
      (kind_ == CRASHED) ? IDS_SAD_TAB_MESSAGE : IDS_KILLED_TAB_MESSAGE)));
  message_->SetFont(base_font.DeriveFont(kMessageFontSizeDelta));
  message_->SetColor(kTextColor);
  message_->SetMultiLine(true);
  layout->StartRowWithPadding(0, column_set_id, 0, kPadding);
  layout->AddView(message_);

  if (tab_contents_) {
    std::wstring help_link(UTF16ToWide(l10n_util::GetStringUTF16(
        (kind_ == CRASHED) ? IDS_SAD_TAB_HELP_LINK : IDS_LEARN_MORE)));
    help_link_ = new views::Link(help_link);
    help_link_->SetFont(base_font.DeriveFont(kMessageFontSizeDelta));
    help_link_->SetNormalColor(kTextColor);
    help_link_->set_listener(this);

    if (kind_ == CRASHED) {
      size_t offset = 0;
      string16 help_text(l10n_util::GetStringFUTF16(IDS_SAD_TAB_HELP_MESSAGE,
                                                    string16(), &offset));
      views::Label* help_prefix =
          new views::Label(UTF16ToWide(help_text.substr(0, offset)));
      help_prefix->SetFont(base_font.DeriveFont(kMessageFontSizeDelta));
      help_prefix->SetColor(kTextColor);
      views::Label* help_suffix =
          new views::Label(UTF16ToWide(help_text.substr(offset)));
      help_suffix->SetFont(base_font.DeriveFont(kMessageFontSizeDelta));
      help_suffix->SetColor(kTextColor);

      const int help_column_set_id = 1;
      views::ColumnSet* help_columns = layout->AddColumnSet(help_column_set_id);
      help_columns->AddPaddingColumn(1, kPadding);
      // Center three middle columns for the help's [prefix][link][suffix].
      for (size_t column = 0; column < 3; column++)
        help_columns->AddColumn(views::GridLayout::CENTER,
            views::GridLayout::CENTER, 0, views::GridLayout::USE_PREF, 0, 0);
      help_columns->AddPaddingColumn(1, kPadding);

      layout->StartRowWithPadding(0, help_column_set_id, 0, kPadding);
      layout->AddView(help_prefix);
      layout->AddView(help_link_);
      layout->AddView(help_suffix);
    } else {
      layout->StartRowWithPadding(0, column_set_id, 0, kPadding);
      layout->AddView(help_link_);

      feedback_link_ = new views::Link(UTF16ToWide(
          l10n_util::GetStringUTF16(IDS_KILLED_TAB_FEEDBACK_LINK)));
      feedback_link_->SetFont(base_font.DeriveFont(kMessageFontSizeDelta));
      feedback_link_->SetNormalColor(kTextColor);
      feedback_link_->set_listener(this);
      layout->StartRowWithPadding(0, column_set_id, 0, kPadding);
      layout->AddView(feedback_link_);
    }
  }
  layout->AddPaddingRow(1, kPadding);
}

void SadTabView::OnPaint(gfx::Canvas* canvas) {
  if (!painted_) {
    // User actually saw the error, keep track for user experience stats.
    UMA_HISTOGRAM_COUNTS("SadTab.Displayed", kind_);
    painted_ = true;
  }
  View::OnPaint(canvas);
}
