// Copyright (c) 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

/**
 * This view displays a summary of the state of each SPDY sessions, and
 * has links to display them in the events tab.
 */
var SpdyView = (function() {
  'use strict';

  // IDs for special HTML elements in spdy_view.html
  var MAIN_BOX_ID = 'spdy-view-tab-content';
  var ENABLED_SPAN_ID = 'spdy-view-enabled-span';
  var USE_ALTERNATE_PROTOCOL_SPAN_ID = 'spdy-view-alternate-protocol-span';
  var FORCE_ALWAYS_SPAN_ID = 'spdy-view-force-always-span';
  var FORCE_OVER_SSL_SPAN_ID = 'spdy-view-force-over-ssl-span';
  var NEXT_PROTOCOLS_SPAN_ID = 'spdy-view-next-protocols-span';
  var ALTERNATE_PROTOCOL_MAPPINGS_DIV_ID =
      'spdy-view-alternate-protocol-mappings-div';
  var SESSION_NONE_SPAN_ID = 'spdy-view-session-none-span';
  var SESSION_LINK_SPAN_ID = 'spdy-view-session-link-span';
  var SESSION_DIV_ID = 'spdy-view-session-div';

  // We inherit from DivView.
  var superClass = DivView;

  /**
   * @constructor
   */
  function SpdyView() {
    assertFirstConstructorCall(SpdyView);

    // Call superclass's constructor.
    superClass.call(this, MAIN_BOX_ID);

    g_browser.addSpdySessionInfoObserver(this);
    g_browser.addSpdyStatusObserver(this);
    g_browser.addSpdyAlternateProtocolMappingsObserver(this);

    this.spdyEnabledSpan_ = $(ENABLED_SPAN_ID);
    this.spdyUseAlternateProtocolSpan_ = $(USE_ALTERNATE_PROTOCOL_SPAN_ID);
    this.spdyForceAlwaysSpan_ = $(FORCE_ALWAYS_SPAN_ID);
    this.spdyForceOverSslSpan_ = $(FORCE_OVER_SSL_SPAN_ID);
    this.spdyNextProtocolsSpan_ = $(NEXT_PROTOCOLS_SPAN_ID);

    this.spdyAlternateProtocolMappingsDiv_ =
        $(ALTERNATE_PROTOCOL_MAPPINGS_DIV_ID);
    this.spdySessionNoneSpan_ = $(SESSION_NONE_SPAN_ID);
    this.spdySessionLinkSpan_ = $(SESSION_LINK_SPAN_ID);
    this.spdySessionDiv_ = $(SESSION_DIV_ID);
  }

  // ID for special HTML element in category_tabs.html
  SpdyView.TAB_HANDLE_ID = 'tab-handle-spdy';

  cr.addSingletonGetter(SpdyView);

  SpdyView.prototype = {
    // Inherit the superclass's methods.
    __proto__: superClass.prototype,

    onLoadLogFinish: function(data) {
      return this.onSpdySessionInfoChanged(data.spdySessionInfo) &&
             this.onSpdyStatusChanged(data.spdyStatus) &&
             this.onSpdyAlternateProtocolMappingsChanged(
                 data.spdyAlternateProtocolMappings);
    },

    /**
     * If |spdySessionInfo| there are any sessions, display a single table with
     * information on each SPDY session.  Otherwise, displays "None".
     */
    onSpdySessionInfoChanged: function(spdySessionInfo) {
      this.spdySessionDiv_.innerHTML = '';

      var hasNoSession =
          (spdySessionInfo == null || spdySessionInfo.length == 0);
      setNodeDisplay(this.spdySessionNoneSpan_, hasNoSession);
      setNodeDisplay(this.spdySessionLinkSpan_, !hasNoSession);

      // Only want to be hide the tab if there's no data.  In the case of having
      // data but no sessions, still show the tab.
      if (!spdySessionInfo)
        return false;

      if (!hasNoSession) {
        var tablePrinter = createSessionTablePrinter(spdySessionInfo);
        tablePrinter.toHTML(this.spdySessionDiv_, 'styledTable');
      }

      return true;
    },

    /**
     * Displays information on the global SPDY status.
     */
    onSpdyStatusChanged: function(spdyStatus) {
      this.spdyEnabledSpan_.textContent = spdyStatus.spdy_enabled;
      this.spdyUseAlternateProtocolSpan_.textContent =
          spdyStatus.use_alternate_protocols;
      this.spdyForceAlwaysSpan_.textContent = spdyStatus.force_spdy_always;
      this.spdyForceOverSslSpan_.textContent = spdyStatus.force_spdy_over_ssl;
      this.spdyNextProtocolsSpan_.textContent = spdyStatus.next_protos;

      return true;
    },

    /**
     * If |spdyAlternateProtocolMappings| is not empty, displays a single table
     * with information on each alternate protocol enabled server.  Otherwise,
     * displays "None".
     */
    onSpdyAlternateProtocolMappingsChanged:
        function(spdyAlternateProtocolMappings) {

      this.spdyAlternateProtocolMappingsDiv_.innerHTML = '';

      if (spdyAlternateProtocolMappings != null &&
          spdyAlternateProtocolMappings.length > 0) {
        var tabPrinter = createAlternateProtocolMappingsTablePrinter(
                spdyAlternateProtocolMappings);
        tabPrinter.toHTML(
            this.spdyAlternateProtocolMappingsDiv_, 'styledTable');
      } else {
        this.spdyAlternateProtocolMappingsDiv_.innerHTML = 'None';
      }
      return true;
    }
  };

  /**
   * Creates a table printer to print out the state of list of SPDY sessions.
   */
  function createSessionTablePrinter(spdySessions) {
    var tablePrinter = new TablePrinter();
    tablePrinter.addHeaderCell('Host');
    tablePrinter.addHeaderCell('Proxy');
    tablePrinter.addHeaderCell('ID');
    tablePrinter.addHeaderCell('Active streams');
    tablePrinter.addHeaderCell('Unclaimed pushed');
    tablePrinter.addHeaderCell('Max');
    tablePrinter.addHeaderCell('Initiated');
    tablePrinter.addHeaderCell('Pushed');
    tablePrinter.addHeaderCell('Pushed and claimed');
    tablePrinter.addHeaderCell('Abandoned');
    tablePrinter.addHeaderCell('Received frames');
    tablePrinter.addHeaderCell('Secure');
    tablePrinter.addHeaderCell('Sent settings');
    tablePrinter.addHeaderCell('Received settings');
    tablePrinter.addHeaderCell('Error');

    for (var i = 0; i < spdySessions.length; i++) {
      var session = spdySessions[i];
      tablePrinter.addRow();

      tablePrinter.addCell(session.host_port_pair);
      tablePrinter.addCell(session.proxy);

      var idCell = tablePrinter.addCell(session.source_id);
      idCell.link = '#events&q=id:' + session.source_id;

      tablePrinter.addCell(session.active_streams);
      tablePrinter.addCell(session.unclaimed_pushed_streams);
      tablePrinter.addCell(session.max_concurrent_streams);
      tablePrinter.addCell(session.streams_initiated_count);
      tablePrinter.addCell(session.streams_pushed_count);
      tablePrinter.addCell(session.streams_pushed_and_claimed_count);
      tablePrinter.addCell(session.streams_abandoned_count);
      tablePrinter.addCell(session.frames_received);
      tablePrinter.addCell(session.is_secure);
      tablePrinter.addCell(session.sent_settings);
      tablePrinter.addCell(session.received_settings);
      tablePrinter.addCell(session.error);
    }
    return tablePrinter;
  }

  /**
   * Creates a table printer to print out the list of alternate protocol
   * mappings.
   */
  function createAlternateProtocolMappingsTablePrinter(
      spdyAlternateProtocolMappings) {
    var tablePrinter = new TablePrinter();
    tablePrinter.addHeaderCell('Host');
    tablePrinter.addHeaderCell('Alternate Protocol');

    for (var i = 0; i < spdyAlternateProtocolMappings.length; i++) {
      var entry = spdyAlternateProtocolMappings[i];
      tablePrinter.addRow();

      tablePrinter.addCell(entry.host_port_pair);
      tablePrinter.addCell(entry.alternate_protocol);
    }
    return tablePrinter;
  }

  return SpdyView;
})();
