// Copyright (c) 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/chromeos/system/statistics_provider.h"

#include "base/memory/singleton.h"
#include "base/task.h"
#include "chrome/browser/chromeos/system/name_value_pairs_parser.h"

namespace chromeos {
namespace system {
namespace {

// The system command that returns the hardware class.
const char kHardwareClassKey[] = "hardware_class";
const char* kHardwareClassTool[] = { "crossystem", "hwid" };
const char kUnknownHardwareClass[] = "unknown";

// Command to get machine hardware info and key/value delimiters.
// /tmp/machine-info is generated by platform/init/chromeos_startup.
const char* kMachineHardwareInfoTool[] = { "cat", "/tmp/machine-info" };
const char kMachineHardwareInfoEq[] = "=";
const char kMachineHardwareInfoDelim[] = " \n";

// Command to get machine OS info and key/value delimiters.
const char* kMachineOSInfoTool[] = { "cat", "/etc/lsb-release" };
const char kMachineOSInfoEq[] = "=";
const char kMachineOSInfoDelim[] = "\n";

// Command to get VPD info and key/value delimiters.
const char* kVpdTool[] = { "cat", "/var/log/vpd_2.0.txt" };
const char kVpdEq[] = "=";
const char kVpdDelim[] = "\n";

}  // namespace

class StatisticsProviderImpl : public StatisticsProvider {
 public:
  // StatisticsProvider implementation:
  virtual bool GetMachineStatistic(const std::string& name,
                                   std::string* result);

  static StatisticsProviderImpl* GetInstance();

 private:
  friend struct DefaultSingletonTraits<StatisticsProviderImpl>;

  StatisticsProviderImpl();

  // Updates the machine statistcs by examining the system.
  void UpdateMachineStatistics();

  NameValuePairsParser::NameValueMap machine_info_;

  DISALLOW_COPY_AND_ASSIGN(StatisticsProviderImpl);
};

bool StatisticsProviderImpl::GetMachineStatistic(
    const std::string& name, std::string* result) {
  NameValuePairsParser::NameValueMap::iterator iter = machine_info_.find(name);
  if (iter != machine_info_.end()) {
    *result = iter->second;
    return true;
  }
  return false;
}

StatisticsProviderImpl::StatisticsProviderImpl() {
  // Get Statistics
  UpdateMachineStatistics();
}

void StatisticsProviderImpl::UpdateMachineStatistics() {
  NameValuePairsParser parser(&machine_info_);
  if (!parser.GetSingleValueFromTool(arraysize(kHardwareClassTool),
                                     kHardwareClassTool,
                                     kHardwareClassKey)) {
    // Use kUnknownHardwareClass if the hardware class command fails.
    parser.AddNameValuePair(kHardwareClassKey, kUnknownHardwareClass);
  }
  parser.ParseNameValuePairsFromTool(arraysize(kMachineHardwareInfoTool),
                                     kMachineHardwareInfoTool,
                                     kMachineHardwareInfoEq,
                                     kMachineHardwareInfoDelim);
  parser.ParseNameValuePairsFromTool(arraysize(kMachineOSInfoTool),
                                     kMachineOSInfoTool,
                                     kMachineOSInfoEq,
                                     kMachineOSInfoDelim);
  parser.ParseNameValuePairsFromTool(
      arraysize(kVpdTool), kVpdTool, kVpdEq, kVpdDelim);
}

StatisticsProviderImpl* StatisticsProviderImpl::GetInstance() {
  return Singleton<StatisticsProviderImpl,
                   DefaultSingletonTraits<StatisticsProviderImpl> >::get();
}

StatisticsProvider* StatisticsProvider::GetInstance() {
  return StatisticsProviderImpl::GetInstance();
}

}  // namespace system
}  // namespace chromeos

// Allows InvokeLater without adding refcounting. StatisticsProviderImpl is a
// Singleton and won't be deleted until it's last InvokeLater is run.
DISABLE_RUNNABLE_METHOD_REFCOUNT(chromeos::system::StatisticsProviderImpl);
