// Copyright (c) 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "webkit/fileapi/file_system_context.h"

#include "base/basictypes.h"
#include "base/file_path.h"
#include "base/logging.h"
#include "base/memory/scoped_ptr.h"
#include "base/message_loop_proxy.h"
#include "base/string_number_conversions.h"
#include "googleurl/src/gurl.h"
#include "testing/gtest/include/gtest/gtest.h"
#include "webkit/quota/mock_special_storage_policy.h"
#include "webkit/quota/quota_manager.h"

namespace fileapi {
namespace {

static const char* const kTestOrigins[] = {
  "https://a.com/",
  "http://b.com/",
  "http://c.com:1/",
  "file:///",
};

scoped_refptr<FileSystemContext> NewFileSystemContext(
    bool allow_file_access,
    bool unlimited_quota,
    scoped_refptr<quota::SpecialStoragePolicy> special_storage_policy) {
  return new FileSystemContext(base::MessageLoopProxy::current(),
                               base::MessageLoopProxy::current(),
                               special_storage_policy,
                               NULL /* quota manager */,
                               FilePath(), false /* is_incognito */,
                               allow_file_access, unlimited_quota, NULL);
}

}  // anonymous namespace

TEST(FileSystemContextTest, IsStorageUnlimited) {
  // Regular cases.
  scoped_refptr<FileSystemContext> context(
      NewFileSystemContext(false, false, NULL));
  for (size_t i = 0; i < ARRAYSIZE_UNSAFE(kTestOrigins); ++i) {
    SCOPED_TRACE(testing::Message() << "IsStorageUnlimited w/o policy #"
                 << i << " " << kTestOrigins[i]);
    EXPECT_FALSE(context->IsStorageUnlimited(GURL(kTestOrigins[i])));
  }

  // With unlimited_quota=true cases.
  context = NewFileSystemContext(false, true, NULL);
  for (size_t i = 0; i < ARRAYSIZE_UNSAFE(kTestOrigins); ++i) {
    SCOPED_TRACE(testing::Message() << "IsStorageUnlimited /w "
                 "unlimited_quota=true #" << i << " " << kTestOrigins[i]);
    EXPECT_TRUE(context->IsStorageUnlimited(GURL(kTestOrigins[i])));
  }

  // With SpecialStoragePolicy.
  scoped_refptr<quota::MockSpecialStoragePolicy> policy(
      new quota::MockSpecialStoragePolicy);
  policy->AddUnlimited(GURL(kTestOrigins[1]));

  context = NewFileSystemContext(false, false, policy);
  for (size_t i = 0; i < ARRAYSIZE_UNSAFE(kTestOrigins); ++i) {
    SCOPED_TRACE(testing::Message() << "IsStorageUnlimited /w policy #"
                 << i << " " << kTestOrigins[i]);
    GURL origin(kTestOrigins[i]);
    EXPECT_EQ(policy->IsStorageUnlimited(origin),
              context->IsStorageUnlimited(origin));
  }
}

}  // namespace fileapi
