/* Copyright (c) 2011 The Chromium Authors. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be
 * found in the LICENSE file.
 */
#ifndef PPAPI_C_DEV_PPP_PRINTING_DEV_H_
#define PPAPI_C_DEV_PPP_PRINTING_DEV_H_

#include "ppapi/c/pp_bool.h"
#include "ppapi/c/pp_instance.h"
#include "ppapi/c/pp_macros.h"
#include "ppapi/c/pp_rect.h"
#include "ppapi/c/pp_resource.h"
#include "ppapi/c/pp_stdint.h"

typedef enum {
  PP_PRINTORIENTATION_NORMAL         = 0,
  PP_PRINTORIENTATION_ROTATED_90_CW  = 1,
  PP_PRINTORIENTATION_ROTATED_180    = 2,
  PP_PRINTORIENTATION_ROTATED_90_CCW = 3
} PP_PrintOrientation_Dev;
PP_COMPILE_ASSERT_SIZE_IN_BYTES(PP_PrintOrientation_Dev, 4);

/* TODO(dmichael): We should make published .h files contain only the latest
                   version, with no explicit version number in the type name.
                   For internal versions to be used by Chrome and NaCl, we
                   should include all versions for every type, with no 'default'
                   version (i.e., all type names are versioned). */
typedef enum {
  PP_PRINTOUTPUTFORMAT_RASTER_0_3     = 0,
  PP_PRINTOUTPUTFORMAT_PDF_0_3        = 1,
  PP_PRINTOUTPUTFORMAT_POSTSCRIPT_0_3 = 2
} PP_PrintOutputFormat_Dev_0_3;
typedef enum {
  PP_PRINTOUTPUTFORMAT_RASTER_0_4     = 1u << 0,
  PP_PRINTOUTPUTFORMAT_PDF_0_4        = 1u << 1,
  PP_PRINTOUTPUTFORMAT_POSTSCRIPT_0_4 = 1u << 2
} PP_PrintOutputFormat_Dev_0_4;

/* TODO(dmichael):  Remove the conditional compilation when NaCl proxy is
                    ported to 0.4. Remove 0.3 when PDF is ported. */
#ifdef PPP_PRINTING_DEV_USE_0_4
typedef PP_PrintOutputFormat_Dev_0_4 PP_PrintOutputFormat_Dev;
#define PP_PRINTOUTPUTFORMAT_RASTER PP_PRINTOUTPUTFORMAT_RASTER_0_4
#define PP_PRINTOUTPUTFORMAT_PDF PP_PRINTOUTPUTFORMAT_PDF_0_4
#define PP_PRINTOUTPUTFORMAT_POSTSCRIPT PP_PRINTOUTPUTFORMAT_POSTSCRIPT_0_4
#else
typedef PP_PrintOutputFormat_Dev_0_3 PP_PrintOutputFormat_Dev;
#define PP_PRINTOUTPUTFORMAT_RASTER PP_PRINTOUTPUTFORMAT_RASTER_0_3
#define PP_PRINTOUTPUTFORMAT_PDF PP_PRINTOUTPUTFORMAT_PDF_0_3
#define PP_PRINTOUTPUTFORMAT_POSTSCRIPT PP_PRINTOUTPUTFORMAT_POSTSCRIPT_0_3
#endif
PP_COMPILE_ASSERT_SIZE_IN_BYTES(PP_PrintOutputFormat_Dev, 4);

struct PP_PrintSettings_Dev {
  /** This is the size of the printable area in points (1/72 of an inch) */
  struct PP_Rect printable_area;
  int32_t dpi;
  PP_PrintOrientation_Dev orientation;
  PP_Bool grayscale;
  PP_PrintOutputFormat_Dev format;
};
PP_COMPILE_ASSERT_STRUCT_SIZE_IN_BYTES(PP_PrintSettings_Dev, 32);

/* The following versioned structs are exact copies of the above unversioned
   struct, except that PP_PrintOutputFormat_Dev_0_3 or
   PP_PrintOutputFormat_Dev_0_4 is used instead of PP_PrintOutputFormat_Dev.
   This is for use by Chrome & NaCl, which always use versioned type names. Each
   must be bit-for-bit compatible with the associated version of
   PPP_PrintOutputFormat_Dev above.
   TODO(dmichael,noelallen): Fix this by having separate internal vs external
                             headers generated by IDL. */
struct PP_PrintSettings_Dev_0_3 {
  struct PP_Rect printable_area;
  int32_t dpi;
  PP_PrintOrientation_Dev orientation;
  PP_Bool grayscale;
  PP_PrintOutputFormat_Dev_0_3 format;
};

struct PP_PrintSettings_Dev_0_4 {
  struct PP_Rect printable_area;
  int32_t dpi;
  PP_PrintOrientation_Dev orientation;
  PP_Bool grayscale;
  PP_PrintOutputFormat_Dev_0_4 format;
};

/**
 * Specifies a contiguous range of page numbers to be printed.
 * The page numbers use a zero-based index.
 */
struct PP_PrintPageNumberRange_Dev {
  uint32_t first_page_number;
  uint32_t last_page_number;
};
PP_COMPILE_ASSERT_STRUCT_SIZE_IN_BYTES(PP_PrintPageNumberRange_Dev, 8);

/* Interface for the plugin to implement printing. */
#define PPP_PRINTING_DEV_INTERFACE_0_3 "PPP_Printing(Dev);0.3"
#define PPP_PRINTING_DEV_INTERFACE_0_4 "PPP_Printing(Dev);0.4"
#ifdef PPP_PRINTING_DEV_USE_0_4
#define PPP_PRINTING_DEV_INTERFACE PPP_PRINTING_DEV_INTERFACE_0_4
#else
#define PPP_PRINTING_DEV_INTERFACE PPP_PRINTING_DEV_INTERFACE_0_3
#endif

#ifdef PPP_PRINTING_DEV_USE_0_4
struct PPP_Printing_Dev {
  /**
   *  Returns a bit field representing the supported print output formats.  For
   *  example, if only Raster and PostScript are supported,
   *  QuerySupportedFormats returns a value equivalent to:
   *  (PP_PRINTOUTPUTFORMAT_RASTER | PP_PRINTOUTPUTFORMAT_POSTSCRIPT)
   */
  uint32_t (*QuerySupportedFormats)(PP_Instance instance);

  /**
   * Begins a print session with the given print settings. Calls to PrintPage
   * can only be made after a successful call to Begin. Returns the number of
   * pages required for the print output at the given page size (0 indicates
   * a failure).
   */
  int32_t (*Begin)(PP_Instance instance,
                   const struct PP_PrintSettings_Dev* print_settings);

  /**
   * Prints the specified pages using the format specified in Begin.
   * Returns a resource that represents the printed output.
   * This is a PPB_ImageData resource if the output format is
   * PP_PrintOutputFormat_Raster and a PPB_Blob otherwise. Returns 0 on
   * failure.
   */
  PP_Resource (*PrintPages)(
      PP_Instance instance,
      const struct PP_PrintPageNumberRange_Dev* page_ranges,
      uint32_t page_range_count);

  /** Ends the print session. Further calls to PrintPage will fail. */
  void (*End)(PP_Instance instance);
};
#else
struct PPP_Printing_Dev {
  /**
   * Returns an array of supported print output formats. The array is allocated
   * using PPB_Core.MemAlloc. The caller is responsible for freeing the array
   * using PPB_Core.MemFree.
   * Sets |*format_count| to 0 returns NULL if printing is not supported at
   * all.
   */
  PP_PrintOutputFormat_Dev* (*QuerySupportedFormats)(
      PP_Instance instance, uint32_t* format_count);

  /* See the alternate definition of PPP_Printing_Dev above for documentation
   * on these functions.
   */
  int32_t (*Begin)(PP_Instance instance,
                   const struct PP_PrintSettings_Dev* print_settings);
  PP_Resource (*PrintPages)(
      PP_Instance instance,
      const struct PP_PrintPageNumberRange_Dev* page_ranges,
      uint32_t page_range_count);
  void (*End)(PP_Instance instance);
};
#endif
/* The following versioned structs are exact copies of the above unversioned
   structs, except that PP_PrintSettings_Dev_0_3 or PP_PrintSettings_Dev_0_4 is
   used instead of PP_PrintSettings_Dev. This is for use by Chrome & NaCl, which
   always use versioned type names. Each must be bit-for-bit compatible with the
   associated version of PPP_Printing_Dev above.
   TODO(dmichael,noelallen): Fix this by having separate internal vs external
                             headers generated by IDL. */
struct PPP_Printing_Dev_0_4 {
  uint32_t (*QuerySupportedFormats)(PP_Instance instance);
  int32_t (*Begin)(PP_Instance instance,
                   const struct PP_PrintSettings_Dev_0_4* print_settings);
  PP_Resource (*PrintPages)(
      PP_Instance instance,
      const struct PP_PrintPageNumberRange_Dev* page_ranges,
      uint32_t page_range_count);
  void (*End)(PP_Instance instance);
};

struct PPP_Printing_Dev_0_3 {
  PP_PrintOutputFormat_Dev* (*QuerySupportedFormats)(
      PP_Instance instance, uint32_t* format_count);
  int32_t (*Begin)(PP_Instance instance,
                   const struct PP_PrintSettings_Dev_0_3* print_settings);
  PP_Resource (*PrintPages)(
      PP_Instance instance,
      const struct PP_PrintPageNumberRange_Dev* page_ranges,
      uint32_t page_range_count);
  void (*End)(PP_Instance instance);
};

#endif  /* PPAPI_C_DEV_PPP_PRINTING_DEV_H_ */

