// Copyright (c) 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/test/webdriver/commands/session_with_id.h"

#include <sstream>
#include <string>

#include "base/values.h"
#include "chrome/app/chrome_command_ids.h"
#include "chrome/common/chrome_constants.h"
#include "chrome/test/webdriver/commands/response.h"
#include "chrome/test/webdriver/session.h"
#include "chrome/test/webdriver/session_manager.h"

namespace webdriver {

SessionWithID::SessionWithID(const std::vector<std::string>& path_segments,
                             const DictionaryValue* const parameters)
    : WebDriverCommand(path_segments, parameters) {}

SessionWithID::~SessionWithID() {}

bool SessionWithID::DoesGet() {
  return true;
}

bool SessionWithID::DoesDelete() {
  return true;
}

void SessionWithID::ExecuteGet(Response* const response) {
  DictionaryValue *temp_value = new DictionaryValue();

  // Standard capabilities defined at
  // http://code.google.com/p/selenium/wiki/JsonWireProtocol#Capabilities_JSON_Object
  temp_value->SetString("browserName", "chrome");
  temp_value->SetString("version", session_->GetBrowserVersion());

#if defined(OS_WIN)
  temp_value->SetString("platform", "windows");
#elif defined(OS_MACOSX)
  temp_value->SetString("platform", "mac");
#elif defined(OS_CHROMEOS)
  temp_value->SetString("platform", "chromeos");
#elif defined(OS_LINUX)
  temp_value->SetString("platform", "linux");
#else
  temp_value->SetString("platform", "unknown");
#endif

  temp_value->SetBoolean("cssSelectorsEnabled", true);
  temp_value->SetBoolean("javascriptEnabled", true);
  temp_value->SetBoolean("takesScreenshot", true);

  // Custom non-standard session info.
  temp_value->SetWithoutPathExpansion(
      "chrome.chromedriverVersion", Value::CreateStringValue("1.0"));
  temp_value->SetWithoutPathExpansion(
      "chrome.automationVersion",
      Value::CreateStringValue(chrome::kChromeVersion));
  temp_value->SetWithoutPathExpansion(
      "chrome.nativeEvents",
      Value::CreateBooleanValue(session_->use_native_events()));

  response->SetValue(temp_value);
}

void SessionWithID::ExecuteDelete(Response* const response) {
  // Session manages its own lifetime, so do not call delete.
  session_->Terminate();
}

}  // namespace webdriver
