#!/usr/bin/python
# Copyright (c) 2011 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

"""Performance test for the HTML5 media tag.

This PyAuto powered script plays media (video or audio) files using the HTML5
tag embedded in an HTML file (specified in the GetPlayerHTMLFileName() method)
and measures decoder's side frames per second (FPS), which is generated by the
HTML file. The parameters needed to run this test are passed in the form of
environment variables (such as the number of runs). media_perf_runner.py
is used for generating these variables (PyAuto does not support direct
parameters).
"""

import pyauto_media
from media_test_base import MediaTestBase
from ui_perf_test_measure_thread import UIPerfTestMeasureThread
from ui_perf_test_utils import UIPerfTestUtils


class MediaPerformanceTest(MediaTestBase):
  """Tests for basic media performance."""
  # The following parallel lists specify data type names, their
  # corresponding units, and their corresponding trace names
  # (used in the perf graph legend).
  FPS_NAME_LIST = ['playertime', 'clocktime', 'fps',
                   'clockfps', 'droppedframe']
  FPS_UNIT_LIST = ['sec', 'sec', 'count', 'count', 'count']
  FPS_TRACE_LIST = ['t', 't', 'c', 'c', 'c']

  # Instance variables that are shared across methods.
  run_counter = 0
  chrome_renderer_process_infos = []
  measure_thread = None
  all_fps_infos = []

  def testHTML5MediaTag(self):
    """Test the HTML5 media tag."""
    MediaTestBase.ExecuteTest(self)

  def _GetFPSLog(self):
    """Get the FPS log from the DOM tree that is produced by player.html.

    Returns:
      a string FPS log from the DOM tree.
    """
    log_location = 'document.getElementById(\'history\').innerHTML'
    return self.GetDOMValue(log_location).strip()

  def PreAllRunsProcess(self):
    """A method to execute before all runs."""
    MediaTestBase.PreAllRunsProcess(self)
    self.all_fps_infos = []

  def PostAllRunsProcess(self):
    """A method to execute after all runs."""
    MediaTestBase.PostAllRunsProcess(self)
    # Print FPS related data.
    if self.all_fps_infos:
      print UIPerfTestUtils.PrintMeasuredData(
          measured_data_list=self.all_fps_infos,
          measured_data_name_list=self.FPS_NAME_LIST,
          measured_data_unit_list=self.FPS_UNIT_LIST,
          remove_first_result=True, parameter_string=self.parameter_str,
          trace_list=self.FPS_TRACE_LIST,
          reference_build=self.reference_build,
          display_filter=['playertime', 'fps', 'droppedframe'])

  def PostEachRunProcess(self, run_counter):
    """A method to execute after each run.

    Terminates the measuring thread and records the measurement in
    measure_thread.chrome_renderer_process_info.

    Args:
      run_counter: one assigned number that is generated at each run. This is
        used to store the results in result list |all_fps_infos|.
    """
    MediaTestBase.PostEachRunProcess(self, run_counter)
    all_fps_string = self._GetFPSLog()
    fps_string_list = all_fps_string.split('<br>')
    fps_list = []
    # Do the paring of the FPS string. |fps_string_list| contains multiple
    # rows.
    for fps_string in fps_string_list:
      if fps_string:
        fps_list_element_list = []
        # Each row has data which is delimited by ' '.
        for elem in fps_string.split(' '):
          # All data is in float-based strings.
          fps_list_element_list.append(float(elem))
        fps_list.append(fps_list_element_list)
    self.all_fps_infos.append(fps_list)

  def GetPlayerHTMLFileName(self):
    """A method to get the player HTML file name."""
    return 'media_fps.html'


if __name__ == '__main__':
  pyauto_media.Main()
