// Copyright (c) 2011 The Native Client Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.


// This script coordinates PNaCl Client-Side translation.
// NOTE: This is currently using deprecated facilities like __urlAsNaClDesc.

function $(id) {
  return document.getElementById(id);
}

/// Utilities for setting up a webpage with PNaCl components. These will be
/// obsoleted by nmf changes that make embedding a PNaCl component more natural.

// Class holding the utility embeds that we care about for PNaCl.
function PnaclEmbeds() {
  this.embedsToWaitFor_ = [];
  this.llcEmbed_ = null;
  this.ldEmbed_ = null;

  this.addEmbedsToWaitFor = function(embed) {
    this.embedsToWaitFor_.push(embed);
  }

  this.getEmbedsToWaitFor = function() {
    return this.embedsToWaitFor_;
  }

  this.setLLC = function(embed) {
    this.llcEmbed_ = embed;
    this.addEmbedsToWaitFor(embed);
  }

  this.getLLC = function(embed) {
    return this.llcEmbed_;
  }

  this.setLD = function(embed) {
    this.ldEmbed_ = embed;
    this.addEmbedsToWaitFor(embed);
  }

  this.getLD = function() {
    return this.ldEmbed_;
  }
}

// @public
// Creates the embed tags for the PNaCl translator nexes and
// returns the collection of wrapped in a PnaclEmbed object.
function injectUtilityEmbeds() {
  // Create a spot to inject the pnacl embed tags.
  var pnaclDiv = document.createElement('div');
  document.body.appendChild(pnaclDiv);
  var pnaclEmbeds = new PnaclEmbeds();
  pnaclEmbeds.setLLC(injectAnEmbed(pnaclDiv,
                                   'pnacl_support/llc_manifest.nmf',
                                   'llc_plugin'));
  pnaclEmbeds.setLD(injectAnEmbed(pnaclDiv,
                                  'pnacl_support/ld_manifest.nmf',
                                  'ld_plugin'));
  return pnaclEmbeds;
}

function injectAnEmbed(pnaclDiv, nacl_url, id) {
  var embed = document.createElement('embed');

  embed.width = 0;
  embed.height = 0;
  embed.src = nacl_url;
  embed.type = 'application/x-nacl';
  embed.id = id; // mostly for debugging...

  pnaclDiv.appendChild(embed);
  return embed;
}


function ISAChecker(someNexe) {
  // These must match up with the ISA strings in the NMF.
  this.X8632 = 'x86-32';
  this.X8664 = 'x86-64';
  this.ARM = 'arm';

  this.naclPlugin_ = someNexe;
  this.memoArch = null;

  this.getArch = function() {
    if (!this.memoArch) {
      this.memoArch = this.naclPlugin_.__getSandboxISA();
    }
    return this.memoArch;
  }

  this.getDashFreeArch = function() {
    var origArch = this.getArch();
    return origArch.replace('-','');
  }
}

function True(x) {
  return true;
}

function Not(pred) {
  return function (x) {
    return !pred(x);
  }
}

// Tracks the loading of LD resources (e.g., linker script)
// and initiates the arch-specific LD link.
function LDState(ldEmbed, isaChecker) {

  this.myPlugin_ = ldEmbed;
  this.ISAChecker_ = isaChecker;

  /// BEGIN
  /// Linker stuff
  ///

  // TODO(jvoung): In the future we will obtain these dependencies from .nmf:
// http://code.google.com/p/nativeclient/issues/detail?id=885
// http://code.google.com/p/nativeclient/wiki/NameResolutionForDynamicLibraries

  this.linkFiles_ = [
      'pnacl_support/@ARCH/crt1.o',
      '@GENERATED_1', // Generated by LLC (no url)
      'pnacl_support/@ARCH/libcrt_platform.a',
      'pnacl_support/@ARCH/libgcc_eh.a',
      'pnacl_support/@ARCH/libgcc.a',
      // TODO(jvoung): just install this as "ld_script" so that
      // the basename is exactly the same on all platforms.
      'pnacl_support/@ARCH/ld_script_@ARCH_untrusted'
                     ];

  this.getLinkFilesFiltered_ = function(pred) {
    var dashFreeISA = this.ISAChecker_.getDashFreeArch();
    var returnVals = [];
    var i = 0;
    var j = 0;
    for (i = 0; i < this.linkFiles_.length; i++) {
      var renamed = this.linkFiles_[i].replace(/@ARCH/g, dashFreeISA);
      if (pred(renamed)) {
        returnVals[j] = renamed;
        j++;
      }
    }
    return returnVals;
  }

  this.getLinkFiles = function() {
    return this.getLinkFilesFiltered_(True);
  }

  this.wasGenerated_ = function(file_url) {
    return file_url.match('@GENERATED');
  }

  this.wasLDScriptFile_ = function(file_url) {
    return file_url.match('ld_script');
  }

  this.getNonGeneratedLinkFiles = function() {
    return this.getLinkFilesFiltered_(Not(this.wasGenerated_));
  }

  this.findGeneratedFilename_ = function() {
    var lf = this.getLinkFilesFiltered_(this.wasGenerated_);
    // Assuming there is only one for now.
    return lf[0];
  }

  this.getNonLinkerScriptLinkFiles = function() {
    return this.getLinkFilesFiltered_(Not(this.wasLDScriptFile_));
  }

  this.findLinkerScriptFilename_ = function() {
    var lf = this.getLinkFilesFiltered_(this.wasLDScriptFile_);
    return lf[0];
  }

  // Command line flags for LD.
  this.linkStaticFlags_ = ['-nostdlib',
                           '-m',
                           '@LD_EMUL_ARCH',
                           '-T',
                           '@LD_SCRIPT_ARCH',
                           '-static'];
  this.LD_EMUL_FLAGS = {};
  this.LD_EMUL_FLAGS[this.ISAChecker_.X8632] = 'elf_nacl';
  this.LD_EMUL_FLAGS[this.ISAChecker_.X8664] = 'elf64_nacl';
  this.LD_EMUL_FLAGS[this.ISAChecker_.ARM] = 'armelf_nacl';

  this.getLDArgs_ = function() {
    // Flags, then files.
    var args = [];
    var arch = this.ISAChecker_.getArch();

    // Flags.
    for (var i = 0; i < this.linkStaticFlags_.length; i++) {
      if (this.linkStaticFlags_[i] == '@LD_EMUL_ARCH') {
        var arg = this.LD_EMUL_FLAGS[arch];
        if (arg) {
          args.push(arg);
        } else {
          throw ('Cannot find LD_EMUL_FLAGS for arch: ' + arch);
        }
      } else if (this.linkStaticFlags_[i] == '@LD_SCRIPT_ARCH') {
        var arg = this.findLinkerScriptFilename_();
        if (arg) {
          args.push(arg);
        } else {
          throw ('Cannot find LD_SCRIPT for arch: ' + arch);
        }
      } else {
        args.push(this.linkStaticFlags_[i]);
      }
    }

    // Files.
    // Skip the linker script, which is already in the flags.
    var lf = this.getNonLinkerScriptLinkFiles();
    for (var i = 0; i < lf.length; i++) {
      var url = lf[i];
      args.push(url);
    }
    return args;
  }

  this.setLDArgs_ = function() {
    var ldArgs = this.getLDArgs_();
    for (var i = 0; i < ldArgs.length; i++) {
      this.myPlugin_.AddArg(ldArgs[i]);
    }
  }

  this.addGeneratedObjectForLD_ = function(filename, fd, size) {
    this.myPlugin_.AddFileWithSize(filename, fd, size);
  }

  this.doLink = function(objFD, objSize) {
    this.setLDArgs_();
    var generatedName = this.findGeneratedFilename_();
    this.addGeneratedObjectForLD_(generatedName,
                                  objFD,
                                  objSize);
    var link_results = this.myPlugin_.Link();
    return link_results;
  }

  //////////////////////////////////////////////////////////////////////
  // Dependency Loading

  this.startLoadingDependencies = function(translator) {
    // Skip files that are generated later.
    var lf = this.getNonGeneratedLinkFiles();
    for (var i = 0; i < lf.length; i++) {
      var fileURL = lf[i];
      this.myPlugin_.__urlAsNaClDesc(fileURL,
                                     new PnaclFileLoader(translator,
                                                         fileURL));
    }
  }

  this.noteLoaded = function(fileToLoad, nacl_fd) {
    // We need to map the URL to the filename that LD will use to
    // refer to the file internally.
    this.myPlugin_.AddFile(fileToLoad, nacl_fd);
  }

  this.getNumLoadDeps = function() {
    var lf = this.getNonGeneratedLinkFiles();
    return lf.length;
  }
} // end LDState


// Tracks the loading of LLC resources (e.g., FD for bitcode)
// and initiates the arch-specific LLC compile.
function LLCState(llcEmbed, isaChecker, bitcodeURL) {
  this.myPlugin_ = llcEmbed;
  this.ISAChecker_ = isaChecker;
  this.bitcodeFD_ = 0;
  this.kNumBCFiles = 1;

  this.getLLCArgs_ = function() {
    var args = this.llcArgs_[this.ISAChecker_.getArch()];
    if (args) {
      return args;
    } else {
      throw ('Cannot find LLC ARGS for arch: ' + arch);
    }
  }

  // TODO(jvoung): slap in ARM versions of command line flags.
  this.llcArgs_ = {};
  this.llcArgs_[this.ISAChecker_.X8632] = ['-march=x86',
                                           '-mcpu=pentium4',
                                           '-asm-verbose=false',
                                           '-filetype=obj'];
  this.llcArgs_[this.ISAChecker_.X8664] = ['-march=x86-64',
                                           '-mcpu=core2',
                                           '-asm-verbose=false',
                                           '-filetype=obj'];

  this.setLLCArgs_ = function() {
    var llcArgs = this.getLLCArgs_();
    for (var i = 0; i < llcArgs.length; i++) {
      this.myPlugin_.AddArg(llcArgs[i]);
    }
  }

  this.compile = function() {
    this.setLLCArgs_();
    return this.myPlugin_.Translate(this.bitcodeFD_);
  }

  //////////////////////////////////////////////////////////////////////
  // Dependency Loading

  this.startLoadingDependencies = function(translator) {
    this.myPlugin_.__urlAsNaClDesc(bitcodeURL,
                                   new PnaclFileLoader(translator,
                                                       bitcodeURL));
  }

  this.noteLoaded = function(fileToLoad, nacl_fd) {
    this.bitcodeFD_ = nacl_fd;
  }

  this.getNumLoadDeps = function() {
    return this.kNumBCFiles;
  }
} // end LLCState


// Setups up the translation pipeline and contains the entry point
// for initiating translation.
function PnaclTranslatorState(bitcodeURL,
                              pnaclEmbeds,
                              nexeEmbed,
                              finishCallback) {
  this.bitcodeURL_ = bitcodeURL;
  this.nexePlugin_ = nexeEmbed;
  this.finishCallback_ = finishCallback;

  this.ISAChecker_ = new ISAChecker(pnaclEmbeds.getLLC());
  this.LDState_ = new LDState(pnaclEmbeds.getLD(),
                              this.ISAChecker_);
  this.LLCState_ = new LLCState(pnaclEmbeds.getLLC(),
                                this.ISAChecker_, bitcodeURL);

  /// BEGIN
  /// Dependency Loading stuff

  // Number of dependencies loaded.
  this.numLoaded_ = 0;

  // Return true if all dependencies have been loaded and FDs are available.
  // TODO(jvoung): instead of calling back to PnaclTranslatorState after
  // each dependency is loaded, call back to the component that requested
  // the resource. The component would then be responsible for
  // checking when it is ready to go, and go. For example, LLC can
  // start compiling once the bitcode is ready rather than wait for the
  // LD resources to be downloaded.
  // With that, getNumLoadDeps(), etc. would not need to be exposed,
  // and most of this class is unneeded.
  this.didLoadDependencies = function() {
    // Should be |Link Files| + |Bitcode Files|
    var finished = this.numLoaded_ == (this.LDState_.getNumLoadDeps() +
                                       this.LLCState_.getNumLoadDeps());
    if (finished) {
      this.generalLog('Finished Downloading Link files and Bitcode!');
    }
    return finished;
  }

  // Once a file is loaded, remember the FD.
  this.noteLoaded = function(fileToLoad, nacl_fd) {
    // Either loads a bitcode file, or a linker file.
    if (fileToLoad == this.bitcodeURL_) {
      this.LLCState_.noteLoaded(fileToLoad, nacl_fd);
    } else {
      this.LDState_.noteLoaded(fileToLoad, nacl_fd);
    }
    this.numLoaded_ += 1;
  }

  this.startLoadingDependencies = function() {
    this.timeStart('Downloads');
    this.LLCState_.startLoadingDependencies(this);
    this.LDState_.startLoadingDependencies(this);
  }

  /// Dependency Loading stuff
  /// END


  /// BEGIN
  /// Main coordination stuff
  ///

  // Given all the loaded FDs, look in the cache or translate, then run.
  this.translateAndRun = function() {
    this.generalLog('Translating bitcode ');
    this.timeStart('Compile');
    var transResults = this.LLCState_.compile();
    if (!transResults) {
      throw 'Call to LLC.Translate failed';
    }
    this.timeEnd('Compile');
    var objFD = transResults[0];
    var objSize = transResults[1];
    this.generalLog('.o file size is: ' + objSize);

    this.generalLog('Linking');
    this.timeStart('Link');
    var linkResults = this.LDState_.doLink(objFD, objSize);
    if (!linkResults) {
      throw 'Call to LD.Link failed';
    }
    var nexeFD = linkResults[0];
    var nexeSize = linkResults[1];
    this.timeEnd('Link');

    this.generalLog('Translated! .nexe size is: ' + nexeSize);

    // NOTE: the size of the shm 'file' for the nexeFD is not needed because
    // size information is already in the ELF headers.
    this.nexePlugin_.__launchExecutableFromFd(nexeFD);
    this.generalLog('Nexe is now running!');
    this.finishCallback_();
  }

  ///
  /// Main coordination stuff
  /// END

  /// BEGIN
  /// Timing Utilities
  ///

  // To be useful, we need this emitted to the buildbot stdout in the form
  // "RESULT...". It may be best to do this in the plugin.
  // Perhaps this should be done by an external library like nacltest.js
  this.timeStartData = {};

  this.timeStart = function(key) {
    this.timeStartData[key] = (new Date).getTime();
  }

  this.timeEnd = function(key) {
    var end_time = (new Date).getTime();
    this.generalLog(key + ' took: ' +
                    (end_time - this.timeStartData[key]) + 'ms');
  }

  ///
  /// Timing Utilities
  /// END

  this.myLogger = new LogToPage('#C0FFEE');
  this.generalLog = function(msg) {
    this.myLogger.logToPage(msg);
  }

  this.failHandlerGenerator = function(ComponentName) {
    return function(object) {
      this.generalLog(ComponentName + ' FAILED: ' + object);
      throw object;
    }
  }
} // end PnaclTranslatorState


// @public
function pnaclTranslatorInit(bitcodeURL,
                             pnaclEmbeds,
                             nexeModule,
                             finishCallback) {
  var translatorState = new PnaclTranslatorState(bitcodeURL,
                                                 pnaclEmbeds,
                                                 nexeModule,
                                                 finishCallback);
  translatorState.generalLog('Entering pnaclTranslatorInit');
  // When we have caching we could skip downloading all the dependencies
  // and translation. Then again, caching is likely not be based on this script.
  translatorState.startLoadingDependencies(translatorState);
}


function LogToPage(borderColor) {
  this.myLogArea_ = null;
  this.borderColor_ = borderColor;

  this.makeMyLogArea_ = function() {
    this.myLogArea_ = document.createElement('div');
    this.myLogArea_.style.border = '2px solid ' + borderColor;
    this.myLogArea_.style.padding = '10px';
    var headerNode = document.createTextNode('PNACL LOGS: ');
    var br = document.createElement('br');
    this.myLogArea_.appendChild(headerNode);
    this.myLogArea_.appendChild(br);
    document.body.appendChild(this.myLogArea_);
  }

  this.logToPage = function(msg) {
    if (!this.myLogArea_) {
      this.makeMyLogArea_();
    }
    var div = document.createElement('div');
    // Preserve whitespace formatting.
    div.style['white-space'] = 'pre';
    var mNode = document.createTextNode(msg);
    div.appendChild(mNode);
    this.myLogArea_.appendChild(div);
  }
}

//////////////////////////////////////////////////////////////////////
// Other Utilities

// Helper to load up files needed by PNaCl.
// This has the side-effect of also initiating translation once all is done.
function PnaclFileLoader(translatorState, fileToLoad) {
  this.onload = function(nacl_fd) {
    translatorState.generalLog('Loaded ' + fileToLoad);
    translatorState.noteLoaded(fileToLoad, nacl_fd);

    // Are we done loading? If so, translate and run!
    if (translatorState.didLoadDependencies()) {
      translatorState.timeEnd('Downloads');
      translatorState.translateAndRun();
    }
  }

  this.onfail = translatorState.failHandlerGenerator('PnaclFileLoader');
}
